import os
import json
from pathlib import Path

from flask import jsonify, request, safe_join, send_from_directory, current_app
from flask import Blueprint
from flask_cors import CORS
from flask_restful import abort

from LMSAPI.api.Models.File import File
from LMSAPI.api.Models.Schoolyear import Schoolyear
from LMSAPI.api.Views.TokenAPI import auth
from LMSAPI.api.Models.Dashboard import Dashboard

dashboard_api = Blueprint("dashboard_api", __name__)
CORS(dashboard_api)


@dashboard_api.route("/lms/api/v1.0/<lname>/dashboard/structure", methods=["GET"])
def get_structure(lname):
    structure_json_path = "./static/dashboard/structure.json"
    if os.path.exists(structure_json_path):
        with open(structure_json_path, "r") as json_file:
            data = json_file.read()
            return jsonify(json.loads(data))
    else:
        return jsonify({"error": "Файл не найден"})


@dashboard_api.route("/lms/api/v1.0/<lname>/dashboard/university", methods=["GET"])
def get_university(lname):
    university_json_path = "./static/dashboard/university.json"
    if os.path.exists(university_json_path):
        with open(university_json_path, "r", encoding="utf-8") as json_file:
            data = json_file.read()
            return jsonify(json.loads(data))
    else:
        return jsonify({"error": "Файл не найден"}), 404


@dashboard_api.route(
    "/lms/api/v1.0/<lname>/dashboard/training_quality", methods=["GET"]
)
def get_training_quality(lname):
    training_quality = {
        "speciality": [
            {"name": "Военные специальности командного профиля", "count": 166},
            {"name": "Военные специальности инженерного профиля", "count": 209},
            {"name": "Другие военные специальности", "count": 22},
        ],
        "honors_degree": [
            {"name": "Завершили обучение с медалью", "perc": 3.3},
            {"name": "Завершили обучение с дипломо с отличием", "perc": 14.2},
        ],
        "fgos": [
            {
                "created_by": [
                    {"name": "ФГОС разработанные вузами МО РФ", "count": 63},
                    {"name": "ФГОС разработанные Минобрнауки России", "count": 46},
                ],
                "areas_of_training": [
                    {"name": "ФГОС подготовки слушателей", "count": 14},
                    {"name": "ФГОС подготовки курсантов с ВО", "count": 60},
                    {"name": "ФГОС подготовки курсантов с СПО", "count": 35},
                ],
            }
        ],
        "avg_mark": [
            {"name": "Государственный междисциплинарный экзамен", "score": 4.35},
            {"name": "Защита ВКР", "score": 4.42},
            {"name": "Государственный экзамен по физической подготовке", "score": 4.41},
        ],
    }
    return jsonify(training_quality)


@dashboard_api.route(
    "/lms/api/v1.0/<lname>/dashboard/information_support", methods=["GET"]
)
def get_information_support(lname):
    information_support = {
        "electronic_textbook": {"base": 14381, "others": 113, "count": 14494},
        "pc": {
            "year_stat_perc": {
                "2017": 7.2,
                "2018": 14.1,
                "2019": 21.0,
                "2020": 27.9,
                "2021": 41.5,
                "2022": [{"jun": 65.1, "aug": 78.9, "sep": 97.0}],
            }
        },
        "library": {
            "users": 84676,
            "ir_loading": {
                "2021": [{"dec": 100}],
                "2022": [{"apr": 1500, "jun": 3000, "aug": 4740}],
            },
        },
    }
    return jsonify(information_support)


@dashboard_api.route("/lms/api/v1.0/<lname>/dashboard/staffing", methods=["GET"])
def get_staffing(lname):
    staffing = {
        "teachers_staffing_perc": {"total": 89, "military": 89, "civil": 90},
        "professor_combat_exp_perc": 32.2,
        "additional_edu": {
            "prof_retraining": 89,
            "advanced_training": 32.7,
            "traineeship": 10.7,
        },
        "scientists": {
            "perc": 71,
            "ratio": [
                {"name": "Доктора наук", "perc": 13},
                {"name": "Кандидаты наук", "perc": 58},
                {"name": "Высшее образование", "perc": 29},
            ],
        },
        "dissertation_defense_perc": 77,
    }
    return jsonify(staffing)


@dashboard_api.route("/lms/api/v1.0/<lname>/dashboard/vka_performance", methods=["GET"])
def get_vka_perfomance(lname):
    perf = {
        "pps_stat": {
            "military_civil_perc": {"military": 49, "civil": 51},
            "scientific_perc": [
                {"name": "Доктора наук", "perc": 12.2},
                {"name": "Кандидаты наук", "perc": 59.9},
                {"name": "Профессора", "perc": 7.2},
                {"name": "Доценты", "perc": 33.2},
                {"name": "Профессора и доценты без ученой степени", "perc": 3},
                {"name": "Не имеют научной степени, ученого звания", "perc": 24.9},
            ],
        },
        "pps_age": [
            {"name": "До 30 лет", "perc": 9.7},
            {"name": "От 36 до 45 лет", "perc": 29.9},
            {"name": "От 56 до 55 лет", "perc": 33.2},
            {"name": "От 56 до 65 лет", "perc": 15.1},
            {"name": "Старше 65", "perc": 12.1},
        ],
        "final_marks": {
            "avg_interdisciplinary_mark": 4.80,
            "avg_graduation_mark": 4.92,
            "avg_state_exam_mark": 4.41,
        },
        "electronic_env": {
            "el_textbook_perc": 90,
            "video_lectures_perc": 76,
            "el_test_perc": 15,
            "local_network_perc": 76,
        },
        "staff_adjunct_doctor": {
            "doctor_perc": 94,
            "full_time_adjunct_perc": 99.1,
            "extramural_adjunct_perc": 58,
        },
        "progress_dynamics": [
            {"month": "sep", "avg_score": 4.8},
            {"month": "oct", "avg_score": 4.6},
            {"month": "nov", "avg_score": 4.5},
            {"month": "dec", "avg_score": 4.1},
            {"month": "jan", "avg_score": 4.5},
            {"month": "feb", "avg_score": 4.8},
            {"month": "mar", "avg_score": 4.7},
            {"month": "apr", "avg_score": 4.6},
            {"month": "may", "avg_score": 4.2},
            {"month": "jun", "avg_score": 4.7},
            {"month": "jul", "avg_score": 4.5},
            {"month": "aug", "avg_score": 4.4},
        ],
        "absence_reason": [
            {"name": "Отпуск", "perc": 91},
            {"name": "Прочее", "perc": 3},
            {"name": "Болезнь", "perc": 3},
            {"name": "Наряд", "perc": 2},
            {"name": "Командировка", "perc": 1},
        ],
    }
    return jsonify(perf)


@dashboard_api.route("/lms/api/v1.0/<lname>/dashboard/main_pps_stat", methods=["GET"])
def get_main_pps_stat(lname):
    pps_stat = Dashboard.get_pps_main_stat(lname)
    return jsonify(pps_stat)


@dashboard_api.route("/lms/api/v1.0/<lname>/dashboard/age_stat", methods=["GET"])
def age_stat(lname):
    stat = Dashboard.get_age_stat(lname)
    return jsonify(stat)


@dashboard_api.route("/lms/api/v1.0/<lname>/dashboard/university_stat", methods=["GET"])
def get_university_stat(lname):
    university_stat = {}
    main_stat_path = "./static/dashboard/main_stat.json"
    main_data = {}
    if os.path.exists(main_stat_path):
        with open(main_stat_path, "r") as json_file:
            data = json_file.read()
            main_data = json.loads(data)

    if not main_data:
        return jsonify({"error": "Файл не найден"})

    university_stat["main_pps_stat"] = {
        "candidate": Dashboard.get_count_xp_personal_file_fil_okin(
            lname, "Кандидат наук"
        ),
        "citizen": Dashboard.get_count_xp_personal_file(lname, "Гражданский персонал"),
        "docent": Dashboard.get_count_xp_personal_file_fil_academic_title(
            lname, "Доцент"
        ),
        "doctor": Dashboard.get_count_xp_personal_file_fil_okin(lname, "Доктор наук"),
        "military": Dashboard.get_count_xp_personal_file(lname, "Военнослужащие"),
        "professor": Dashboard.get_count_xp_personal_file_fil_academic_title(
            lname, "Профессор"
        ),
        "staffing_scientists": Dashboard.get_count_personnel_pps(lname),
    }
    university_stat["doc_ad_staffing"] = {
        "doctor": Dashboard.get_doctors_avg_info(lname),
        "fulltime_adjuncture": Dashboard.get_adjuncture_avg_info(lname)
    }
    university_stat["avg_gia_mark"] = Dashboard.get_avg_gia_mark(lname)
    university_stat["mark_stat"] = Dashboard.get_university_mark_stat(lname)
    return jsonify(university_stat)


@dashboard_api.route(
    "/lms/api/v1.0/<lname>/dashboard/faculty_stat/<int:faculty_id>", methods=["GET"]
)
def get_faculty_stat(lname, faculty_id):
    faculty_stat = {"main_pps_stat": {
            "candidate": Dashboard.get_count_xp_personal_file_fil_okin_by_faculty(
                lname, "Кандидат наук", faculty_id
            ),
            "citizen": Dashboard.get_count_xp_personal_file_by_faculty(
                lname, "Гражданский персонал", faculty_id
            ),
            "docent": Dashboard.get_count_xp_personal_file_fil_academic_title_by_faculty(
                lname, "Доцент", faculty_id
            ),
            "doctor": Dashboard.get_count_xp_personal_file_fil_okin_by_faculty(
                lname, "Доктор наук", faculty_id
            ),
            "military": Dashboard.get_count_xp_personal_file_by_faculty(
                lname, "Военнослужащие", faculty_id
            ),
            "professor": Dashboard.get_count_xp_personal_file_fil_academic_title_by_faculty(
                lname, "Профессор", faculty_id
            ),
            "staffing_scientists": Dashboard.get_count_personnel_pps_by_faculty(
                lname, faculty_id
            )
        },
        "pps_age_stat": Dashboard.get_pps_age_stat_by_faculty(lname, faculty_id),
        "avg_gia_mark": Dashboard.get_avg_gia_mark_by_faculty(lname, faculty_id),
        "mark_stat": Dashboard.get_university_mark_stat_by_faculty(lname, faculty_id)
    }
    return jsonify(faculty_stat)


@dashboard_api.route(
    "/lms/api/v1.0/<lname>/dashboard/stats", methods=["GET"]
)
def get_ege_stats(lname):
    cached_result = Dashboard.get_stats_from_redis(lname)
    if cached_result is not None:
        return cached_result

    year = request.args.get("xp_key", type=int, default=Schoolyear(lname).getCurrentSchoolYear()["xp_key"])

    ege_stats, conn = Dashboard.get_ege_stats(lname, year)
    teachers = Dashboard.get_teachers_stats(conn, year)
    eioc = Dashboard.get_eioc_stats(conn, year, None)
    vsoko = Dashboard.get_vsoko_stats(conn, year, None)
    workers = Dashboard.get_workers_stats(conn, year)
    students = Dashboard.get_students_stats(lname, conn, year)

    # преобразуем списки в словари {mpid: result}
    teachers_map = {x["mpid"]: x["result"] for x in teachers}
    eioc_map = {x["mpid"]: x["result"] for x in eioc}
    vsoko_map = {x["mpid"]: x["result"] for x in vsoko}
    workers_map = {x["mpid"]: x["result"] for x in workers}
    students_map = {x["mpid"]: x["result"] for x in students}

    # объединяем
    combined = []
    for item in ege_stats:
        item.pop("avg_dvi")
        item.pop("avg_ege")
        mpid = item["mpid"]
        item["teachers_result"] = teachers_map.get(mpid)
        item["eioc_result"] = eioc_map.get(mpid)
        item["vsoko_result"] = vsoko_map.get(mpid)
        item["workers_result"] = workers_map.get(mpid)
        item["students_result"] = students_map.get(mpid)
        combined.append(item)

    Dashboard.set_stats_to_redis(lname, combined)

    return jsonify(combined)



@dashboard_api.route(
    "/lms/api/v1.0/<lname>/dashboard/ege_stats/people/<int:mpid>", methods=["GET"]
)
def get_ege_stats_people_by_mpid(lname, mpid: int):
    year = request.args.get("xp_key", type=int, default=Schoolyear(lname).getCurrentSchoolYear()["xp_key"])
    ege_stats = Dashboard.get_ege_stats_people_by_mpid(lname, mpid, year)
    return jsonify(ege_stats)


@dashboard_api.route(
    "/lms/api/v1.0/<lname>/dashboard/ege_stats/subjects/<int:mpid>", methods=["GET"]
)
def get_ege_stats_subjects_by_mpid(lname, mpid: int):
    year = request.args.get("xp_key", type=int, default=Schoolyear(lname).getCurrentSchoolYear()["xp_key"])
    result = {}
    stats, conn = Dashboard.get_ege_stats(lname, year)
    result["mpid_stats"] = next((item for item in stats if item["mpid"] == mpid), {})
    result["subjects_stats"] = Dashboard.get_ege_stats_academic_subjects_by_mpid(lname, mpid, year)
    return jsonify(result)


@dashboard_api.route(
    "/lms/api/v1.0/<lname>/dashboard/teachers/degree/<int:mpid>", methods=["GET"]
)
def get_teachers_degree_stats(lname, mpid: int):
    year = request.args.get("xp_key", type=int, default=Schoolyear(lname).getCurrentSchoolYear()["xp_key"])
    result = {}
    stats = Dashboard.get_teachers_stats(lname, year)
    result["mpid_stats"] = next((item for item in stats if item["mpid"] == mpid), {})
    result["degree_stats"] = Dashboard.get_stats_staff_by_academic_degree_or_academic_title(lname, mpid, year)
    return jsonify(result)


@dashboard_api.route(
    "/lms/api/v1.0/<lname>/dashboard/teachers/staff/<int:mpid>", methods=["GET"]
)
def get_teachers_staff_stats(lname, mpid: int):
    year = request.args.get("xp_key", type=int, default=Schoolyear(lname).getCurrentSchoolYear()["xp_key"])
    stats = Dashboard.get_stats_staff_and_stag(lname, mpid, year)
    return jsonify(stats)


@dashboard_api.route(
    "/lms/api/v1.0/<lname>/dashboard/workers/stage/<int:mpid>", methods=["GET"]
)
def get_workers_staff_stats(lname, mpid: int):
    year = request.args.get("xp_key", type=int, default=Schoolyear(lname).getCurrentSchoolYear()["xp_key"])
    result = {}
    stats = Dashboard.get_workers_stats(lname, year)
    result["mpid_stats"] = next((item for item in stats if item["mpid"] == mpid), {})
    result["stage_stats"] = Dashboard.get_workers_stage_count_stats(lname, mpid, year)
    return jsonify(result)


@dashboard_api.route(
    "/lms/api/v1.0/<lname>/dashboard/eioc/<int:mpid>", methods=["GET"]
)
def get_eioc_stats(lname, mpid: int):
    year = request.args.get("xp_key", type=int, default=Schoolyear(lname).getCurrentSchoolYear()["xp_key"])
    eioc = Dashboard.get_eioc_stats(lname, year, mpid)
    mpid_stats = next((item for item in eioc if item["mpid"] == mpid), {})
    return jsonify(mpid_stats)


@dashboard_api.route(
    "/lms/api/v1.0/<lname>/dashboard/vsoko/<int:mpid>", methods=["GET"]
)
def get_vsoko_stats(lname, mpid: int):
    year = request.args.get("xp_key", type=int, default=Schoolyear(lname).getCurrentSchoolYear()["xp_key"])
    vsoko = Dashboard.get_vsoko_stats(lname, year, mpid)
    mpid_stats = next((item for item in vsoko if item["mpid"] == mpid), {})
    return jsonify(mpid_stats)


@dashboard_api.route("/lms/api/v1.0/<lname>/dashboard/<int:f_doc_hs>/<path:filename>", methods=["GET"])
@auth.login_required
def download_vsoko_local_act_file(lname, f_doc_hs, filename):
    f = File(lname)

    # ОБЯЗАТЕЛЬНО: приводим к Path, даже если метод вернул str
    dirpath = Path(f.get_vsoko_local_act_directory(f_doc_hs))

    if not dirpath.exists() or not dirpath.is_dir():
        abort(404, description="Directory not found")

    # безопасно собираем путь
    safe_path = safe_join(str(dirpath), filename)
    if safe_path is None:
        abort(400, description="Invalid filename")

    full_path = Path(safe_path)

    # не позволяем выйти за пределы директории
    try:
        full_path.resolve().relative_to(dirpath.resolve())
    except Exception:
        abort(400, description="Invalid path")

    if not full_path.exists() or not full_path.is_file():
        abort(404, description="File not found")

    as_attachment = request.args.get("download") in ("1", "true", "yes")

    return send_from_directory(
        directory=str(dirpath),
        filename=full_path.name,
        as_attachment=as_attachment,
        conditional=True
    )


@dashboard_api.route(
    "/lms/api/v1.0/<lname>/dashboard/students/staff/<int:mpid>", methods=["GET"]
)
def get_students_avg_percent(lname, mpid: int):
    year = request.args.get("xp_key", type=int, default=Schoolyear(lname).getCurrentSchoolYear()["xp_key"])
    stats = Dashboard.get_students_avg_percent(lname, mpid, year)
    return jsonify(stats)


@dashboard_api.route(
    "/lms/api/v1.0/<lname>/dashboard/students/stats/<int:mpid>", methods=["GET"]
)
def get_students_stats(lname, mpid: int):
    year = request.args.get("xp_key", type=int, default=Schoolyear(lname).getCurrentSchoolYear()["xp_key"])
    stats = Dashboard.get_students_completion_histogram(lname, mpid, year)
    return jsonify(stats)



@dashboard_api.route(
    "/lms/api/v1.0/<lname>/dashboard/college", methods=["GET"]
)
def get_dashboard_college(lname):
    year = request.args.get("xp_key", type=int, default=Schoolyear(lname).getCurrentSchoolYear()["xp_key"])
    # заглушка для тестирования
    # year = 38
    conn = current_app.ms.db(lname).connect()
    disciplines = Dashboard.get_college_disciplines(conn, year)

    for disc in disciplines:
        eioc = Dashboard.get_eioc_stats(conn, year, disc["mpid"])
        vsoko = Dashboard.get_vsoko_stats(conn, year, disc["mpid"])
        students =  Dashboard.get_college_students_stats(lname, conn, year, disc["mpid"])
        workers = Dashboard.get_college_workers_stats(conn, year, disc["mpid"])

        eioc_map = {x["mpid"]: x["result"] for x in eioc}
        vsoko_map = {x["mpid"]: x["result"] for x in vsoko}
        workers_map = {x["mpid"]: x["result"] for x in workers}
        students_map = {x["mpid"]: x["result"] for x in students}

        mpid = disc["mpid"]
        disc["eioc_result"] = eioc_map.get(mpid)
        disc["vsoko_result"] = vsoko_map.get(mpid)
        disc["workers_result"] = workers_map.get(mpid)
        disc["students_result"] = students_map.get(mpid)

    return jsonify(disciplines)