from functools import wraps

from flask import jsonify, request, g
from flask import Blueprint
from LMSAPI.api.Views.TokenAPI import auth
from LMSAPI.api.Models.Faculty import Faculty
from flask_cors import CORS
import json

from LMSAPI.api.utils.access_utils import user_permission_modes
from LMSAPI.api.utils.swagger_utils import swagger_doc
from LMSAPI.api.utils.swagger_comments import SwaggerComments

faculty_api = Blueprint("faculty_api", __name__)

CORS(faculty_api)


@faculty_api.route("/lms/api/v1.0/<lname>/faculty", methods=["GET"])
@swagger_doc(SwaggerComments.faculty_api_getFacultyList)
def getFacultyList(lname):
    """
        Получить список факультетов (только действующих)
    """
    facultyList = Faculty(lname).getFacultyList(lname)
    return (
        jsonify([fa.serialize() for fa in facultyList]),
        200,
        {"Content-Type": "text/json; charset=utf-8"},
    )


@faculty_api.route("/lms/api/v1.0/<lname>/faculty/by_user", methods=["GET"])
@auth.login_required
def get_faculty_user(lname):
    """returns a json of the user's faculties"""
    mid = g.user.mid
    faculty_user = Faculty(lname).get_faculty_list_user(lname, mid)
    return (
        jsonify([fu.serialize() for fu in faculty_user]),
        200,
        {"Content-Type": "text/json; charset=utf-8"},
    )


@faculty_api.route("/lms/api/v1.0/<lname>/faculty/teacher", methods=["GET"])
@auth.login_required
def getFacultyListForTeacher(lname):
    if not g.user.isTeacher:
        return jsonify({"error": "Только для учителя"})
    facultyList = Faculty(lname).getFacultyListForTeacher(lname, g.user.mid)

    return (
        jsonify([fa.serialize() for fa in facultyList]),
        200,
        {"Content-Type": "text/json; charset=utf-8"},
    )


@faculty_api.route("/lms/api/v1.0/<lname>/journal_of_accounting_of_training_sessions/faculty/teacher", methods=["GET"])
@auth.login_required
def get_faculty_list_for_teacher_journal_of_accounting_of_training_sessions(lname):
    """
    Режим: "Журнал учета учебных занятий"
    """
    if not g.user.isTeacher:
        return jsonify({"error": "Только для учителя"})
    faculty_list = Faculty(lname).get_faculty_list_for_teacher_journal_of_accounting_of_training_sessions(lname)

    return (
        jsonify(faculty_list),
        200,
        {"Content-Type": "text/json; charset=utf-8"},
    )


@faculty_api.route("/lms/api/v1.0/<lname>/faculty/<int:id>", methods=["GET"])
@auth.login_required
@user_permission_modes("Справочники", "Факультеты", ["Нет"])
def getFaculty(lname, id):
    result = Faculty(lname).getFaculty(lname, id)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@faculty_api.route("/lms/api/v1.0/<lname>/faculty", methods=["POST"])
@auth.login_required
@user_permission_modes("Справочники", "Факультеты", ["Нет"])
def addFaculty(lname):
    data = request.data

    dataDict = {}
    if request.json:
        dataDict = request.json
    else:
        dataDict = json.loads(data)
    facultyid = Faculty(lname).addFaculty(dataDict)

    if facultyid is None:
        return jsonify(success=False), 404

    return jsonify({"facultyid": facultyid})


@faculty_api.route("/lms/api/v1.0/<lname>/faculty/srw", methods=["GET"])
# SRW == Scientific Research Work
def getFacultyForSRW(lname):
    result = Faculty(lname).getFacultyForSRW(lname)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@faculty_api.route("/lms/api/v1.0/<lname>/journal/faculty", methods=["GET"])
@auth.login_required
def get_journal_faculty(lname):
    result = Faculty(lname).get_journal_faculty()
    if result is None:
        return jsonify(success=False), 404
    return jsonify(result)
