from datetime import date

from flask import Flask, jsonify, request, current_app, g
from flask_restful import Resource, Api
from sqlalchemy import create_engine
from sqlalchemy.sql import text
from flask import Blueprint

from LMSAPI.api.Models.Schoolyear import Schoolyear
from LMSAPI.api.Views.TokenAPI import auth
from LMSAPI.api.Models.User import User
from LMSAPI.api.Models.Cathedra import Cathedra
from LMSAPI.api.Models.Group import Group
from flask_cors import CORS

from LMSAPI.api.utils.access_utils import user_permission_modes
from LMSAPI.api.utils.swagger_utils import swagger_doc
from LMSAPI.api.utils.swagger_comments import SwaggerComments

# https://stackoverflow.com/questions/39769666/flask-how-to-use-app-context-inside-blueprints
# Use app context from blueprint

group_api = Blueprint("group_api", __name__)

CORS(group_api)


@group_api.route(
    "/lms/api/v1.0/<lname>/group/<int:cid>/<int:fid>/<int:yid>/<int:educationyid>",
    methods=["GET"],
)
def getGroups(lname, cid, fid, yid, educationyid):
    """GET with ID of Cathedra to get list of groups"""
    result = Group(lname).getGroupList(lname, cid, fid, yid, educationyid)

    if result is None:
        return jsonify(success=False), 404

    return jsonify([fa.serialize() for fa in result])


@group_api.route(
    "/lms/api/v1.0/<lname>/group/teacher/<int:cid>/<int:trimid>", methods=["GET"]
)
@auth.login_required
def getGroupsForTeacher(lname, cid, trimid):
    result = Group(lname).getGroupListForTeacher(lname, cid, trimid)

    if result is None:
        return jsonify(success=False), 404

    return jsonify([fa.serialize() for fa in result])


@group_api.route(
    "/lms/api/v1.0/<string:lname>/group/<int:school_year_id>/<int:kurs>",
    methods=["GET"],
)
@auth.login_required
# @user_permission_modes("Режимы", "Журнал учета учебных занятий", ["Нет"])
def get_groups_variable_stuff(lname: str, school_year_id: int, kurs: int):
    filters = request.args
    faculty_id = filters.get("faculty_id", default=0, type=int)

    if faculty_id and not faculty_id.isdigit():
        return jsonify(msg="faculty_id must be digit"), 400

    cathedra_id = filters.get("cathedra_id", default=0, type=int)

    if cathedra_id and not cathedra_id.isdigit():
        return jsonify(msg="cathedra_id must be digit"), 400

    result = Group(lname).get_group_list_variable_stuff(
        lname,
        cathedra_id=cathedra_id,
        kurs=kurs,
        school_year_id=school_year_id,
        faculty_id=faculty_id,
    )

    return jsonify(result)


@group_api.route(
    "/lms/api/v1.0/<string:lname>/group/journal/<int:school_year_id>/<int:trim_id>",
    methods=["GET"],
)
@auth.login_required
# @user_permission_modes("Режимы", "Журнал учета учебных занятий", ["Нет"])
def get_groups_journal(lname: str, school_year_id: int, trim_id: int):
    filters = request.args
    faculty_id = filters.get("faculty_id", default=0, type=int)

    if faculty_id and not faculty_id.isdigit():
        return jsonify(msg="faculty_id must be digit"), 400

    cathedra_id = filters.get("cathedra_id", default=0, type=int)

    if cathedra_id and not cathedra_id.isdigit():
        return jsonify(msg="cathedra_id must be digit"), 400

    result = Group(lname).get_group_list_journal(
        lname,
        cathedra_id=cathedra_id,
        trim_id=trim_id,
        school_year_id=school_year_id,
        faculty_id=faculty_id,
    )

    return jsonify(result)


@group_api.route(
    "/lms/api/v1.0/<string:lname>/group/journal_of_accounting_of_training_sessions/<int:xp_key>/<int:trim_id>",
    methods=["GET"],
)
@auth.login_required
def get_groups_journal_of_accounting_of_training_sessions(lname: str, xp_key: int, trim_id: int):
    """
        Получение групп для режима "Журнал учета учебных занятий"
    """

    is_admin = int(g.user.access_modes["Режимы"]["Журнал учета учебных занятий. Администратор"] in ["Полный"])

    filters = request.args
    faculty_id = filters.get("faculty_id", default=0, type=int)
    cathedra_id = filters.get("cathedra_id", default=0, type=int)

    result = Group(lname).get_groups_journal_of_accounting_of_training_sessions(
        mid=g.user.mid,
        is_admin=is_admin,
        xp_key=xp_key,
        trmid=trim_id,
        idcathedra=cathedra_id,
        idfaculty=faculty_id
    )

    return jsonify(result)


@group_api.route(
    "/lms/api/v1.0/<string:lname>/military_professional_portrait_of_a_student/group",
    methods=["GET"],
)
@auth.login_required
def get_groups_military_professional_portrait_of_a_student(lname: str):
    """
        Получение групп для режима "Военно-профессиональный портрет обучающегося"
    """
    # На случай добавления прав
    # is_admin = int(g.user.access_modes["Режимы"]["Военно-профессиональный портрет обучающегося. Администратор"] in ["Полный"])

    filters = request.args
    year_id = filters.get("year_id", default=0, type=int)
    faculty_id = filters.get("faculty_id", default=0, type=int)
    military_profession_id = filters.get("military_profession_id", default=0, type=int)

    result = Group(lname).get_groups_military_professional_portrait_of_a_student(
        year_id=year_id,
        faculty_id=faculty_id,
        military_profession_id=military_profession_id
    )

    return jsonify(result)


@group_api.route(
    "/lms/api/v1.0/<lname>/group/<int:faculty_id>/<int:course>/<chosen_date>",
    methods=["GET"],
)
@auth.login_required
def getGroupsByCourse(lname, faculty_id, course, chosen_date):
    result = Group.getGroupListByCourse(lname, faculty_id, course, chosen_date)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@group_api.route(
    "/lms/api/v1.0/<lname>/class_schedule/group",
    methods=["GET"],
)
@auth.login_required
@swagger_doc(SwaggerComments.group_api_get_groups_to_schedule_classes)
def get_groups_to_schedule_classes(lname):
    """
    Получить список групп для составления расписания занятий
    """
    xp_key = request.args.get("xp_key", default=0, type=int)
    eyid = request.args.get("eyid", default=0, type=int)
    idfaculty = request.args.get("idfaculty", default=0, type=int)

    result = Group.get_groups_to_schedule_classes(lname, xp_key, eyid, idfaculty)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)



@group_api.route("/lms/api/v1.0/<lname>/group", methods=["GET"])
@auth.login_required
def getAllGroups(lname):
    result = Group(lname).getAllGroups(lname)

    if result is None:
        return jsonify(success=False), 404

    return jsonify([fa.serialize() for fa in result])


@group_api.route("/lms/api/v1.0/<lname>/group/<int:id>", methods=["GET"])
@auth.login_required
def getGroup(lname, id):
    """GET with ID to get details for a Group"""

    result = Group(lname).getGroup(lname, id)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@group_api.route("/lms/api/v1.0/<lname>/group/<int:id>/students", methods=["GET"])
@auth.login_required
def getGroupStudents(lname, id):
    """GET with ID to get details for a Group"""

    result = Group(lname).getGroupStudents(lname, id)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@group_api.route("/lms/api/v1.0/<lname>/variable_composition/group/<int:xp_key>", methods=["GET"])
@auth.login_required
@swagger_doc(SwaggerComments.group_api_get_groups_by_faculty_cathedra_year_eyid)
def get_groups_by_faculty_cathedra_year_eyid(lname, xp_key):
    """
    Получить список учебных групп (режим "Переменный состав")
    """
    idfaculty = request.args.get("idfaculty", default=0, type=int)
    idcathedra = request.args.get("idcathedra", default=0, type=int)
    eyid = request.args.get("eyid", default=-1, type=int)
    admin_flag = int(g.user.access_modes["Справочники"]["Переменный состав. Администратор"] in ["Полный"])
    result = Group(lname).get_groups_by_faculty_cathedra_year_eyid(lname, xp_key, idfaculty, idcathedra, eyid, g.user.mid, admin_flag)

    if result is None:
        return (
            jsonify(
                success=False,
            ),
            404,
        )
    return jsonify(result)


@group_api.route("/lms/api/v1.0/<lname>/role/group", methods=["GET"])
@auth.login_required
@user_permission_modes(
    "Режимы",
    "Ввод индивидуальных достижений." or "Рейтинг распределение и цифровой портрет.",
    ["Нет"],
)
def get_groups_by_role(lname):
    xp_key = request.args.get("xp_key", default=0, type=int)
    idfaculty = request.args.get("idfaculty", default=0, type=int)
    idcathedra = request.args.get("idcathedra", default=0, type=int)
    result = Group(lname).get_groups_by_role(lname, xp_key, idfaculty, idcathedra)

    if result is None:
        return (
            jsonify(
                success=False,
            ),
            404,
        )
    if not result:
        return (
            jsonify(
                success=False,
                message="""Не найдены группы для пользователя с правами "Ввод индивидуальных достижений." или "Рейтинг распределение и цифровой портрет." и фильтрами по факультету, кафедре и году""",
            ),
            404,
        )

    return jsonify(result)


@group_api.route("/lms/api/v1.0/<lname>/admin/group", methods=["GET"])
@auth.login_required
@user_permission_modes(
    "Режимы",
    "Ввод индивидуальных достижений. Администратор."
    or "Рейтинг распределение и цифровой портрет. Администратор.",
    ["Нет"],
)
def get_groups_by_admin(lname):
    xp_key = request.args.get("xp_key", default=0, type=int)
    idfaculty = request.args.get("idfaculty", default=0, type=int)
    idcathedra = request.args.get("idcathedra", default=0, type=int)
    result = Group(lname).get_groups_by_admin(lname, xp_key, idfaculty, idcathedra)

    if result is None:
        return (
            jsonify(
                success=False,
            ),
            404,
        )
    if not result:
        return (
            jsonify(
                success=False,
                message="""Не найдены группы с фильтрами по факультету, кафедре и году""",
            ),
            404,
        )

    return jsonify(result)


@group_api.route("/lms/api/v1.0/<lname>/group/<int:gid>/students_by_period/<begdate>/<enddate>", methods=["GET"])
@auth.login_required
def get_group_students_by_period(lname, gid, begdate, enddate):
    """
        Получение студентов из группы которые были в её составе на данный период
    """
    result = Group(lname).get_group_students_by_period(begdate, enddate, gid)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)
