import json
import os
import threading
import uuid
import zipfile
import tarfile
from flask import Blueprint, jsonify, request, g, current_app
from flask_cors import CORS
from werkzeug.utils import secure_filename

from LMSAPI.api.Models.File import File
from LMSAPI.api.Models.ImportApplicant import ImportApplicant
from LMSAPI.api.Views.TokenAPI import auth
from LMSAPI.api.utils.gpghlms import statusPGPKey, decryptPGPFile
from LMSAPI.api.utils.access_utils import user_permission_modes


import_applicant_api = Blueprint("import_applicant_api", __name__)

CORS(import_applicant_api)


def copy_files_if_not_exists(lname, xp_key, json_file, mid):
    """
    Копирует файлы из исходной директории в целевую, избегая дублирования.
    """
    # Создаем директорию для xp_key
    File(lname).create_xp_applicant_directory(xp_key)
    file_path = File(lname).get_xp_applicant_directory(xp_key)

    # Убираем '/data.json' из пути json_file
    mid_folder = json_file.rstrip("/data.json")  # Убираем расширение .json из пути
    mid_folder = os.path.join(mid_folder, str(mid))  # Добавляем папку с mid

    # Проверяем и копируем файлы, если папка для mid существует
    if os.path.isdir(mid_folder):
        for root, _, files in os.walk(mid_folder):
            for file_name in files:
                source_file = os.path.join(root, file_name)
                target_file = os.path.join(file_path, file_name)

                # Проверка на существование файла в целевой директории
                if not os.path.exists(target_file):
                    import shutil

                    shutil.copy2(source_file, target_file)  # Копирование файла


def run_import_applicant_task(app, lname, task_id, temp_dir, zip_path):
    with app.app_context():
        try:
            # Распаковываем ZIP
            extract_dir = os.path.join(temp_dir, "extracted")
            os.makedirs(extract_dir, exist_ok=True)


            if (zip_path.find('.tar')>0):
                with tarfile.TarFile(zip_path, "r") as tar_ref:
                    tar_ref.extractall(extract_dir)

            else:
                with zipfile.ZipFile(zip_path, "r") as zip_ref:
                    for zip_info in zip_ref.infolist():
                        # Перекодируем имя файла из указанной кодировки
                        filename = zip_info.filename.encode("cp437").decode(
                            "cp866", errors="replace"
                        )
                        zip_info.filename = filename
                        # Извлекаем файл с корректным именем
                        zip_ref.extract(zip_info, extract_dir)

            # Рекурсивный поиск файлов data.json
            json_files = []
            for root, _, files in os.walk(extract_dir):
                for file_name in files:
                    if file_name == "data.json":
                        json_files.append(os.path.join(root, file_name))

            # Читаем и добавляем данные из каждого data.json
            for json_file in json_files:
                encodings = [
                    "utf-8",
                    "utf-16",
                    "utf-16-le",
                    "latin1",
                    "windows-1251",
                    "utf-8-sig",
                    "cp866",
                ]
                json_data = None  # Чтобы проверять успешное считывание

                # Меняем кодировку у json
                for encoding in encodings:
                    try:
                        with open(json_file, "r", encoding=encoding) as f:
                            content = f.read().strip()

                            if content.startswith(
                                "\ufeff"
                            ):  # Удаление BOM, если присутствует
                                content = content.encode("utf-8").decode("utf-8-sig")

                            json_data = json.loads(content)  # Попытка загрузить JSON
                            break  # Успех - выходим из цикла
                    except (UnicodeDecodeError, ValueError) as e:
                        print(
                            "Failed to read {json_file} with encoding {encoding}: {e}".format(
                                json_file=json_file, encoding=encoding, e=e
                            )
                        )

                if json_data is None:
                    ImportApplicant(lname).create_import_applicant_task_info(
                        task_id, "failed_open_json", json_file
                    )
                    continue

                for data in json_data:
                    task_info_id = ImportApplicant(
                        lname
                    ).create_import_applicant_task_info_full(
                        task_id, "in_progress", json_file, data
                    )
                    check_exist = ImportApplicant(lname).check_exist_xp_applicant_v1(
                        data
                    )
                    if not check_exist:
                        xp_key = ImportApplicant(lname).create_xp_applicant(data)
                        if not xp_key:
                            ImportApplicant(
                                lname
                            ).update_status_import_applicant_task_info(
                                task_info_id, "failed_insert_data", xp_key
                            )

                        # Копирование файлов
                        copy_files_if_not_exists(lname, xp_key, json_file, data["mid"])

                        ImportApplicant(lname).update_status_import_applicant_task_info(
                            task_info_id, "completed", xp_key
                        )
                    else:
                        # Копирование файлов
                        copy_files_if_not_exists(
                            lname, check_exist.xp_key, json_file, data["mid"]
                        )

                        ImportApplicant(lname).update_status_import_applicant_task_info(
                            task_info_id, "updated", check_exist.xp_key
                        )

            ImportApplicant(lname).update_status_import_applicant_task(
                task_id, "completed"
            )

        except Exception as e:
            print(e)
            ImportApplicant(lname).update_status_import_applicant_task(
                task_id, "failed"
            )

        finally:
            # Удаляем временные файлы
            if os.path.exists(temp_dir):
                import shutil

                shutil.rmtree(temp_dir)


@import_applicant_api.route("/lms/api/v1.0/<lname>/import_applicant", methods=["POST"])
@auth.login_required
@user_permission_modes("Справочники", "Кандидаты", ["Нет"])
def import_applicant_from_zip(lname):
    # Проверяем, был ли отправлен файл
    if "file" not in request.files:
        return jsonify({"error": "No file part in the request"}), 400

    file = request.files["file"]
    filename = file.filename

    if filename == "":
        return jsonify({"error": "No file selected"}), 400

    # Проверяем расширение файла
    if not filename.lower().endswith(".zip"):
        if not filename.lower().endswith(".gpg"):
            return jsonify({"error": "Загруженный файл не является файлом ZIP или GPG."}), 400

    task_id = uuid.uuid4()  # Генерируем уникальный идентификатор задачи

    temp_dir = os.path.join(
        File(lname).getBaseFileDirecotry(), "uploaded_zip", str(task_id)
    )
    os.makedirs(temp_dir, exist_ok=True)

    zip_path = os.path.join(temp_dir, filename)
    file.save(zip_path)
    gpg_result = None
    if filename.lower().endswith(".gpg"):
        lservers = current_app.config.get("LMSSERVERS")
        lserver = next((lserver for lserver in lservers if lserver.get("NAME") == lname), None)
        jakarta_serial = lserver.get("KEYSERIAL")
        jakarta_keyname = lserver.get("KEYNAME")

        if jakarta_serial is not None and jakarta_keyname is not None:
            source = 'config'
        else:
            jakarta_serial = current_app.jakarta_serial
            jakarta_keyname = current_app.jakarta_keyname
            source = 'key'

        if jakarta_serial is None or jakarta_keyname is None :
            return { "result": "error", "error": "Секретного ключа не существует (USB/Config)", "keyerror": current_app.jakarta_error}


        decrypt_status = decryptPGPFile(zip_path,jakarta_serial, jakarta_keyname)        
        gpg_result = decrypt_status['result']
        if decrypt_status['status'] != "ok":
            return jsonify({"error": "Ошибка расшифровки GPG", "gpg_result":gpg_result }), 400
        zip_path = zip_path[:-4]

    ImportApplicant(lname).create_import_applicant_task(
        task_id, g.user.mid, "in_progress"
    )
    # Запускаем задачу в отдельном потоке
    app = current_app._get_current_object()
    thread = threading.Thread(
        target=run_import_applicant_task,
        args=(app, lname, task_id, temp_dir, zip_path),
    )
    thread.start()
    return jsonify({"task_id": task_id, "gpg_result": gpg_result}), 202


@import_applicant_api.route(
    "/lms/api/v1.0/<lname>/import_applicant_task/<string:task_id>", methods=["GET"]
)
@auth.login_required
@user_permission_modes("Справочники", "Кандидаты", ["Нет"])
def get_task_details(lname, task_id):
    task_id = uuid.UUID(task_id).hex
    import_applicant_task = ImportApplicant(lname).get_import_applicant_task(task_id)

    if not import_applicant_task:
        return jsonify({"error": "Task not found"}), 404

    if import_applicant_task and import_applicant_task.mid != g.user.mid:
        return (
            jsonify({"error": "У данного пользователя нет доступа к этой информации"}),
            404,
        )

    if import_applicant_task.status == "completed":
        limit = request.args.get("limit", default=100, type=int)
        offset = request.args.get("offset", default=0, type=int)
        return (
            jsonify(
                {
                    "status": import_applicant_task.status,
                    "result": ImportApplicant(lname).get_import_applicant_task_info(
                        task_id, limit, offset
                    ),
                }
            ),
            200,
        )
    elif import_applicant_task.status == "failed":
        return jsonify({"status": import_applicant_task.status}), 200

    return jsonify({"status": import_applicant_task.status}), 200
