from flask import Flask, jsonify, request, current_app, send_from_directory, g
from flask_restful import Resource, Api
from sqlalchemy import create_engine
from sqlalchemy.sql import text
from flask import Blueprint
from werkzeug import secure_filename
from LMSAPI.api.Models.UserActionLog import UserActionLog
from LMSAPI.api.Views.TokenAPI import auth
from LMSAPI.api.Models.User import User
from LMSAPI.api.Models.Faculty import Faculty
from LMSAPI.api.Models.File import File
from flask_cors import CORS
import json, os, sys

from LMSAPI.api.utils.access_utils import user_permission_modes

# https://stackoverflow.com/questions/39769666/flask-how-to-use-app-context-inside-blueprints
# Use app context from blueprint

portfolio_api = Blueprint("portfolio_api", __name__)

CORS(portfolio_api)


@portfolio_api.route("/lms/api/v1.0/<lname>/portfolio/<int:id>", methods=["GET"])
@auth.login_required
def getPortfolio(lname, id):
    """GET to generate a list of endpoints and their docstrings"""

    if g.user.mid != id:
        return jsonify(success=False), 403

    result = User(lname).get_user_portfolio(id)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@portfolio_api.route("/lms/api/v1.0/<lname>/portfolio", methods=["POST"])
@auth.login_required
def setPortfolio(lname):
    data = request.data
    if sys.version_info[0] < 3:
        dataDict = json.loads(data)
    else:
        dataDict = json.loads(data.decode("utf-8"))

    result = User(lname).set_user_portfolio(g.user.mid, dataDict)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(dataDict)


@portfolio_api.route("/lms/api/v1.0/<lname>/portfolio/file/<int:mid>", methods=["GET"])
@auth.login_required
def getUserFiles(lname, mid):
    if g.user.mid != mid:
        return jsonify(success=False), 403

    f = File(lname)
    return jsonify(f.getUserPortfolio(mid))


@portfolio_api.route("/lms/api/v1.0/<lname>/portfolio/file/<int:mid>", methods=["POST"])
@auth.login_required
def postUserFile(lname, mid):
    if g.user.mid != mid:
        return jsonify(success=False), 403

    ip_address = (
        request.headers.get("X-Forwarded-For")
        or request.headers.get("X-Real-IP")
        or request.environ.get("REMOTE_ADDR")
    )

    submitted_file = request.files["file"]
    if submitted_file:
        File(lname).createUserPortfolioDirectory(mid)
        file_path = File(lname).getUserPortfolioDirectory(mid)
        return File.save_file_to_path(
            lname=lname,
            submitted_file=submitted_file,
            file_path=file_path,
            action_source=request.environ.get("HTTP_USER_AGENT"),
            login=g.user.id,
            user_mid=g.user.mid,
            ip_address=ip_address,
        )

    return jsonify({"success": False, "error": "upload file with key file"})


@portfolio_api.route(
    "/lms/api/v1.0/<lname>/portfolio/file/<int:userId>/<filename>", methods=["DELETE"]
)
@auth.login_required
def deleteUserFile(lname, userId, filename):
    if g.user.mid != userId:
        return jsonify(success=False), 403

    try:
        os.remove(os.path.join(File(lname).getUserPortfolioDirectory(userId), filename))
        return jsonify(success=True)

    except:
        current_app.logger.error("getUserFile(" + str(userId) + "," + filename + ")")
        return jsonify(success=False), 404


@portfolio_api.route(
    "/lms/api/v1.0/<lname>/portfolio/file/<int:userId>/<filename>", methods=["GET"]
)
@auth.login_required
def getUserFile(lname, userId, filename):
    if g.user.mid != userId:
        return jsonify(success=False), 403

    path = File(lname).getUserPortfolioDirectory(userId)
    ip_address = (
        request.headers.get("X-Forwarded-For")
        or request.headers.get("X-Real-IP")
        or request.environ.get("REMOTE_ADDR")
    )
    return File.send_from_directory(
        lname=lname,
        file_path=path,
        file_name=filename,
        action_source=request.environ.get("HTTP_USER_AGENT"),
        login=g.user.id,
        user_mid=g.user.mid,
        ip_address=ip_address,
    )


@portfolio_api.route("/lms/api/v1.0/<lname>/practice/grades", methods=["GET"])
@auth.login_required
def get_practice_grades(lname):
    result = User(lname).get_practice_grades()

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)
