import os
import urllib

from flask import send_file

from flask import Flask, jsonify, request
from flask import Blueprint
from werkzeug.utils import secure_filename

from LMSAPI.api.Models.File import File
from flask_cors import CORS

from LMSAPI.api.Models.ScholarshipCandidatesFilesStorage import ScholarshipCandidatesFilesStorage
from LMSAPI.api.Views.TokenAPI import auth

scholarship_files_storage = Blueprint("scholarship_files_storage", __name__, url_prefix="/lms/api/v1.0/<lname>")

CORS(scholarship_files_storage)


@scholarship_files_storage.route(
    "/scholarship_candidates/files_storage/<int:mid>",
    methods=["GET"],
)
@auth.login_required
def get_scholarship_candidates_files_list(lname, mid):
    result = ScholarshipCandidatesFilesStorage(lname).get_files_by_mid(mid)
    return jsonify(result), 200


@scholarship_files_storage.route(
    "/scholarship_candidates/files_storage/<int:mid>/<int:chapter>/<string:file_name>",
    methods=["GET"],
)
@auth.login_required
def download_scholarship_candidate_file(lname, mid, chapter, file_name):
    file_path = File(lname).get_scholarship_candidates_file(mid, chapter, file_name)

    if not file_path:
        return jsonify({"error": "Файл не найден"}), 404

    # Кодируем имя файла для заголовка (для браузеров)
    encoded_name = urllib.parse.quote(file_name)

    # Передаём заголовок вручную
    response = send_file(file_path, as_attachment=True)
    response.headers["Content-Disposition"] = "attachment; filename*=UTF-8''{encoded_name}".format(encoded_name=encoded_name)

    return response


@scholarship_files_storage.route(
    "/scholarship_candidates/files_storage",
    methods=["POST"],
)
@auth.login_required
def add_scholarship_candidates_file_storage(lname):
    files = request.files.getlist("files")
    chapter = request.form.get("chapter")
    mid = request.form.get("mid")

    if not files or not chapter or not mid:
        return jsonify({"error": "Необходимо указать файлы, chapter и mid"}), 400

    try:
        chapter = int(chapter)
        mid = int(mid)
    except ValueError:
        return jsonify({"error": "chapter и mid должны быть целыми числами"}), 400

    # Допустимые MIME-типы и расширения
    allowed_mime_types = {
        "application/pdf",
        "application/msword",                       # .doc
        "application/vnd.openxmlformats-officedocument.wordprocessingml.document",  # .docx
        "application/vnd.ms-excel",                # .xls
        "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",        # .xlsx
        "image/jpeg",
        "image/png",
        "image/tiff",
        "image/bmp",
        "image/webp"
    }

    allowed_extensions = {'.pdf', '.doc', '.docx', '.xls', '.xlsx', '.jpeg', '.jpg', '.png', '.tif', '.tiff', '.bmp', '.webp'}

    saved_files_info = []

    for file in files:
        if not file or file.filename == '':
            continue

        # Проверка расширения
        ext = os.path.splitext(file.filename)[-1].lower()
        if ext not in allowed_extensions and file.content_type not in allowed_mime_types:
            return jsonify({"error": "Файл {filename} имеет недопустимый формат.".format(filename=file.filename)}), 400

        # Сохраняем файл и получаем относительный путь
        relative_path = File(lname).save_scholarship_candidates_files(mid, chapter, file)

        # Сохраняем в БД
        result = ScholarshipCandidatesFilesStorage(lname).create_files(
            lname,
            file_path=relative_path,
            file_name=file.filename.strip(),
            mid=mid,
            сhapter=chapter
        )

        saved_files_info.extend(result)

    return jsonify(saved_files_info), 201


@scholarship_files_storage.route(
    "/scholarship_candidates/files_storage/<int:mid>/<int:chapter>/<string:file_name>",
    methods=["DELETE"],
)
@auth.login_required
def delete_scholarship_candidates_file_storage(lname, mid: int, chapter: int, file_name: str):
    if not file_name or not mid or not chapter:
        return jsonify({"error": "Необходимо указать file_name, mid и chapter"}), 400
    # Удаляем файл
    File(lname).delete_scholarship_candidates_files(mid, chapter, file_name)
    # Удаляем из базы
    deleted = ScholarshipCandidatesFilesStorage(lname).delete_file_record(lname, file_name, mid, chapter)
    if deleted:
        return jsonify({"message": "Файл удалён"}), 200
    else:
        return jsonify({"error": "Запись в БД не найдена"}), 404


