from flask import Flask, jsonify, request, current_app, g, send_from_directory
from flask_restful import Resource, Api
from sqlalchemy import create_engine
from sqlalchemy.sql import text
from flask import Blueprint
from LMSAPI.api.Models.UserActionLog import UserActionLog
from LMSAPI.api.Views.TokenAPI import auth
from LMSAPI.api.Models.User import User
from LMSAPI.api.Models.File import File
from LMSAPI.api.Models.StudentProject import StudentProject
from flask_cors import CORS
import json, os, sys

from LMSAPI.api.utils.access_utils import user_permission_modes

student_project_api = Blueprint('student_project_api', __name__)

"""
1. list (filters (termid,  cid))
for student only his projects

2. add, edit , delete (only teacher)
edit and delete only own records

3. reviews (xp_student_project_review) project as parameter
list, add,edit, delete 
user can edit adn delete only own reviews

4. files 
1. task  - /projects/[id]/taks
2. result  - /projects/[id]/result
3. reviews - /projects/[id]/reviews/[revId]"""

CORS(student_project_api)


# Get all projects
@student_project_api.route('/lms/api/v1.0/<lname>/student_project/<int:termid>/<int:cid>', methods=['GET'])
@auth.login_required
def getStudentProjects(lname, termid, cid):
    result = StudentProject(lname).getStudentProjects(termid, g.user.mid, 0, cid, g.user.isTeacher, g.user.gid)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


# Get project and all details
@student_project_api.route('/lms/api/v1.0/<lname>/student_project/<int:pid>', methods=['GET'])
@auth.login_required
def getStudentProject(lname, pid):
    result = StudentProject(lname).getStudentProjectData(pid)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


# Get project and all details
@student_project_api.route('/lms/api/v1.0/<lname>/student_project', methods=['POST'])
@auth.login_required
def createStudentProject(lname):
    if (g.user.isTeacher == False):
        return jsonify(accessDenied=False), 401

    sp = StudentProject(lname)

    data = request.data
    if sys.version_info[0] < 3:
        dataDict = json.loads(data)
    else:
        dataDict = json.loads(data.decode('utf-8'))
    sp.type = dataDict['type']
    sp.student = dataDict['student']
    sp.teacher = g.user.mid
    sp.cid = dataDict['cid']
    sp.issuedate = dataDict['issuedate']
    sp.duedate = dataDict['duedate']
    sp.task = dataDict['task']
    sp.status = dataDict['status']

    result = sp.createStudentProject()

    if result == 0:
        return jsonify(success=False)

    return jsonify(result)


@student_project_api.route('/lms/api/v1.0/<lname>/student_project/<int:id>', methods=['PUT'])
@auth.login_required
def editStudentProject(lname, id):
    sp = StudentProject(lname)

    if (sp.getStudentProject(id) == False):
        return jsonify(projectNotFound=False), 404

    data = request.data
    if sys.version_info[0] < 3:
        dataDict = json.loads(data)
    else:
        dataDict = json.loads(data.decode('utf-8'))

    if (g.user.isTeacher == True):

        sp.teacher = g.user.mid

        if 'type' in dataDict:
            sp.type = dataDict['type']

        if 'issuedate' in dataDict:
            sp.issuedate = dataDict['issuedate']
        if 'duedate' in dataDict:
            sp.duedate = dataDict['duedate']
        if 'task' in dataDict:
            sp.task = dataDict['task']
        if 'status' in dataDict:
            sp.status = dataDict['status']
        if 'grade' in dataDict:
            sp.grade = dataDict['grade']
    else:
        if 'result' in dataDict:
            sp.result = dataDict['result']
        if 'status' in dataDict:
            sp.status = dataDict['status']

    result = sp.editStudentProject()

    if result != True:
        return jsonify(success=False)

    return jsonify(result)


@student_project_api.route('/lms/api/v1.0/<lname>/student_project/<int:id>', methods=['DELETE'])
@auth.login_required
def deleteStudentProject(lname, id):
    sp = StudentProject(lname)

    if (sp.getStudentProject(id) == False):
        return jsonify(projectNotFound=False), 404

    if (sp.teacher != g.user.mid):
        return jsonify("Teacher is not owner of project"), 401

    result = sp.deleteStudentProject()

    if result != True:
        return jsonify(success=False)

    return jsonify(result)


@student_project_api.route('/lms/api/v1.0/<lname>/student_project/projectreview/<int:pid>', methods=['POST'])
@auth.login_required
def createReview(lname, pid):
    current_app.logger.debug("HAH")

    if (g.user.isTeacher == False):
        return jsonify("Error: Not teacher"), 401

    data = request.data
    if sys.version_info[0] < 3:
        dataDict = json.loads(data)
    else:
        dataDict = json.loads(data.decode('utf-8'))

    if 'review' in dataDict:
        review = dataDict['review']

    if (review == None):
        return jsonify("missing review parameter"), 502

    sp = StudentProject(lname)

    if (sp.getStudentProject(pid) == False):
        return jsonify(projectNotFound=False), 404

    result = sp.createStudentProjectReview(g.user.mid, review)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@student_project_api.route('/lms/api/v1.0/<lname>/student_project/projectreview/<int:pid>', methods=['PUT'])
@auth.login_required
def editReview(lname, pid):
    data = request.data
    if sys.version_info[0] < 3:
        dataDict = json.loads(data)
    else:
        dataDict = json.loads(data.decode('utf-8'))
    if 'review' in dataDict:
        review = dataDict['review']

    if (review == None):
        return jsonify("missing review parameter"), 502

    sp = StudentProject(lname)
    if (sp.getStudentProjectReview(pid) == False):
        return jsonify(projectReviewNotFound=pid), 404

    if (sp.srmid != g.user.mid):
        return jsonify("Error: Not your review id"), 401

    result = sp.editStudentProjectReview(review)

    if result is False:
        return jsonify(success=False), 202

    return jsonify(result)


@student_project_api.route('/lms/api/v1.0/<lname>/student_project/projectreview/<int:pid>', methods=['DELETE'])
@auth.login_required
def deleteReview(lname, pid):
    sp = StudentProject(lname)
    if (sp.getStudentProjectReview(pid) == False):
        return jsonify(projectReviewNotFound=False), 404

    if (sp.srmid != g.user.mid):
        return jsonify("Error: Not your review id"), 401

    result = sp.deleteStudentProjectReview(pid)

    if result is False:
        return jsonify(success=False), 202

    return jsonify(result)


@student_project_api.route('/lms/api/v1.0/<lname>/student_project/<int:id>/tasks', methods=['GET'])
@auth.login_required
def getStudentProjectFiles(lname, id):
    sp = StudentProject(lname)
    if (sp.getStudentProjectReview(id) == False):
        return jsonify(projectReviewNotFound=False), 404

    if (g.user.isTeacher == False):
        if (sp.student != g.user.mid):
            return jsonify("Error: Not your project or not teacher"), 401

    result = File(lname).getStudentProjectTaskDirectoryFiles(id)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@student_project_api.route('/lms/api/v1.0/<lname>/student_project/<int:id>/result', methods=['GET'])
@auth.login_required
def getStudentProjectResultFiles(lname, id):
    sp = StudentProject(lname)
    if (sp.getStudentProjectReview(id) == False):
        return jsonify(projectReviewNotFound=False), 404

    if (g.user.isTeacher == False):
        if (sp.student != g.user.mid):
            return jsonify("Error: Not your project or not teacher"), 401

    result = File(lname).getStudentProjectResultDirectoryFiles(id)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@student_project_api.route(
    '/lms/api/v1.0/<lname>/student_project/teacher/<int:termid>/<int:cid>/<int:studentid>/<int:gid>',
    methods=['GET'])
@auth.login_required
@user_permission_modes("Справочники", "Практики и научно-исследовательские работы", ["Нет"])
def getTeacherProjects(lname, termid, cid, studentid, gid):
    if (g.user.isTeacher == False):
        return jsonify("Error: Not teacher"), 401

    result = StudentProject(lname).getStudentProjects(termid, studentid, g.user.mid, cid, g.user.isTeacher, gid)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@student_project_api.route('/lms/api/v1.0/<lname>/student_project/task/<int:id>/<filename>', methods=['GET'])
@auth.login_required
def getTaskFile(lname, id, filename):
    sp = StudentProject(lname)
    if (sp.getStudentProject(id) == False):
        return jsonify(projectReviewNotFound=False), 404

    if (g.user.isTeacher == False):
        if (sp.student != g.user.mid):
            return jsonify("Error: Not your project or not teacher"), 401
    path = File(lname).getStudentProjectTaskDirectory(id)
    ip_address = (
            request.headers.get('X-Forwarded-For') or
            request.headers.get('X-Real-IP') or
            request.environ.get("REMOTE_ADDR")
    )
    return File.send_from_directory(
        lname=lname,
        file_path=path,
        file_name=filename,
        action_source=request.environ.get("HTTP_USER_AGENT"),
        login=g.user.id,
        user_mid=g.user.mid,
        ip_address=ip_address
    )


@student_project_api.route('/lms/api/v1.0/<lname>/student_project/task/<int:id>', methods=['POST'])
@auth.login_required
def postTaskFile(lname, id):
    sp = StudentProject(lname)
    if (sp.getStudentProject(id) == False):
        return jsonify(projectReviewNotFound=False), 404

    if (g.user.isTeacher == False):
        return jsonify("Error: Not teacher owning project id"), 401

    ip_address = (
            request.headers.get('X-Forwarded-For') or
            request.headers.get('X-Real-IP') or
            request.environ.get("REMOTE_ADDR")
    )

    submitted_file = request.files['file']
    if submitted_file:
        File(lname).createStudentProjectTaskDirectory(id)
        file_path = File(lname).getStudentProjectTaskDirectory(id)
        return File.save_file_to_path(
            lname=lname,
            submitted_file=submitted_file,
            file_path=file_path,
            action_source=request.environ.get("HTTP_USER_AGENT"),
            login=g.user.id,
            user_mid=g.user.mid,
            ip_address=ip_address
        )

    return jsonify({'success': False, 'error': "upload file with key file"})


@student_project_api.route('/lms/api/v1.0/<lname>/student_project/task/<int:id>/<filename>', methods=['DELETE'])
@auth.login_required
def deleteTaskFile(lname, id, filename):
    sp = StudentProject(lname)
    if (sp.getStudentProject(id) == False):
        return jsonify(projectReviewNotFound=False), 404

    if (g.user.isTeacher == False):
        return jsonify("Error: Not teacher owning project id"), 401

    try:
        os.remove(os.path.join(File(lname).getStudentProjectTaskDirectory(id), filename))
        return jsonify(success=True)
    except:
        current_app.logger.error('deleteTaskFile(' + str(g.user.mid) + ',' + filename + ')')
        return jsonify(success=False), 404


@student_project_api.route('/lms/api/v1.0/<lname>/student_project/result/<int:id>', methods=['POST'])
@auth.login_required
def postResultFile(lname, id):
    sp = StudentProject(lname)
    if (sp.getStudentProject(id) == False):
        return jsonify(projectReviewNotFound=False), 404

    if (g.user.mid != sp.student):
        return jsonify("Error: Not student for project id"), 401

    ip_address = (
            request.headers.get('X-Forwarded-For') or
            request.headers.get('X-Real-IP') or
            request.environ.get("REMOTE_ADDR")
    )

    submitted_file = request.files['file']
    if submitted_file:
        File(lname).createStudentProjectResultDirectory(id)
        file_path = File(lname).getStudentProjectResultDirectory(id)
        return File.save_file_to_path(
            lname=lname,
            submitted_file=submitted_file,
            file_path=file_path,
            action_source=request.environ.get("HTTP_USER_AGENT"),
            login=g.user.id,
            user_mid=g.user.mid,
            ip_address=ip_address
        )

    return jsonify({'success': False, 'error': "upload file with key file"})


@student_project_api.route('/lms/api/v1.0/<lname>/student_project/result/<int:id>/<filename>', methods=['DELETE'])
@auth.login_required
def deleteResultFile(lname, id, filename):
    sp = StudentProject(lname)
    if (sp.getStudentProject(id) == False):
        return jsonify(projectReviewNotFound=False), 404

    if (g.user.mid != sp.student):
        return jsonify("Error: Not student for project id"), 401

    try:
        os.remove(os.path.join(File(lname).getStudentProjectResultDirectory(id), filename))
        return jsonify(success=True)
    except:
        current_app.logger.error('deleteTaskFile(' + str(g.user.mid) + ',' + filename + ')')
        return jsonify(success=False), 404


@student_project_api.route('/lms/api/v1.0/<lname>/student_project/result/<int:id>/<filename>', methods=['GET'])
@auth.login_required
def getResultFile(lname, id, filename):
    sp = StudentProject(lname)
    if (sp.getStudentProject(id) == False):
        return jsonify(projectReviewNotFound=False), 404

    if (g.user.isTeacher == False):
        if (sp.student != g.user.mid):
            return jsonify("Error: Not student for project id or not teacher"), 401
    ip_address = (
            request.headers.get('X-Forwarded-For') or
            request.headers.get('X-Real-IP') or
            request.environ.get("REMOTE_ADDR")
    )
    return File.send_from_directory(
        lname=lname,
        file_path=File(lname).getStudentProjectResultDirectory(id),
        file_name=filename,
        action_source=request.environ.get("HTTP_USER_AGENT"),
        login=g.user.id,
        user_mid=g.user.mid,
        ip_address=ip_address
    )


@student_project_api.route('/lms/api/v1.0/<lname>/student_project/review/<int:rid>/<filename>', methods=['GET'])
@auth.login_required
def getReviewFile(lname, rid, filename):
    sp = StudentProject(lname)

    if (sp.getStudentProjectReview(rid) == False):
        return jsonify(projectReviewNotFound=False), 404

    if (sp.getStudentProject(sp.srproject) == False):
        return jsonify(projectReviewNotFound=False), 404

    if (g.user.isTeacher == False):
        if (sp.student != g.user.mid):
            return jsonify("Error: Not student for project id or not teacher"), 401
    path = File(lname).getStudentProjectReviewDirectory(sp.srproject, rid)
    ip_address = (
            request.headers.get('X-Forwarded-For') or
            request.headers.get('X-Real-IP') or
            request.environ.get("REMOTE_ADDR")
    )
    return File.send_from_directory(
        lname=lname,
        file_path=path,
        file_name=filename,
        action_source=request.environ.get("HTTP_USER_AGENT"),
        login=g.user.id,
        user_mid=g.user.mid,
        ip_address=ip_address
    )


@student_project_api.route('/lms/api/v1.0/<lname>/student_project/review/<int:rid>', methods=['POST'])
@auth.login_required
def postReviewFile(lname, rid):
    sp = StudentProject(lname)

    if (sp.getStudentProjectReview(rid) == False):
        return jsonify(projectReviewNotFound=False), 404

    if (sp.getStudentProject(sp.srproject) == False):
        return jsonify(projectReviewNotFound=False), 404

    if (g.user.mid != sp.srmid):
        return jsonify("Error: Not teacher owning review id"), 401

    ip_address = (
            request.headers.get('X-Forwarded-For') or
            request.headers.get('X-Real-IP') or
            request.environ.get("REMOTE_ADDR")
    )

    submitted_file = request.files['file']
    if submitted_file:
        File(lname).createStudentProjectReviewDirectory(sp.srproject, rid)
        file_path = File(lname).getStudentProjectReviewDirectory(sp.srproject, rid)
        return File.save_file_to_path(
            lname=lname,
            submitted_file=submitted_file,
            file_path=file_path,
            action_source=request.environ.get("HTTP_USER_AGENT"),
            login=g.user.id,
            user_mid=g.user.mid,
            ip_address=ip_address
        )

    return jsonify({'success': False, 'error': "upload file with key file"})


@student_project_api.route('/lms/api/v1.0/<lname>/student_project/review/<int:rid>/<filename>', methods=['DELETE'])
@auth.login_required
def deleteReviewFile(lname, rid, filename):
    sp = StudentProject(lname)

    if (sp.getStudentProjectReview(rid) == False):
        return jsonify(projectReviewNotFound=False), 404

    if (sp.getStudentProject(sp.srproject) == False):
        return jsonify(projectNotFound=False), 404

    if (g.user.mid != sp.srmid):
        return jsonify("Error: Not teacher owning review id"), 401

    try:
        os.remove(os.path.join(File(lname).getStudentProjectReviewDirectory(sp.srproject, rid), filename))
        return jsonify(success=True)
    except:
        current_app.logger.error('deleteTaskFile(' + str(g.user.mid) + ',' + filename + ')')
        return jsonify(success=False), 404
