from flask import Blueprint, request, jsonify, g
from flask_cors import CORS

from LMSAPI.api.Models.TeacherReview import TeacherReviewService, TeacherReviewVoteService
from LMSAPI.api.Views.TokenAPI import auth

teacher_reviews_api = Blueprint('teacher_reviews_api', __name__)
CORS(teacher_reviews_api)


@teacher_reviews_api.route("/lms/api/v1.0/<lname>/reviews/statistics", methods=["GET"])
@auth.login_required
def get_review_statistics(lname):
    try:
        limit = request.args.get("limit", type=int)
        offset = request.args.get("offset", type=int)
        search = request.args.get("search", type=str)
        result = TeacherReviewService(lname).get_review_statistics(
            limit=limit,
            offset=offset,
            search=search
        )
        return jsonify(result)
    except Exception as e:
        return jsonify({"error": str(e)}), 500


@teacher_reviews_api.route("/lms/api/v1.0/<lname>/reviews/teacher/<int:teacher_id>", methods=["GET"])
@auth.login_required
def get_reviews_by_teacher(lname, teacher_id):
    try:
        limit = request.args.get("limit", type=int, default=10)
        offset = request.args.get("offset", type=int, default=0)
        result = TeacherReviewService(lname).get_reviews_by_teacher(teacher_id, limit=limit, offset=offset)
        return jsonify(result)
    except Exception as e:
        return jsonify({"error": str(e)}), 500


@teacher_reviews_api.route("/lms/api/v1.0/<lname>/reviews", methods=["POST"])
@auth.login_required
def create_review(lname):
    data = request.get_json()
    teacher_id = data.get("teacher_id")
    student_id = g.user.mid
    comment = data.get("comment", "").strip()
    scores = data.get("scores", [])
    anonymous = data.get("anonymous", True)

    if not teacher_id:
        return jsonify({"error": "teacher_id обязательно"}), 400

    if not scores or not isinstance(scores, list):
        return jsonify({"error": "Должен быть хотя бы один критерий с оценкой"}), 400

    try:
        review = TeacherReviewService(lname).create_review(
            teacher_id=teacher_id,
            student_id=student_id,
            comment=comment,
            anonymous=anonymous,
            scores=scores
        )
        return jsonify({"review_id": review.id}), 201
    except ValueError as e:
        return jsonify({"error": str(e)}), 400


@teacher_reviews_api.route("/lms/api/v1.0/<lname>/reviews/<int:review_id>", methods=["DELETE"])
@auth.login_required
def delete_review(lname, review_id):
    try:
        TeacherReviewService(lname).delete_review(review_id)
        return jsonify({"success": True})
    except ValueError as e:
        return jsonify({"error": str(e)}), 404


@teacher_reviews_api.route("/lms/api/v1.0/<lname>/reviews/vote", methods=["POST"])
@auth.login_required
def vote_review(lname):
    data = request.get_json()
    review_id = data.get("review_id")
    user_id = g.user.mid
    vote_type = data.get("vote_type")

    try:
        result = TeacherReviewVoteService(lname).vote_review(
            review_id=review_id,
            user_id=user_id,
            vote_type=vote_type
        )
        return jsonify(result), 200
    except ValueError as e:
        return jsonify({"error": str(e)}), 400
