import os
import pickle
from collections import defaultdict
from decimal import Decimal
from pathlib import Path
from statistics import mean

from flask import current_app
from sqlalchemy import text

from LMSAPI.api.Models.File import File
from LMSAPI.api.Models.QuestionAnswers import QuestionAnswers
from LMSAPI.api.Models.QuestionSet import QuestionSet
from LMSAPI.api.Models.Schoolyear import Schoolyear


class Dashboard:
    @staticmethod
    def university_main_stat(lname: str) -> dict:
        sql = """
        WITH pps AS(
        SELECT p.mid
        FROM people p
        LEFT JOIN xp_personal_file xpf
        ON p.mid = xpf.mid
        WHERE xpf.category_stat = 'Преподавательский состав'
        ),

        pps_gp AS(
            SELECT p.mid
            FROM people p
            LEFT JOIN xp_personal_file xpf
            ON p.mid = xpf.mid
            WHERE xpf.category_stat = 'Преподавательский состав'
            AND (xpf.perstype = 'Гражданский персонал' OR xpf.perstype IS NULL)
        ),
        pps_v AS (
            SELECT p.mid
            FROM people p
            LEFT JOIN xp_personal_file xpf
            ON p.mid = xpf.mid
            WHERE xpf.category_stat = 'Преподавательский состав'
            AND xpf.perstype = 'Военнослужащие'
        ),

        pps_doctor AS (
            SELECT
                xpf.mid
            FROM xp_personal_file xpf

            LEFT JOIN academicdegree ad
            ON xpf."AcademicDegree" = ad.agid
            WHERE idokin = 618
            AND xpf.category_stat = 'Преподавательский состав'
        ),

        pps_kandidat AS (
            SELECT
                xpf.mid
            FROM xp_personal_file xpf

            LEFT JOIN academicdegree ad
            ON xpf."AcademicDegree" = ad.agid
            WHERE idokin = 619
            AND xpf.category_stat = 'Преподавательский состав'
        ),
        pps_professor AS (
            SELECT
                xpf.mid
            FROM xp_personal_file xpf
            LEFT JOIN academictitle act
            ON xpf."AcademicTitle" = act.atid
            WHERE act.name = 'Профессор'
            AND xpf.category_stat = 'Преподавательский состав'
        ),

        pps_docent AS (
            SELECT
                xpf.mid
            FROM xp_personal_file xpf
            LEFT JOIN academictitle act
            ON xpf."AcademicTitle" = act.atid
            WHERE act.name = 'Доцент'
            AND xpf.category_stat = 'Преподавательский состав'
        ),

        pps_doctorant AS (
            SELECT xpf.category
            FROM xp_personal_file xpf
            LEFT JOIN prep_struc_category psc
            ON xpf.category = psc.cid
            WHERE psc.name = 'Докторант'
            AND xpf.category_stat = 'Преподавательский состав'
        ),


        pps_aduct AS (
            SELECT xpf.category
            FROM xp_personal_file xpf
            LEFT JOIN prep_struc_category psc
            ON xpf.category = psc.cid
            WHERE psc.name = 'Адъюнкт'
            AND xpf.category_stat = 'Преподавательский состав'
        )


        SELECT
            ROUND((SELECT COUNT(*) FROM pps_gp) * 100.0 / (SELECT COUNT(*) FROM pps))::int perc_civil,
            ROUND((SELECT COUNT(*) FROM pps_v) * 100.0 / (SELECT COUNT(*) FROM pps))::int perc_voin,
            ROUND((SELECT COUNT(*) FROM pps_doctor) * 100.0 / (SELECT COUNT(*) FROM pps))::int perc_doctor,
            ROUND((SELECT COUNT(*) FROM pps_kandidat) * 100.0 / (SELECT COUNT(*) FROM pps))::int perc_kandidat,
            ROUND((SELECT COUNT(*) FROM pps_docent) * 100.0 / (SELECT COUNT(*) FROM pps))::int perc_docent,
            ROUND((SELECT COUNT(*) FROM pps_doctorant) * 100.0 / (SELECT COUNT(*) FROM pps))::int perc_doctorant,
            ROUND((SELECT COUNT(*) FROM pps_aduct) * 100.0 / (SELECT COUNT(*) FROM pps))::int perc_aduct
        """
        conn = current_app.ms.db(lname).connect()
        query = conn.execute(sql)
        pps_perc = query.first()
        if not pps_perc:
            return False
        pps_perc = dict(pps_perc)
        return pps_perc

    @staticmethod
    def get_university_age_stat(lname: str) -> dict:
        sql = """
        WITH up_tp_35_pps AS (
            SELECT COUNT(*) cnt
        FROM people p
        LEFT JOIN xp_personal_file xpf
        ON p.mid = xpf.mid
        WHERE xpf.category_stat = 'Преподавательский состав'
        AND p.birthdate > current_date - interval '35 years'
        ),

        bw_36_to_46 AS (
        SELECT COUNT(*) cnt
        FROM people p
        LEFT JOIN xp_personal_file xpf
        ON p.mid = xpf.mid
        WHERE xpf.category_stat = 'Преподавательский состав'
        AND p.birthdate BETWEEN current_date - interval '46 years' AND current_date - interval '35 years'
        ),

        bw_46_to_55 AS (
        SELECT COUNT(*) cnt
        FROM people p
        LEFT JOIN xp_personal_file xpf
        ON p.mid = xpf.mid
        WHERE xpf.category_stat = 'Преподавательский состав'
        AND p.birthdate BETWEEN current_date - interval '56 years' AND current_date - interval '46 years'
        ),

        bw_56_to_65 AS (
        SELECT COUNT(*) cnt
        FROM people p
        LEFT JOIN xp_personal_file xpf
        ON p.mid = xpf.mid
        WHERE xpf.category_stat = 'Преподавательский состав'
        AND p.birthdate BETWEEN current_date - interval '66 years' AND current_date - interval '56 years'
        ),

        over_65 AS (
        SELECT COUNT(*) cnt
        FROM people p
        LEFT JOIN xp_personal_file xpf
        ON p.mid = xpf.mid
        WHERE xpf.category_stat = 'Преподавательский состав'
        AND p.birthdate <= current_date - interval '66 years'
        )

        SELECT
        ROUND((SELECT cnt FROM up_tp_35_pps) * 100.0 / COUNT(*))::int up_to_35,
        ROUND((SELECT cnt FROM bw_36_to_46) * 100.0 / COUNT(*))::int bw_36_to_46,
        ROUND((SELECT cnt FROM bw_46_to_55) * 100.0 / COUNT(*))::int bw_46_to_55,
        ROUND((SELECT cnt FROM bw_56_to_65) * 100.0 / COUNT(*))::int bw_56_to_65,
        ROUND((SELECT cnt FROM over_65)     * 100.0 / COUNT(*))::int over_65
        FROM xp_personal_file xpf
        INNER JOIN people p
        ON p.mid = xpf.mid
        WHERE xpf.category_stat = 'Преподавательский состав'
        AND p.birthdate IS NOT NULL
        """
        conn = current_app.ms.db(lname).connect()
        query = conn.execute(sql)
        pps_perc = query.first()
        if not pps_perc:
            return False
        pps_perc = dict(pps_perc)
        return pps_perc

    @staticmethod
    def get_university_mark_stat(lname) -> list:
        sql = """
        WITH RECURSIVE current_school_year AS (
          SELECT
            begdate
          , enddate
          , xp_key
          FROM
            school_year
          WHERE
            current_date BETWEEN begdate AND enddate
        )
        , months_range AS (
          SELECT
            generate_series(
              date_part('year', csy.begdate)::integer * 12 + date_part('month', csy.begdate)::integer - 1
            , date_part('year', csy.enddate)::integer * 12 + date_part('month', csy.enddate)::integer - 1
            ) year_month
          FROM
            current_school_year csy
        )
        , grade_stats AS (
        SELECT 
            EXTRACT(YEAR FROM vs.lesson_date) AS "year",
            EXTRACT(MONTH FROM vs.lesson_date) AS "month",
            ROUND(CAST(SUM(xp_sum_grade(gr.grade)) / NULLIF(SUM(xp_grades_count(gr.grade)), 0) AS NUMERIC), 2)::float AS grade	
        FROM school_year sy
        JOIN current_school_year csy ON csy.xp_key = sy.xp_key
        JOIN group_history gh ON gh.school_year = sy.xp_key
        JOIN groupname g ON g.gid = gh.gid AND g.cid < 0
        JOIN vw_schedule vs ON vs.school_year = sy.xp_key and vs.main_gid = g.gid
        JOIN nnz_sh_grades gr ON gr.sheid = vs.sheid
        GROUP BY EXTRACT(YEAR FROM vs.lesson_date), EXTRACT(MONTH FROM vs.lesson_date)
        )
        ,month_year AS (
          SELECT
            (year_month / 12)::integer "year"
          , (year_month % 12 + 1)::integer "month"
          FROM
            months_range
        )
        SELECT my."year", my."month", gs.grade
        FROM month_year my
        LEFT JOIN grade_stats gs ON gs."year" = my."year" AND gs."month" = my."month";
        """

        conn = current_app.ms.db(lname).connect()
        query = conn.execute(text(sql))
        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    @staticmethod
    def get_university_mark_stat_by_faculty(lname, faculty) -> list:
        sql = """
            WITH RECURSIVE current_school_year AS (
              SELECT
                begdate
              , enddate
              , xp_key
              FROM
                school_year
              WHERE
                current_date BETWEEN begdate AND enddate
            )
            , months_range AS (
              SELECT
                generate_series(
                  date_part('year', csy.begdate)::integer * 12 + date_part('month', csy.begdate)::integer - 1
                , date_part('year', csy.enddate)::integer * 12 + date_part('month', csy.enddate)::integer - 1
                ) year_month
              FROM
                current_school_year csy
            )
            , grade_stats AS (
            SELECT 
                EXTRACT(YEAR FROM vs.lesson_date) AS "year",
                EXTRACT(MONTH FROM vs.lesson_date) AS "month",
                ROUND(CAST(SUM(xp_sum_grade(gr.grade)) / NULLIF(SUM(xp_grades_count(gr.grade)), 0) AS NUMERIC), 2)::float AS grade	
            FROM school_year sy
            JOIN current_school_year csy ON csy.xp_key = sy.xp_key
            JOIN group_history gh ON gh.school_year = sy.xp_key
            JOIN groupname g ON g.gid = gh.gid AND g.cid < 0 AND  g.idfaculty = :faculty
            JOIN vw_schedule vs ON vs.school_year = sy.xp_key and vs.main_gid = g.gid
            JOIN nnz_sh_grades gr ON gr.sheid = vs.sheid
            GROUP BY EXTRACT(YEAR FROM vs.lesson_date), EXTRACT(MONTH FROM vs.lesson_date)
            )
            ,month_year AS (
              SELECT
                (year_month / 12)::integer "year"
              , (year_month % 12 + 1)::integer "month"
              FROM
                months_range
            )
            SELECT my."year", my."month", gs.grade
            FROM month_year my
            LEFT JOIN grade_stats gs ON gs."year" = my."year" AND gs."month" = my."month";
            """

        conn = current_app.ms.db(lname).connect()
        stmt = text(sql)
        stmt = stmt.bindparams(faculty=faculty)
        query = conn.execute(stmt)
        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    @staticmethod
    def get_count_xp_personal_file(lname: str, name: str):
        conn = current_app.ms.db(lname).connect()

        count_pps = Dashboard.get_count_pps(lname)

        select_count = """
            SELECT ROUND(COUNT(*) * 100.0 / :count_pps, 2) AS percentage
            FROM xp_personal_file pf
            JOIN people p on p.mid = pf.mid
            WHERE pf.workstatus = 'Работает' 
                AND (pf.category_stat = 'Руководящий состав' 
                     OR pf.category_stat = 'Преподавательский состав' 
                     OR pf.category_stat = 'Научный состав')
                AND pf.perstype = :name
            """
        stmt = text(select_count)
        stmt = stmt.bindparams(name=name, count_pps=count_pps)
        query = conn.execute(stmt)
        result = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]
        # Преобразуем Decimal в float для JSON
        for record in result:
            for k, v in record.items():
                if isinstance(v, Decimal):
                    record[k] = float(v)
        return result[0]["percentage"] if result else None

    @staticmethod
    def get_count_xp_personal_file_by_faculty(lname: str, name: str, faculty: int):
        conn = current_app.ms.db(lname).connect()

        select_count = """
            WITH RECURSIVE dep AS 
            ( 
              SELECT id, "name", owner_dep, "name"::text AS fullname 
              FROM vw_divisions 
              WHERE owner_dep IS NULL OR owner_dep = 0 
              UNION ALL 
              SELECT vd.id, vd."name", vd.owner_dep, CONCAT(dep."name", ' / ', vd."name") AS fullname 
                FROM dep, vw_divisions vd
                WHERE vd.owner_dep = dep.id
            ),
             mids AS (
            SELECT DISTINCT vs.mid 
            FROM dep 
            JOIN vw_staff vs ON dep.id = vs.depid
            where dep.id = :faculty or owner_dep = :faculty
            )
            SELECT ROUND(COUNT(*) * 100.0 / (select count(mid) from mids), 2) AS percentage
            FROM mids
            JOIN xp_personal_file pf on mids.mid = pf.mid
            JOIN people p on p.mid = pf.mid
            WHERE pf.workstatus = 'Работает' 
                AND (pf.category_stat = 'Руководящий состав' 
                     OR pf.category_stat = 'Преподавательский состав' 
                     OR pf.category_stat = 'Научный состав')
                AND pf.perstype = :name
            """
        stmt = text(select_count)
        stmt = stmt.bindparams(name=name, faculty=faculty)
        query = conn.execute(stmt)
        result = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]
        # Преобразуем Decimal в float для JSON
        for record in result:
            for k, v in record.items():
                if isinstance(v, Decimal):
                    record[k] = float(v)
        return result[0]["percentage"] if result else None

    @staticmethod
    def get_count_xp_personal_file_fil_okin(lname: str, name: str):
        conn = current_app.ms.db(lname).connect()

        count_pps = Dashboard.get_count_pps(lname)

        select_count = """
            select ROUND(COUNT(*) * 100.0 / :count_pps, 2) AS percentage
            from xp_personal_file pf
            JOIN people p on p.mid = pf.mid
            left join academicdegree a on pf."AcademicDegree" = a.agid
            left join okin o on o.idokin = a.idokin
            where pf.workstatus = 'Работает' 
                and (pf.category_stat = 'Руководящий состав' or pf.category_stat = 'Преподавательский состав' or pf.category_stat = 'Научный состав')
                and o.name = :name
            """
        stmt = text(select_count)
        stmt = stmt.bindparams(name=name, count_pps=count_pps)
        query = conn.execute(stmt)
        result = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]
        # Преобразуем Decimal в float для JSON
        for record in result:
            for k, v in record.items():
                if isinstance(v, Decimal):
                    record[k] = float(v)
        return result[0]["percentage"] if result else None

    @staticmethod
    def get_count_xp_personal_file_fil_okin_by_faculty(lname: str, name: str, faculty: int):
        conn = current_app.ms.db(lname).connect()

        select_count = """
            WITH RECURSIVE dep AS 
            ( 
              SELECT id, "name", owner_dep, "name"::text AS fullname 
              FROM vw_divisions 
              WHERE owner_dep IS NULL OR owner_dep = 0 
              UNION ALL 
              SELECT vd.id, vd."name", vd.owner_dep, CONCAT(dep."name", ' / ', vd."name") AS fullname 
                FROM dep, vw_divisions vd
                WHERE vd.owner_dep = dep.id
            ),
             mids AS (
            SELECT DISTINCT vs.mid 
            FROM dep 
            JOIN vw_staff vs ON dep.id = vs.depid
            where dep.id = :faculty or owner_dep = :faculty
            )
            SELECT ROUND(COUNT(*) * 100.0 / (select count(mid) from mids), 2) AS percentage
            FROM mids
            JOIN xp_personal_file pf on mids.mid = pf.mid
            JOIN people p on p.mid = pf.mid
            left join academicdegree a on pf."AcademicDegree" = a.agid
            left join okin o on o.idokin = a.idokin
            where pf.workstatus = 'Работает' 
                and (pf.category_stat = 'Руководящий состав' or pf.category_stat = 'Преподавательский состав' or pf.category_stat = 'Научный состав')
                and o.name = :name
                """
        stmt = text(select_count)
        stmt = stmt.bindparams(name=name, faculty=faculty)
        query = conn.execute(stmt)
        result = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]
        # Преобразуем Decimal в float для JSON
        for record in result:
            for k, v in record.items():
                if isinstance(v, Decimal):
                    record[k] = float(v)
        return result[0]["percentage"] if result else None

    @staticmethod
    def get_count_xp_personal_file_fil_academic_title(lname: str, name: str):
        conn = current_app.ms.db(lname).connect()

        count_pps = Dashboard.get_count_pps(lname)

        select_count = """
            select ROUND(COUNT(*) * 100.0 / :count_pps, 2) AS percentage
            from xp_personal_file pf
            JOIN people p on p.mid = pf.mid
            left join AcademicTitle a on pf."AcademicTitle" = a.atid
            where pf.workstatus = 'Работает' 
                and (pf.category_stat = 'Руководящий состав' or pf.category_stat = 'Преподавательский состав' or pf.category_stat = 'Научный состав')
                and a.name = :name
            """
        stmt = text(select_count)
        stmt = stmt.bindparams(name=name, count_pps=count_pps)
        query = conn.execute(stmt)
        result = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]
        # Преобразуем Decimal в float для JSON
        for record in result:
            for k, v in record.items():
                if isinstance(v, Decimal):
                    record[k] = float(v)
        return result[0]["percentage"] if result else None

    @staticmethod
    def get_count_xp_personal_file_fil_academic_title_by_faculty(lname: str, name: str, faculty: int):
        conn = current_app.ms.db(lname).connect()

        select_count = """
            WITH RECURSIVE dep AS 
            ( 
              SELECT id, "name", owner_dep, "name"::text AS fullname 
              FROM vw_divisions 
              WHERE owner_dep IS NULL OR owner_dep = 0 
              UNION ALL 
              SELECT vd.id, vd."name", vd.owner_dep, CONCAT(dep."name", ' / ', vd."name") AS fullname 
                FROM dep, vw_divisions vd
                WHERE vd.owner_dep = dep.id
            ),
             mids AS (
            SELECT DISTINCT vs.mid 
            FROM dep 
            JOIN vw_staff vs ON dep.id = vs.depid
            where dep.id = :faculty or owner_dep = :faculty
            )
            select ROUND(COUNT(*) * 100.0 / (select count(mid) from mids), 2) AS percentage
            FROM mids
            JOIN xp_personal_file pf on mids.mid = pf.mid
            JOIN people p on p.mid = pf.mid
            left join AcademicTitle a on pf."AcademicTitle" = a.atid
            where pf.workstatus = 'Работает' 
                and (pf.category_stat = 'Руководящий состав' or pf.category_stat = 'Преподавательский состав' or pf.category_stat = 'Научный состав')
                and a.name = :name
            """
        stmt = text(select_count)
        stmt = stmt.bindparams(name=name, faculty=faculty)
        query = conn.execute(stmt)
        result = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]
        # Преобразуем Decimal в float для JSON
        for record in result:
            for k, v in record.items():
                if isinstance(v, Decimal):
                    record[k] = float(v)
        return result[0]["percentage"] if result else None

    @staticmethod
    def get_count_pps(lname: str):
        conn = current_app.ms.db(lname).connect()

        select_count = """
            select count(*)
            from xp_personal_file pf
            JOIN people p on p.mid = pf.mid
            where pf.workstatus = 'Работает' and (pf.category_stat = 'Руководящий состав' or pf.category_stat = 'Преподавательский состав' or pf.category_stat = 'Научный состав')
            """
        stmt = text(select_count)
        query = conn.execute(stmt)
        count = query.scalar()
        return count

    @staticmethod
    def get_count_personnel_pps(lname: str):
        conn = current_app.ms.db(lname).connect()

        count_pps = Dashboard.get_count_pps(lname)

        select_count = """
            select ROUND(COUNT(*) * 100.0 / :count_pps, 2) AS percentage
            from xp_personal_file pf
            JOIN people p on p.mid = pf.mid
            left join academicdegree a on pf."AcademicDegree" = a.agid
            left join AcademicTitle at on pf."AcademicTitle" = at.atid
            left join okin o on o.idokin = a.idokin
            where pf.workstatus = 'Работает' 
                and (pf.category_stat = 'Руководящий состав' or pf.category_stat = 'Преподавательский состав' or pf.category_stat = 'Научный состав')
                and (at.name = 'Доцент' or at.name = 'Профессор' or o.name = 'Доктор наук' or o.name = 'Кандидат наук')
            """
        stmt = text(select_count)
        stmt = stmt.bindparams(count_pps=count_pps)
        query = conn.execute(stmt)
        result = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]
        # Преобразуем Decimal в float для JSON
        for record in result:
            for k, v in record.items():
                if isinstance(v, Decimal):
                    record[k] = float(v)
        return result[0]["percentage"] if result else None

    @staticmethod
    def get_count_personnel_pps_by_faculty(lname: str, faculty):
        conn = current_app.ms.db(lname).connect()

        select_count = """
            WITH RECURSIVE dep AS 
            ( 
              SELECT id, "name", owner_dep, "name"::text AS fullname 
              FROM vw_divisions 
              WHERE owner_dep IS NULL OR owner_dep = 0 
              UNION ALL 
              SELECT vd.id, vd."name", vd.owner_dep, CONCAT(dep."name", ' / ', vd."name") AS fullname 
                FROM dep, vw_divisions vd
                WHERE vd.owner_dep = dep.id
            ),
             mids AS (
            SELECT DISTINCT vs.mid 
            FROM dep 
            JOIN vw_staff vs ON dep.id = vs.depid
            where dep.id = :faculty or owner_dep = :faculty
            )
            SELECT ROUND(COUNT(*) * 100.0 / (select count(mid) from mids), 2) AS percentage
            FROM mids
            JOIN xp_personal_file pf on mids.mid = pf.mid
            JOIN people p on p.mid = pf.mid
            WHERE pf.workstatus = 'Работает' 
                AND (pf.category_stat = 'Руководящий состав' 
                     OR pf.category_stat = 'Преподавательский состав' 
                     OR pf.category_stat = 'Научный состав')
                AND pf.perstype = 'Гражданский персонал'
            """
        stmt = text(select_count)
        stmt = stmt.bindparams(faculty=faculty)
        query = conn.execute(stmt)
        result = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]
        # Преобразуем Decimal в float для JSON
        for record in result:
            for k, v in record.items():
                if isinstance(v, Decimal):
                    record[k] = float(v)
        return result[0]["percentage"] if result else None

    @staticmethod
    def get_avg_gia_mark(lname: str):
        conn = current_app.ms.db(lname).connect()

        sql = """
            WITH sy AS
            (
                SELECT vs.school_year
                FROM eventtools et
                JOIN nnz_schedule nsc ON nsc.pair_type_id = et.typeid
                JOIN vw_schedule vs ON vs.sheid = nsc.sheid
                JOIN journalcertification jc ON jc.f_nnz_schedule = vs.sheid
                JOIN term_grades tg ON tg.jcid = jc.jcid
                JOIN school_year sy ON sy.xp_key = vs.school_year
                WHERE et.processtypeid = 8
                ORDER BY sy.begdate desc
                LIMIT 1
            )
            SELECT 
                et.typeid,
                et.typename,
                ROUND(AVG(CAST(tg.grade AS NUMERIC)), 2) AS avg_grade
            FROM eventtools et
            JOIN sy ON True
            LEFT JOIN nnz_schedule nsc ON nsc.pair_type_id = et.typeid
            LEFT JOIN vw_schedule vs ON vs.sheid = nsc.sheid AND vs.school_year = sy.school_year	
            LEFT JOIN journalcertification jc ON jc.f_nnz_schedule = vs.sheid					
            LEFT JOIN term_grades tg ON tg.jcid = jc.jcid
            WHERE et.processtypeid = 8 AND tg.grade ~ '^\d+(\.\d+)?$'
            GROUP BY et.typeid, et.typename
            ORDER BY et.typeid;

            """
        stmt = text(sql)
        query = conn.execute(stmt)
        data = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]
        # Преобразуем Decimal в float для JSON
        for record in data:
            for k, v in record.items():
                if isinstance(v, Decimal):
                    record[k] = float(v)
        return data

    @staticmethod
    def get_avg_gia_mark_by_faculty(lname: str, faculty: int):
        conn = current_app.ms.db(lname).connect()

        sql = """
            WITH sy AS
            (
                SELECT vs.school_year
                FROM eventtools et
                JOIN nnz_schedule nsc ON nsc.pair_type_id = et.typeid
                JOIN vw_schedule vs ON vs.sheid = nsc.sheid
                JOIN journalcertification jc ON jc.f_nnz_schedule = vs.sheid
                JOIN term_grades tg ON tg.jcid = jc.jcid
                JOIN school_year sy ON sy.xp_key = vs.school_year
                WHERE et.processtypeid = 8
                ORDER BY sy.begdate desc
                LIMIT 1
            )
            SELECT 
                et.typeid,
                et.typename,
                ROUND(AVG(CAST(tg.grade AS NUMERIC)), 2) AS avg_grade
            FROM eventtools et
            JOIN sy ON True
            LEFT JOIN nnz_schedule nsc ON nsc.pair_type_id = et.typeid
            LEFT JOIN vw_schedule vs ON vs.sheid = nsc.sheid AND vs.school_year = sy.school_year
            LEFT JOIN groupname gn ON gn.gid = vs.main_gid AND gn.idfaculty = :faculty
            LEFT JOIN journalcertification jc ON jc.f_nnz_schedule = vs.sheid AND jc.gid = gn.gid
            LEFT JOIN term_grades tg ON tg.jcid = jc.jcid
            WHERE et.processtypeid = 8 AND tg.grade ~ '^\d+(\.\d+)?$'
            GROUP BY et.typeid, et.typename
            ORDER BY et.typeid;
                """
        stmt = text(sql)
        stmt = stmt.bindparams(faculty=faculty)
        query = conn.execute(stmt)
        data = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]
        # Преобразуем Decimal в float для JSON
        for record in data:
            for k, v in record.items():
                if isinstance(v, Decimal):
                    record[k] = float(v)
        return data

    @staticmethod
    def get_pps_age_stat_by_faculty(lname: str, faculty: int):
        conn = current_app.ms.db(lname).connect()

        sql = """
            WITH RECURSIVE dep AS 
            (
              SELECT id, "name", owner_dep, "name"::text AS fullname 
              FROM vw_divisions 
              WHERE owner_dep IS NULL OR owner_dep = 0 
              UNION ALL 
              SELECT vd.id, vd."name", vd.owner_dep, CONCAT(dep."name", ' / ', vd."name") AS fullname 
                FROM dep, vw_divisions vd
                WHERE vd.owner_dep = dep.id
            ),
            people_data AS (
              SELECT p.*, 
                     CASE 
                         WHEN p.birthdate IS NOT NULL THEN EXTRACT(YEAR FROM AGE(CURRENT_DATE, p.birthdate))::int 
                         ELSE NULL 
                     END AS age
              FROM dep 
              JOIN vw_staff vs ON dep.id = vs.depid
              JOIN people p ON vs.mid = p.mid
              WHERE dep.id = :faculty OR dep.owner_dep = :faculty
            ),
            age_groups AS (
              SELECT 
                  CASE
                      WHEN age IS NULL THEN 'Не указана дата рождения'
                      WHEN age < 30 THEN 'до 30 лет'
                      WHEN age BETWEEN 31 AND 40 THEN 'от 31 до 40 лет'
                      WHEN age BETWEEN 41 AND 50 THEN 'от 41 до 50 лет'
                      WHEN age BETWEEN 51 AND 60 THEN 'от 51 до 60 лет'
                      WHEN age > 60 THEN 'старше 60 лет'
                  END AS age_group,
                  COUNT(*) AS count
              FROM people_data
              GROUP BY age_group
            ),
            template_groups AS (
              SELECT unnest(ARRAY[
                'до 30 лет', 
                'от 31 до 40 лет', 
                'от 41 до 50 лет', 
                'от 51 до 60 лет', 
                'старше 60 лет', 
                'Не указана дата рождения'
              ]) AS age_group
            ),
            total_count AS (
              SELECT SUM(count) AS total FROM age_groups
            )
            SELECT 
                tg.age_group,
                COALESCE(ag.count, 0) AS count,
                ROUND(COALESCE(ag.count, 0) * 100.0 / NULLIF(tc.total, 0), 2) AS percentage
            FROM template_groups tg
            LEFT JOIN age_groups ag ON tg.age_group = ag.age_group
            CROSS JOIN total_count tc
            ORDER BY array_position(ARRAY[
                'до 30 лет', 
                'от 31 до 40 лет', 
                'от 41 до 50 лет', 
                'от 51 до 60 лет', 
                'старше 60 лет', 
                'Не указана дата рождения'
            ], tg.age_group);
            """
        stmt = text(sql)
        stmt = stmt.bindparams(faculty=faculty)
        query = conn.execute(stmt)
        data = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]
        # Преобразуем Decimal в float для JSON
        for record in data:
            for k, v in record.items():
                if isinstance(v, Decimal):
                    record[k] = float(v)
        return data

    @staticmethod
    def get_adjuncture_avg_info(lname: str):
        conn = current_app.ms.db(lname).connect()

        sql = """
            SELECT (
                (
                    SELECT COUNT(*)
                        FROM preparation_structure ps /*ОПОП*/
                    JOIN edu_direction ed ON ed.program_level = ps.ps_id AND ed.dir_state AND COALESCE(ed."EduForm", 'очная') IN ('очная', 'очно-заочная')/*ФГОС*/
                    JOIN militaryprofession mp ON mp.edu_direct_id = ed.edu_direct_id /*ВС*/
                    JOIN groupname g ON g.f_militaryprofession = mp.mpid AND COALESCE(yeargraduation, 0) = 0 AND g.cid < 0 /*Группа*/
                    JOIN groupuser gu ON gu.gid = g.gid /*Состав группы*/
                    WHERE ps.listener_category = 3 /*3 - адъюнктура, 4 - докторантура*/
                    AND EXISTS(SELECT 1 FROM receiveplan rp WHERE (rp.militaryprofession = mp.mpid OR rp.militaryprofession IS NULL)
                                AND rp.edudirection = ed.edu_direct_id AND rp.yearid = g.syid)
                ) * 1.0 / (
                    SELECT SUM(amount) 
                        FROM receiveplan rp
                    JOIN edu_direction ed ON rp.edudirection = ed.edu_direct_id AND COALESCE(ed."EduForm", 'очная') IN ('очная', 'очно-заочная')
                    JOIN militaryprofession mp ON mp.edu_direct_id = ed.edu_direct_id AND (rp.militaryprofession = mp.mpid OR rp.militaryprofession IS NULL)
                    JOIN preparation_structure ps ON ed.program_level = ps.ps_id AND ed.dir_state
                    JOIN groupname g ON g.f_militaryprofession = mp.mpid AND COALESCE(yeargraduation, 0) = 0 AND g.cid < 0 AND rp.yearid = g.syid
                    WHERE ps.listener_category = 3
                )
            ) * 100 as avg_adjuncture
        """
        stmt = text(sql)
        query = conn.execute(stmt)
        adjuncture_perc = [dict(zip(tuple(query.keys()), i)) for i in query.cursor][0]
        if adjuncture_perc:
            for k, v in adjuncture_perc.items():
                if isinstance(v, Decimal):
                    adjuncture_perc[k] = float(v)
        avg_adjuncture = adjuncture_perc.get("avg_adjuncture")
        if avg_adjuncture is not None:
            return round(avg_adjuncture, 2)
        return None

    @staticmethod
    def get_doctors_avg_info(lname: str):
        conn = current_app.ms.db(lname).connect()

        sql = """
            SELECT (
                (
                    SELECT COUNT(*)
                        FROM preparation_structure ps /*ОПОП*/
                    JOIN edu_direction ed ON ed.program_level = ps.ps_id AND ed.dir_state/*ФГОС*/
                    JOIN militaryprofession mp ON mp.edu_direct_id = ed.edu_direct_id /*ВС*/
                    JOIN groupname g ON g.f_militaryprofession = mp.mpid AND COALESCE(yeargraduation, 0) = 0 AND g.cid < 0 /*Группа*/
                    JOIN groupuser gu ON gu.gid = g.gid /*Состав группы*/
                    WHERE ps.listener_category = 4 /*3 - адъюнктура, 4 - докторантура*/
                    AND EXISTS(SELECT 1 FROM receiveplan rp WHERE (rp.militaryprofession = mp.mpid OR rp.militaryprofession IS NULL)
                                AND rp.edudirection = ed.edu_direct_id AND rp.yearid = g.syid)
                ) * 1.0 / (
                    SELECT SUM(amount) 
                        FROM receiveplan rp
                    JOIN edu_direction ed ON rp.edudirection = ed.edu_direct_id
                    JOIN militaryprofession mp ON mp.edu_direct_id = ed.edu_direct_id AND (rp.militaryprofession = mp.mpid OR rp.militaryprofession IS NULL)
                    JOIN preparation_structure ps ON ed.program_level = ps.ps_id AND ed.dir_state
                    JOIN groupname g ON g.f_militaryprofession = mp.mpid AND COALESCE(yeargraduation, 0) = 0 AND g.cid < 0 AND rp.yearid = g.syid
                    WHERE ps.listener_category = 4 
                )
            ) * 100 as avg_doctors
        """
        stmt = text(sql)
        query = conn.execute(stmt)
        doctors_perc = [dict(zip(tuple(query.keys()), i)) for i in query.cursor][0]
        if doctors_perc:
            for k, v in doctors_perc.items():
                if isinstance(v, Decimal):
                    doctors_perc[k] = float(v)

        avg_doctors = doctors_perc.get("avg_doctors")
        if avg_doctors is not None:
            return round(avg_doctors, 2)
        return None

    @staticmethod
    def create_tmp_mpid_gid(conn, year):
        sql = """
            SELECT xp_fill_groupuser_period_all(
                (SELECT begdate FROM school_year WHERE xp_key = :year),
                (SELECT enddate FROM school_year WHERE xp_key = :year)
            );
            CREATE TEMP TABLE tmp_mpid_gid AS
            SELECT DISTINCT 
                mp.mpid,
                mp.name,
                g.gid,
                g.name as gname,
                gp.mid,
                ed.edu_direct_id,
                ed.ranked_all,
                ed.experience_adj,
                o.idokin,
                g.syid
            FROM groupuser_period gp
            JOIN groupname g ON g.gid = gp.gid
            LEFT JOIN group_history gh ON gh.gid = g.gid AND gh.school_year = :year
            LEFT JOIN educationyears ey ON ey.number = gh.year
            JOIN militaryprofession mp ON g.f_militaryprofession = mp.mpid
            JOIN edu_direction ed ON mp.edu_direct_id = ed.edu_direct_id
            JOIN preparation_structure ps ON ed.program_level = ps.ps_id
            JOIN okin o ON o.name = ps.okin_level 
            WHERE (
                      ed.p_mastering_y IS NULL OR 
                      ey.number =
                          CASE 
                              WHEN ed.p_mastering_m IS NULL OR ed.p_mastering_m = 0 
                              THEN ed.p_mastering_y
                              ELSE ed.p_mastering_y + 1
                          END
                  );
            """
        stmt = text(sql)
        conn.execute(stmt, {"year": year})

    @staticmethod
    def get_ege_stats(lname: str, year):
        conn = current_app.ms.db(lname).connect()
        Dashboard.create_tmp_mpid_gid(conn, year)
        sql = """
            WITH ege AS (
                SELECT 
                    mpid,
                    name,
                    ROUND(
                        SUM(
                            COALESCE(mark_russ,0) + COALESCE(mark_math,0) + COALESCE(mark_lang,0)
                          + COALESCE(mark_info,0) + COALESCE(mark_hist,0) + COALESCE(mark_phys,0)
                          + COALESCE(mark_geog,0)
                        )::numeric / NULLIF(COUNT(DISTINCT student_id) *
                            ( (CASE WHEN MAX(russ_pass) > 10 THEN 1 ELSE 0 END)
                            + (CASE WHEN MAX(math_pass) > 10 THEN 1 ELSE 0 END)
                            + (CASE WHEN MAX(lang_pass) > 10 THEN 1 ELSE 0 END)
                            + (CASE WHEN MAX(info_pass) > 10 THEN 1 ELSE 0 END)
                            + (CASE WHEN MAX(hist_pass) > 10 THEN 1 ELSE 0 END)
                            + (CASE WHEN MAX(phys_pass) > 10 THEN 1 ELSE 0 END)
                            + (CASE WHEN MAX(geog_pass) > 10 THEN 1 ELSE 0 END)
                            ), 0
                        ),0
                    ) AS avg_ege
                FROM (
                    SELECT DISTINCT 
                        tmg.mpid,
                        tmg.name,
                        rpg.russ_pass, rpg.math_pass, rpg.lang_pass, rpg.info_pass,
                        rpg.hist_pass, rpg.phys_pass, rpg.geog_pass,
                        a.id AS student_id,
                        CASE WHEN rpg.russ_pass > 10 THEN COALESCE(a.child_mark_russ, 100) END AS mark_russ,
                        CASE WHEN rpg.math_pass > 10 THEN COALESCE(a.child_mark_math, 100) END AS mark_math,
                        CASE WHEN rpg.lang_pass > 10 THEN COALESCE(a.child_mark_lang, 100) END AS mark_lang,
                        CASE WHEN rpg.info_pass > 10 THEN COALESCE(a.child_mark_info, 100) END AS mark_info,
                        CASE WHEN rpg.hist_pass > 10 THEN COALESCE(a.child_mark_hist, 100) END AS mark_hist,
                        CASE WHEN rpg.phys_pass > 10 THEN COALESCE(a.child_mark_phys, 100) END AS mark_phys,
                        CASE WHEN rpg.geog_pass > 10 THEN COALESCE(a.child_mark_geog, 100) END AS mark_geog
                    FROM tmp_mpid_gid tmg
                    LEFT JOIN people p ON p.mid = tmg.mid
                    LEFT JOIN xp_applicant a ON a.id = p.mid
                    LEFT JOIN receiveplan rp ON rp.militaryprofession = tmg.mpid 
                                       AND rp.edudirection = tmg.edu_direct_id 
                                       AND rp.yearid = tmg.syid
                    LEFT JOIN receive_pass_grade rpg ON rpg.planid = rp.planid
                ) t
                GROUP BY mpid, name
            ),
            dvi AS (
                SELECT 
                    r.mpid,
                    r.name,
                    ROUND(
                        SUM(
                            COALESCE(r.mark_dvi1,0) + COALESCE(r.mark_dvi2,0) + COALESCE(r.mark_dvi3,0) + COALESCE(r.mark_dvi4,0)
                        )::numeric / NULLIF(COUNT(DISTINCT r.student_id) * s.n_subjects, 0),
                    0) AS avg_dvi,
                    s.n_subjects
                FROM (
                    SELECT DISTINCT 
                        tmg.mpid,
                        tmg.name,
                        a.id AS student_id,
                        CASE WHEN rpg.dvi1 = true AND rpg.pass1 > 5 THEN
                            CASE 
                                WHEN rpg.pass_course1 = a.f_courses_child_mark_01 THEN COALESCE(a.child_mark_disc_01, 100)
                                WHEN rpg.pass_course1 = a.f_courses_child_mark_02 THEN COALESCE(a.child_mark_disc_02, 100)
                                WHEN rpg.pass_course1 = a.f_courses_child_mark_03 THEN COALESCE(a.child_mark_disc_03, 100)
                                WHEN rpg.pass_course1 = a.f_courses_child_mark_04 THEN COALESCE(a.child_mark_disc_04, 100)
                                ELSE 100 END
                        END AS mark_dvi1,
                        CASE WHEN rpg.dvi2 = true AND rpg.pass2 > 5 THEN
                            CASE 
                                WHEN rpg.pass_course2 = a.f_courses_child_mark_01 THEN COALESCE(a.child_mark_disc_01, 100)
                                WHEN rpg.pass_course2 = a.f_courses_child_mark_02 THEN COALESCE(a.child_mark_disc_02, 100)
                                WHEN rpg.pass_course2 = a.f_courses_child_mark_03 THEN COALESCE(a.child_mark_disc_03, 100)
                                WHEN rpg.pass_course2 = a.f_courses_child_mark_04 THEN COALESCE(a.child_mark_disc_04, 100)
                                ELSE 100 END
                        END AS mark_dvi2,
                        CASE WHEN rpg.dvi3 = true AND rpg.pass3 > 5 THEN
                            CASE 
                                WHEN rpg.pass_course3 = a.f_courses_child_mark_01 THEN COALESCE(a.child_mark_disc_01, 100)
                                WHEN rpg.pass_course3 = a.f_courses_child_mark_02 THEN COALESCE(a.child_mark_disc_02, 100)
                                WHEN rpg.pass_course3 = a.f_courses_child_mark_03 THEN COALESCE(a.child_mark_disc_03, 100)
                                WHEN rpg.pass_course3 = a.f_courses_child_mark_04 THEN COALESCE(a.child_mark_disc_04, 100)
                                ELSE 100 END
                        END AS mark_dvi3,
                        CASE WHEN rpg.dvi4 = true AND rpg.pass4 > 5 THEN
                            CASE 
                                WHEN rpg.pass_course4 = a.f_courses_child_mark_01 THEN COALESCE(a.child_mark_disc_01, 100)
                                WHEN rpg.pass_course4 = a.f_courses_child_mark_02 THEN COALESCE(a.child_mark_disc_02, 100)
                                WHEN rpg.pass_course4 = a.f_courses_child_mark_03 THEN COALESCE(a.child_mark_disc_03, 100)
                                WHEN rpg.pass_course4 = a.f_courses_child_mark_04 THEN COALESCE(a.child_mark_disc_04, 100)
                                ELSE 100 END
                        END AS mark_dvi4
                    FROM tmp_mpid_gid tmg
                    LEFT JOIN people p ON p.mid = tmg.mid
                    LEFT JOIN xp_applicant a ON a.id = p.mid
                    LEFT JOIN receiveplan rp ON rp.militaryprofession = tmg.mpid 
                                       AND rp.edudirection = tmg.edu_direct_id 
                                       AND rp.yearid = tmg.syid
                    LEFT JOIN receive_pass_grade rpg ON rpg.planid = rp.planid
                ) r
                JOIN (
                    SELECT 
                        mp.mpid,
                        mp.name,
                        ( CASE WHEN BOOL_OR(dvi1 AND pass1 > 5) THEN 1 ELSE 0 END
                        + CASE WHEN BOOL_OR(dvi2 AND pass2 > 5) THEN 1 ELSE 0 END
                        + CASE WHEN BOOL_OR(dvi3 AND pass3 > 5) THEN 1 ELSE 0 END
                        + CASE WHEN BOOL_OR(dvi4 AND pass4 > 5) THEN 1 ELSE 0 END
                        ) AS n_subjects
                    FROM receive_pass_grade rpg
                    JOIN receiveplan rp ON rpg.planid = rp.planid
                    JOIN militaryprofession mp ON mp.mpid = rp.militaryprofession
                    JOIN edu_direction ed ON mp.edu_direct_id = ed.edu_direct_id
                    JOIN preparation_structure ps ON ed.program_level = ps.ps_id
                    JOIN okin o ON o.name = ps.okin_level
					JOIN tmp_mpid_gid tmg ON tmg.mpid = mp.mpid
                    WHERE rp.yearid = tmg.syid
                    GROUP BY mp.mpid, mp.name
                ) s ON s.mpid = r.mpid
                GROUP BY r.mpid, r.name, s.n_subjects
            )
            SELECT DISTINCT
                e.mpid,
                e.name,
                COALESCE(e.avg_ege, 0)::integer AS avg_ege,
                COALESCE(d.avg_dvi, 0)::integer AS avg_dvi,
                CASE 
                    WHEN tmg.idokin NOT IN (1676, 1677) THEN -1
                    ELSE (
                        CASE 
                           WHEN (
                                CASE 
                                   WHEN d.n_subjects > 0 AND d.avg_dvi IS NOT NULL 
                                        THEN ROUND((COALESCE(e.avg_ege,0) + COALESCE(d.avg_dvi,0)) / 2, 0)::integer
                                   ELSE COALESCE(e.avg_ege,0)::integer
                                END
                           ) >= 65 THEN 10
                           WHEN (
                                CASE 
                                   WHEN d.n_subjects > 0 AND d.avg_dvi IS NOT NULL 
                                        THEN ROUND((COALESCE(e.avg_ege,0) + COALESCE(d.avg_dvi,0)) / 2, 0)::integer
                                   ELSE COALESCE(e.avg_ege,0)::integer
                                END
                           ) >= 60 THEN 5
                           ELSE 0
                        END
                    )
                END AS ege_result
            FROM ege e
            LEFT JOIN dvi d ON d.mpid = e.mpid
            LEFT JOIN tmp_mpid_gid tmg ON tmg.mpid = e.mpid;
            """
        stmt = text(sql)
        query = conn.execute(stmt, {"year": year})
        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor], conn

    @staticmethod
    def get_ege_stats_people_by_mpid(lname: str, mpid: int, year):
        conn = current_app.ms.db(lname).connect()
        Dashboard.create_tmp_mpid_gid(conn, year)
        sql = """            
            WITH ege AS (
                SELECT DISTINCT 
                    a.id AS student_id,
                    tmg.gid,
                    ROUND(
                        (
                            COALESCE(NULLIF(CASE WHEN rpg.russ_pass > 10 THEN COALESCE(a.child_mark_russ, 100) END, NULL), 0) +
                            COALESCE(NULLIF(CASE WHEN rpg.math_pass > 10 THEN COALESCE(a.child_mark_math, 100) END, NULL), 0) +
                            COALESCE(NULLIF(CASE WHEN rpg.lang_pass > 10 THEN COALESCE(a.child_mark_lang, 100) END, NULL), 0) +
                            COALESCE(NULLIF(CASE WHEN rpg.info_pass > 10 THEN COALESCE(a.child_mark_info, 100) END, NULL), 0) +
                            COALESCE(NULLIF(CASE WHEN rpg.hist_pass > 10 THEN COALESCE(a.child_mark_hist, 100) END, NULL), 0) +
                            COALESCE(NULLIF(CASE WHEN rpg.phys_pass > 10 THEN COALESCE(a.child_mark_phys, 100) END, NULL), 0) +
                            COALESCE(NULLIF(CASE WHEN rpg.geog_pass > 10 THEN COALESCE(a.child_mark_geog, 100) END, NULL), 0)
                        )::numeric
                        /
                        NULLIF( 
                            (CASE WHEN rpg.russ_pass > 10 THEN 1 ELSE 0 END) +
                            (CASE WHEN rpg.math_pass > 10 THEN 1 ELSE 0 END) +
                            (CASE WHEN rpg.lang_pass > 10 THEN 1 ELSE 0 END) +
                            (CASE WHEN rpg.info_pass > 10 THEN 1 ELSE 0 END) +
                            (CASE WHEN rpg.hist_pass > 10 THEN 1 ELSE 0 END) +
                            (CASE WHEN rpg.phys_pass > 10 THEN 1 ELSE 0 END) +
                            (CASE WHEN rpg.geog_pass > 10 THEN 1 ELSE 0 END), 0)
                    )::integer AS avg_mark_ege
                FROM tmp_mpid_gid tmg
                LEFT JOIN people p ON p.mid = tmg.mid
                LEFT JOIN xp_applicant a ON a.id = p.mid
                LEFT JOIN receiveplan rp ON rp.militaryprofession = tmg.mpid 
                                   AND rp.edudirection = tmg.edu_direct_id 
                                   AND rp.yearid = tmg.syid
                JOIN receive_pass_grade rpg ON rpg.planid = rp.planid
                WHERE tmg.mpid = :mpid AND tmg.idokin IN (1676, 1677)
            ), 
            dvi AS (
                SELECT DISTINCT 
                    a.id AS student_id,
                    ROUND(
                        (
                            COALESCE(NULLIF(
                                CASE WHEN rpg.dvi1 = true AND rpg.pass1 > 5 THEN
                                    CASE 
                                        WHEN rpg.pass_course1 = a.f_courses_child_mark_01 THEN COALESCE(a.child_mark_disc_01, 100)
                                        WHEN rpg.pass_course1 = a.f_courses_child_mark_02 THEN COALESCE(a.child_mark_disc_02, 100)
                                        WHEN rpg.pass_course1 = a.f_courses_child_mark_03 THEN COALESCE(a.child_mark_disc_03, 100)
                                        WHEN rpg.pass_course1 = a.f_courses_child_mark_04 THEN COALESCE(a.child_mark_disc_04, 100)
                                        ELSE 100 END
                                END, NULL), 0) +
                            COALESCE(NULLIF(
                                CASE WHEN rpg.dvi2 = true AND rpg.pass2 > 5 THEN
                                    CASE 
                                        WHEN rpg.pass_course2 = a.f_courses_child_mark_01 THEN COALESCE(a.child_mark_disc_01, 100)
                                        WHEN rpg.pass_course2 = a.f_courses_child_mark_02 THEN COALESCE(a.child_mark_disc_02, 100)
                                        WHEN rpg.pass_course2 = a.f_courses_child_mark_03 THEN COALESCE(a.child_mark_disc_03, 100)
                                        WHEN rpg.pass_course2 = a.f_courses_child_mark_04 THEN COALESCE(a.child_mark_disc_04, 100)
                                        ELSE 100 END
                                END, NULL), 0) +
                            COALESCE(NULLIF(
                                CASE WHEN rpg.dvi3 = true AND rpg.pass3 > 5 THEN
                                    CASE 
                                        WHEN rpg.pass_course3 = a.f_courses_child_mark_01 THEN COALESCE(a.child_mark_disc_01, 100)
                                        WHEN rpg.pass_course3 = a.f_courses_child_mark_02 THEN COALESCE(a.child_mark_disc_02, 100)
                                        WHEN rpg.pass_course3 = a.f_courses_child_mark_03 THEN COALESCE(a.child_mark_disc_03, 100)
                                        WHEN rpg.pass_course3 = a.f_courses_child_mark_04 THEN COALESCE(a.child_mark_disc_04, 100)
                                        ELSE 100 END
                                END, NULL), 0) +
                            COALESCE(NULLIF(
                                CASE WHEN rpg.dvi4 = true AND rpg.pass4 > 5 THEN
                                    CASE 
                                        WHEN rpg.pass_course4 = a.f_courses_child_mark_01 THEN COALESCE(a.child_mark_disc_01, 100)
                                        WHEN rpg.pass_course4 = a.f_courses_child_mark_02 THEN COALESCE(a.child_mark_disc_02, 100)
                                        WHEN rpg.pass_course4 = a.f_courses_child_mark_03 THEN COALESCE(a.child_mark_disc_03, 100)
                                        WHEN rpg.pass_course4 = a.f_courses_child_mark_04 THEN COALESCE(a.child_mark_disc_04, 100)
                                        ELSE 100 END
                                END, NULL), 0)
                        )::numeric
                        /
                        NULLIF(
                            (CASE WHEN rpg.dvi1 = true AND rpg.pass1 > 5 THEN 1 ELSE 0 END) +
                            (CASE WHEN rpg.dvi2 = true AND rpg.pass2 > 5 THEN 1 ELSE 0 END) +
                            (CASE WHEN rpg.dvi3 = true AND rpg.pass3 > 5 THEN 1 ELSE 0 END) +
                            (CASE WHEN rpg.dvi4 = true AND rpg.pass4 > 5 THEN 1 ELSE 0 END), 0)
                    )::integer AS avg_mark_dvi
                FROM tmp_mpid_gid tmg
                LEFT JOIN people p ON p.mid = tmg.mid
                LEFT JOIN xp_applicant a ON a.id = p.mid
                LEFT JOIN receiveplan rp ON rp.militaryprofession = tmg.mpid 
                                   AND rp.edudirection = tmg.edu_direct_id 
                                   AND rp.yearid = tmg.syid
                LEFT JOIN receive_pass_grade rpg ON rpg.planid = rp.planid
                WHERE tmg.mpid = :mpid AND tmg.idokin IN (1676, 1677)
            )
            SELECT 
                tmg.mid,
                tmg.gid,
				xp_format_fio(p.lastname,p.firstname,p.patronymic,0) as fio,
				gn.name as group_name,
				f.idfaculty,
				f.faculty,
                ege.avg_mark_ege,
                dvi.avg_mark_dvi,
                ROUND(((COALESCE(ege.avg_mark_ege,0) + COALESCE(dvi.avg_mark_dvi,0))::numeric 
                       / NULLIF((CASE WHEN ege.avg_mark_ege IS NOT NULL THEN 1 ELSE 0 END) + 
                                (CASE WHEN dvi.avg_mark_dvi IS NOT NULL THEN 1 ELSE 0 END),0)
                ))::integer AS avg_total
            FROM tmp_mpid_gid tmg
            LEFT JOIN ege ON ege.student_id = tmg.mid
            LEFT JOIN dvi ON tmg.mid = dvi.student_id
			JOIN people p ON tmg.mid = p.mid
			JOIN groupname gn ON tmg.gid = gn.gid
			LEFT OUTER JOIN cathedras c ON c.idcathedra = gn.idcathedra
        	LEFT OUTER JOIN faculty f ON f.idfaculty = COALESCE(gn.idfaculty, c.faculty)
            WHERE tmg.mpid = :mpid 
            ORDER BY fio;
            """
        stmt = text(sql)
        query = conn.execute(stmt, {'mpid': mpid, "year": year})
        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]


    @staticmethod
    def get_ege_stats_academic_subjects_by_mpid(lname: str, mpid: int, year):
        conn = current_app.ms.db(lname).connect()
        Dashboard.create_tmp_mpid_gid(conn, year)
        sql = """
            WITH all_marks AS (
            SELECT 
                a.id AS student_id,
        
                -- ЕГЭ
                CASE WHEN rpg.russ_pass > 10 THEN COALESCE(a.child_mark_russ, 100) END AS mark_russ,
                CASE WHEN rpg.math_pass > 10 THEN COALESCE(a.child_mark_math, 100) END AS mark_math,
                CASE WHEN rpg.lang_pass > 10 THEN COALESCE(a.child_mark_lang, 100) END AS mark_lang,
                CASE WHEN rpg.info_pass > 10 THEN COALESCE(a.child_mark_info, 100) END AS mark_info,
                CASE WHEN rpg.hist_pass > 10 THEN COALESCE(a.child_mark_hist, 100) END AS mark_hist,
                CASE WHEN rpg.phys_pass > 10 THEN COALESCE(a.child_mark_phys, 100) END AS mark_phys,
                CASE WHEN rpg.geog_pass > 10 THEN COALESCE(a.child_mark_geog, 100) END AS mark_geog,
        
                -- ДВИ
                rpg.pass_course1,
                CASE WHEN rpg.dvi1 = true AND rpg.pass1 > 5 THEN
                    CASE 
                        WHEN rpg.pass_course1 = a.f_courses_child_mark_01 THEN COALESCE(a.child_mark_disc_01, 100)
                        WHEN rpg.pass_course1 = a.f_courses_child_mark_02 THEN COALESCE(a.child_mark_disc_02, 100)
                        WHEN rpg.pass_course1 = a.f_courses_child_mark_03 THEN COALESCE(a.child_mark_disc_03, 100)
                        WHEN rpg.pass_course1 = a.f_courses_child_mark_04 THEN COALESCE(a.child_mark_disc_04, 100)
                        ELSE 100 END
                END AS mark_dvi1,
        
                rpg.pass_course2,
                CASE WHEN rpg.dvi2 = true AND rpg.pass2 > 5 THEN
                    CASE 
                        WHEN rpg.pass_course2 = a.f_courses_child_mark_01 THEN COALESCE(a.child_mark_disc_01, 100)
                        WHEN rpg.pass_course2 = a.f_courses_child_mark_02 THEN COALESCE(a.child_mark_disc_02, 100)
                        WHEN rpg.pass_course2 = a.f_courses_child_mark_03 THEN COALESCE(a.child_mark_disc_03, 100)
                        WHEN rpg.pass_course2 = a.f_courses_child_mark_04 THEN COALESCE(a.child_mark_disc_04, 100)
                        ELSE 100 END
                END AS mark_dvi2,
        
                rpg.pass_course3,
                CASE WHEN rpg.dvi3 = true AND rpg.pass3 > 5 THEN
                    CASE 
                        WHEN rpg.pass_course3 = a.f_courses_child_mark_01 THEN COALESCE(a.child_mark_disc_01, 100)
                        WHEN rpg.pass_course3 = a.f_courses_child_mark_02 THEN COALESCE(a.child_mark_disc_02, 100)
                        WHEN rpg.pass_course3 = a.f_courses_child_mark_03 THEN COALESCE(a.child_mark_disc_03, 100)
                        WHEN rpg.pass_course3 = a.f_courses_child_mark_04 THEN COALESCE(a.child_mark_disc_04, 100)
                        ELSE 100 END
                END AS mark_dvi3,
        
                rpg.pass_course4,
                CASE WHEN rpg.dvi4 = true AND rpg.pass4 > 5 THEN
                    CASE 
                        WHEN rpg.pass_course4 = a.f_courses_child_mark_01 THEN COALESCE(a.child_mark_disc_01, 100)
                        WHEN rpg.pass_course4 = a.f_courses_child_mark_02 THEN COALESCE(a.child_mark_disc_02, 100)
                        WHEN rpg.pass_course4 = a.f_courses_child_mark_03 THEN COALESCE(a.child_mark_disc_03, 100)
                        WHEN rpg.pass_course4 = a.f_courses_child_mark_04 THEN COALESCE(a.child_mark_disc_04, 100)
                        ELSE 100 END
                END AS mark_dvi4
        
            FROM tmp_mpid_gid tmg
            JOIN people p ON p.mid = tmg.mid
            JOIN xp_applicant a ON a.id = p.mid
            JOIN receiveplan rp ON rp.militaryprofession = tmg.mpid 
                               AND rp.edudirection = tmg.edu_direct_id 
                               AND rp.yearid = tmg.syid
            JOIN receive_pass_grade rpg ON rpg.planid = rp.planid
            WHERE tmg.mpid = :mpid AND tmg.idokin IN (1676, 1677)
            )
            SELECT
              COALESCE(ROUND(AVG(am.mark_russ))::integer, -1) AS avg_russ,
              COALESCE(ROUND(AVG(am.mark_math))::integer, -1) AS avg_math,
              COALESCE(ROUND(AVG(am.mark_lang ))::integer, -1) AS avg_lang,
              COALESCE(ROUND(AVG(am.mark_info))::integer, -1) AS avg_info,
              COALESCE(ROUND(AVG(am.mark_hist))::integer, -1) AS avg_hist,
              COALESCE(ROUND(AVG(am.mark_phys))::integer, -1) AS avg_phys,
              COALESCE(ROUND(AVG(am.mark_geog))::integer, -1) AS avg_geog,
            
              am.pass_course1, am.pass_course2, am.pass_course3, am.pass_course4,
              mc1.name AS pass_course_name1,
              mc2.name AS pass_course_name2,
              mc3.name AS pass_course_name3,
              mc4.name AS pass_course_name4,
            
              COALESCE(ROUND(AVG(am.mark_dvi1))::integer, -1) AS avg_dvi1,
              COALESCE(ROUND(AVG(am.mark_dvi2))::integer, -1) AS avg_dvi2,
              COALESCE(ROUND(AVG(am.mark_dvi3))::integer, -1) AS avg_dvi3,
              COALESCE(ROUND(AVG(am.mark_dvi4))::integer, -1) AS avg_dvi4
            
            FROM (SELECT 1) v
            LEFT JOIN all_marks am ON TRUE
            LEFT JOIN meta_course mc1 ON mc1.id = am.pass_course1
            LEFT JOIN meta_course mc2 ON mc2.id = am.pass_course2
            LEFT JOIN meta_course mc3 ON mc3.id = am.pass_course3
            LEFT JOIN meta_course mc4 ON mc4.id = am.pass_course4
            GROUP BY
              am.pass_course1, am.pass_course2, am.pass_course3, am.pass_course4,
              mc1.name, mc2.name, mc3.name, mc4.name;
        """
        stmt = text(sql)
        query = conn.execute(stmt, {'mpid': mpid, "year": year})
        result = query.fetchone()
        return dict(zip(query.keys(), result))

    @staticmethod
    def get_teachers_stats(conn, year):
        if type(conn) == str:
            conn = current_app.ms.db(conn).connect()
            Dashboard.create_tmp_mpid_gid(conn, year)
        sql = """            
            WITH all_teachers AS (
                SELECT DISTINCT tmg.mpid, peo.mid, tmg.ranked_all
                FROM tmp_mpid_gid tmg
                LEFT JOIN vw_schedule s ON s.main_gid = tmg.gid AND s.school_year = :year
                LEFT JOIN people peo ON peo.mid = ANY(s.teacher_mid)
            ),
            teachers_with_rank AS (
                SELECT DISTINCT pf.mid
                FROM xp_personal_file pf
                WHERE pf."AcademicTitle" IS NOT NULL AND pf."AcademicTitle" != 0
            ),
            teachers_with_degree AS (
                SELECT DISTINCT pf.mid
                FROM xp_personal_file pf
                WHERE pf."AcademicDegree" IS NOT NULL AND pf."AcademicDegree" != 0
            ),
            ranked_teachers AS (
                SELECT DISTINCT mid FROM teachers_with_rank
                UNION
                SELECT DISTINCT mid FROM teachers_with_degree
            )
            SELECT
                at.mpid,
                ROUND(
                    COUNT(DISTINCT rt.mid)::numeric 
                    / NULLIF(COUNT(DISTINCT at.mid),0) * 100, 
                    2
                )::integer AS percentage,
                at.ranked_all::integer,
                CASE 
                    WHEN ROUND(
                             COUNT(DISTINCT rt.mid)::numeric 
                             / NULLIF(COUNT(DISTINCT at.mid),0) * 100, 
                             2
                         ) >= at.ranked_all 
                    THEN 20 ELSE 0
                END AS result
            FROM all_teachers at
            LEFT JOIN ranked_teachers rt ON at.mid = rt.mid
            GROUP BY at.mpid, at.ranked_all;
        """
        stmt = text(sql)
        query = conn.execute(stmt, {"year": year})
        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    @staticmethod
    def get_stats_staff_by_academic_degree_or_academic_title(lname: str, mpid: int, year: int):
        conn = current_app.ms.db(lname).connect()
        Dashboard.create_tmp_mpid_gid(conn, year)
        sql = """
            WITH all_teachers AS (
                SELECT DISTINCT tmg.mpid, peo.mid
                FROM tmp_mpid_gid tmg
                LEFT JOIN vw_schedule s ON s.main_gid = tmg.gid AND s.school_year = :year
                LEFT JOIN people peo ON peo.mid = ANY(s.teacher_mid)
                WHERE tmg.mpid = :mpid
            ),
            teacher_info AS (
                SELECT 
                    at.mpid,
                    pf.mid,
                    atit.name AS academic_title,
                    adeg."name" AS academic_degree
                FROM all_teachers at
                JOIN xp_personal_file pf ON at.mid = pf.mid
                LEFT JOIN academictitle atit ON pf."AcademicTitle" = atit.atid
                LEFT JOIN academicdegree adeg ON pf."AcademicDegree" = adeg.agid
            )
            SELECT
                mpid,
                academic_title AS category,
                COUNT(DISTINCT mid) AS cnt,
                'title' AS type
            FROM teacher_info
            WHERE academic_title IS NOT NULL
            GROUP BY mpid, academic_title
            
            UNION ALL
            
            SELECT
                mpid,
                academic_degree AS category,
                COUNT(DISTINCT mid) AS cnt,
                'degree' AS type
            FROM teacher_info
            WHERE academic_degree IS NOT NULL
            GROUP BY mpid, academic_degree
            
            ORDER BY type, category;
        """
        stmt = text(sql)
        query = conn.execute(stmt, {"mpid": mpid, "year": year})
        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    @staticmethod
    def get_stats_staff_and_stag(lname: str, mpid: int, year: int):
        conn = current_app.ms.db(lname).connect()
        Dashboard.create_tmp_mpid_gid(conn, year)
        sql = """
            WITH RECURSIVE teacher_stag AS (
                SELECT 
                    ts.mid,
                    ts.datebeginstag::date AS datebegin,
                    COALESCE(ts.dateendstag::date, CURRENT_DATE) AS dateend
                FROM xp_teacher_stag ts
                WHERE isprofile != 0
            ),
            dep AS ( 
                SELECT id, "name", owner_dep, "name"::text AS fullname 
                FROM vw_divisions 
                WHERE owner_dep IS NULL OR owner_dep = 0 
                UNION ALL 
                SELECT vd.id, vd."name", vd.owner_dep, CONCAT(dep."name", ' / ', vd."name") AS fullname 
                FROM dep, vw_divisions vd
                WHERE vd.owner_dep = dep.id
            ),
            ordered AS (
                SELECT 
                    mid, datebegin, dateend,
                    ROW_NUMBER() OVER (PARTITION BY mid ORDER BY datebegin) AS rn
                FROM teacher_stag
            ),
            rec AS (
                -- первая запись по каждому преподавателю
                SELECT mid, datebegin, dateend, rn
                FROM ordered
                WHERE rn = 1
                UNION ALL
                -- рекурсивно объединяем, если даты пересекаются
                SELECT o.mid,
                       LEAST(r.datebegin, o.datebegin),
                       GREATEST(r.dateend, o.dateend),
                       o.rn
                FROM rec r
                JOIN ordered o 
                  ON o.mid = r.mid
                 AND o.rn = r.rn + 1
                 AND o.datebegin <= r.dateend
            ),
            merged AS (
                SELECT mid, MIN(datebegin) AS datebegin, MAX(dateend) AS dateend
                FROM rec
                GROUP BY mid
            ),
            calc AS (
                SELECT mid, SUM(dateend - datebegin) AS days
                FROM merged
                GROUP BY mid
            )
            SELECT 
                p.mid,
                xp_format_fio(p.LastName, p.FirstName, p.Patronymic,0)::varchar(90) fio,
                COALESCE(ROUND(c.days / 365.0, 2)::integer, 0) AS years_stag,
				COALESCE(atit.name, adeg."name", null) AS category,
                dep.name
            FROM tmp_mpid_gid tmg
            JOIN vw_schedule s ON s.main_gid = tmg.gid AND s.school_year = :year
            JOIN people p ON p.mid = ANY(s.teacher_mid)
            LEFT JOIN calc c ON c.mid = p.mid
            LEFT JOIN vw_staff vs ON vs.mid = p.mid
            LEFT JOIN dep ON dep.id = vs.depid
			JOIN xp_personal_file pf ON p.mid = pf.mid
			LEFT JOIN academictitle atit ON pf."AcademicTitle" = atit.atid
			LEFT JOIN academicdegree adeg ON pf."AcademicDegree" = adeg.agid
            WHERE tmg.mpid = :mpid
            GROUP BY p.mid, c.days, dep.name, atit.name, adeg."name"
            ORDER BY fio;
            """
        stmt = text(sql)
        query = conn.execute(stmt, {"mpid": mpid, "year": year})
        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    @staticmethod
    def get_eioc_stats(conn, year, mpid):
        if isinstance(conn, str):
            conn = current_app.ms.db(conn).connect()
            Dashboard.create_tmp_mpid_gid(conn, year)

        where = ""
        args = {"year": year}
        if mpid:
            where += " AND t.mpid = :mpid"
            args["mpid"] = mpid

        sql = """            
            BEGIN;

            -- 2) Короткая таблица с mpid и syid из исходной tmp_mpid_gid (для удобства и защиты от изменений)
            DROP TABLE IF EXISTS tmp_mpid;
            CREATE TEMP TABLE tmp_mpid AS
            SELECT DISTINCT mpid, syid, gid, name
            FROM tmp_mpid_gid;
            
            CREATE INDEX ON tmp_mpid(mpid);
            CREATE INDEX ON tmp_mpid(syid);
            
            -- 3) Связываем qualif_demands -> curriculum только для наших mpid (делаем 1 проход)
            DROP TABLE IF EXISTS tmp_mpid_curr;
            CREATE TEMP TABLE tmp_mpid_curr AS
            SELECT
              t.mpid,
              t.syid,
              cur.idcurriculum,
              cur.q_demand_id
            FROM tmp_mpid t
            JOIN qualif_demands qd ON qd.mpid = t.mpid
            JOIN curriculum cur ON cur.q_demand_id = qd.q_demand_id;
            
            CREATE INDEX ON tmp_mpid_curr(mpid);
            CREATE INDEX ON tmp_mpid_curr(idcurriculum);
            
            -- 4) Получаем все релевантные дисциплины (курсы) для этих mpid и фильтруем по блокам (через все связи) — 1 проход
            DROP TABLE IF EXISTS tmp_relevant_courses;
            CREATE TEMP TABLE tmp_relevant_courses AS
            SELECT
              mc.mpid,
              mc.idcurriculum,
              qdd.cid,
              c.title
            FROM tmp_mpid_curr mc
            JOIN q_demand_disciplines qdd ON qdd.q_demand_id = mc.q_demand_id
            JOIN courses c ON c.cid = qdd.cid
            -- связываем с группами/циклами/блоками, чтобы оставить только блоки 0/1 (ваши 1/2)
            JOIN discipline_groups dg 
              ON dg.dis_group_id = COALESCE(NULLIF(qdd.dis_group_id,0), c.dis_group_id)
            JOIN discipline_cycles dc ON dg.dis_cycle_id = dc.dis_cycle_id
            JOIN discipline_blocks db ON dc.block = db.id AND db.block_type IN (0,1)
            -- учитываем, что curriculum действителен для нужного syid: проверяем в studyyears2curriculum
            JOIN studyyears2curriculum s2c ON s2c.idcurriculum = mc.idcurriculum AND s2c.syid = mc.syid;
            
            CREATE INDEX ON tmp_relevant_courses(mpid);
            CREATE INDEX ON tmp_relevant_courses(idcurriculum);
            CREATE INDEX ON tmp_relevant_courses(cid);
            
            -- 5) Для каждой записи из tmp_relevant_courses смотрим, есть ли для этой пары (idcurriculum, cid) запись в curriculum_detail -> curriculum_course.
            --    Если для пары нет curriculum_course (т.е. РПД не сформирована/не утверждена) — это missing.
            DROP TABLE IF EXISTS tmp_missing;
            CREATE TEMP TABLE tmp_missing AS
            SELECT
              rc.mpid,
              -- jsonb массив объектов {id, name} для дисциплин без РПД
              COALESCE(
                jsonb_agg(DISTINCT jsonb_build_object('id', rc.cid, 'name', rc.title) )
                  FILTER (WHERE cc.id_curriculum_course IS NULL),
                '[]'::jsonb
              ) AS missing_rpds,
              COUNT(*) FILTER (WHERE cc.id_curriculum_course IS NULL) AS cnt_missing
            FROM tmp_relevant_courses rc
            LEFT JOIN curriculum_detail cd ON cd.idcurriculum = rc.idcurriculum AND cd.cid = rc.cid
            LEFT JOIN curriculum_course cc ON cd.id_curr_detail = ANY(COALESCE(cc.id_curr_details, ARRAY[]::int[]) || COALESCE(ARRAY[cc.id_curr_detail], ARRAY[]::int[])) AND status = 2
            GROUP BY rc.mpid;
            
            CREATE INDEX ON tmp_missing(mpid);
            
            COMMIT;
            
            
            -- 6) Собираем итог: для каждого mpid указываем флаг и jsonb массив missing (если нет записи в tmp_mpid_curr -> нет плана)
            SELECT DISTINCT
              t.mpid,
              t.name,
            
              COALESCE(doc.docs, '{}')       AS doc_f_doc_hs,
              COALESCE(inet.docs, '{}')      AS internet_f_doc_hs,
              COALESCE(local.docs, '{}')     AS local_act_f_doc_hs,
            
              (COALESCE(array_length(doc.docs, 1), 0) > 0)    AS has_doc,
              (COALESCE(array_length(inet.docs, 1), 0) > 0)   AS has_internet_access,
              (COALESCE(array_length(local.docs, 1), 0) > 0)  AS has_local_act,
            
              CASE WHEN COUNT(s.sheid) > 0 THEN TRUE ELSE FALSE END AS has_schedule,
            
              CASE WHEN EXISTS (
                  SELECT 1
                  FROM options o
                  WHERE o.name IN ('library_url', 'local_library_url', 'external_library_url')
                    AND o.value IS NOT NULL
                    AND o.value <> ''
              ) THEN TRUE ELSE FALSE END AS has_library_access,
            
              -- has_portfolio_formation: TRUE если у ВСЕХ заполнено Hobby (т.е. missing_hobby пуст)
              CASE
                WHEN COALESCE(jsonb_array_length(mh.missing_hobby), 0) = 0 THEN TRUE
                ELSE FALSE
              END AS has_portfolio_formation,
            
              TRUE AS has_teacher_interactions,
            
              CASE
                WHEN EXISTS (SELECT 1 FROM tmp_mpid_curr mc WHERE mc.mpid = t.mpid) -- есть план
                 AND COALESCE(missing.cnt_missing, 0) = 0 -- и нет missing
                THEN TRUE ELSE FALSE
              END AS has_up_rpd,
            
              COALESCE(missing.missing_rpds, '[]'::jsonb) AS missing_rpds,
            
              -- НОВОЕ: jsonb массив объектов {mid, fio} с теми, у кого Hobby пустое или NULL
              COALESCE(mh.missing_hobby, '[]'::jsonb) AS missing_hobby
            
            FROM tmp_mpid t
            LEFT JOIN tmp_missing missing ON missing.mpid = t.mpid
            LEFT JOIN vw_schedule s 
                   ON s.main_gid = t.gid 
                  AND s.school_year = :year
            
            CROSS JOIN LATERAL (
                SELECT ARRAY_AGG(DISTINCT d.f_doc_hs) AS docs
                FROM f_doc_hs d
                WHERE d.c_type_doc = 10 AND d.stat = 1
                  AND ((d.date_end IS NULL AND CURRENT_DATE > d.date)
                       OR (d.date_end IS NOT NULL AND d.date_end > CURRENT_DATE))
            ) AS doc
            
            CROSS JOIN LATERAL (
                SELECT ARRAY_AGG(DISTINCT d.f_doc_hs) AS docs
                FROM f_doc_hs d
                WHERE d.c_type_doc = 7 AND d.stat = 1
                  AND ((d.date_end IS NULL AND CURRENT_DATE > d.date)
                       OR (d.date_end IS NOT NULL AND d.date_end > CURRENT_DATE))
            ) AS inet
            
            CROSS JOIN LATERAL (
                SELECT ARRAY_AGG(DISTINCT d.f_doc_hs) AS docs
                FROM f_doc_hs d
                WHERE d.c_type_doc = 6 AND d.stat = 1
                  AND ((d.date_end IS NULL AND CURRENT_DATE > d.date)
                       OR (d.date_end IS NOT NULL AND d.date_end > CURRENT_DATE))
            ) AS local
            
            JOIN tmp_mpid_gid tmg ON tmg.mpid = t.mpid
            -- заменил прямой JOIN на xpf на LATERAL, чтобы не умножать строки в основном запросе
            LEFT JOIN LATERAL (
              SELECT
                COALESCE(
                  jsonb_agg( DISTINCT jsonb_build_object('mid', xpf2.mid, 'fio', xp_format_fio(p.lastname,p.firstname,p.patronymic,0)) )
                    FILTER (WHERE xpf2."Hobby" IS NULL OR trim(xpf2."Hobby") = ''),
                  '[]'::jsonb
                ) AS missing_hobby
              FROM tmp_mpid_gid tmg2
              LEFT JOIN xp_personal_file xpf2 ON tmg2.mid = xpf2.mid
              LEFT JOIN people p ON p.mid = xpf2.mid
              WHERE tmg2.mpid = t.mpid
            ) AS mh ON TRUE
            
            WHERE 1=1 {where}
            GROUP BY t.mpid, t.name, doc.docs, inet.docs, local.docs, missing.cnt_missing, missing.missing_rpds, mh.missing_hobby
            ORDER BY t.mpid;
        """

        stmt = text(sql.replace('{where}', where))
        query = conn.execute(stmt, args)
        rows = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

        for row in rows:
            # doc_files
            doc_ids = row.pop("doc_f_doc_hs") or []
            row["doc_files"] = [
                {"f_doc_hs": did, "files": File(conn.engine.url.database).get_vsoko_local_act_files(did)}
                for did in doc_ids
            ]
            row["has_doc"] = any(bool(item["files"]) for item in row["doc_files"])
            row["has_library_access"] = any(bool(item["files"]) for item in row["doc_files"])

            # internet_files
            inet_ids = row.pop("internet_f_doc_hs") or []
            row["internet_files"] = [
                {"f_doc_hs": did, "files": File(conn.engine.url.database).get_vsoko_local_act_files(did)}
                for did in inet_ids
            ]
            row["has_internet_access"] = any(bool(item["files"]) for item in row["internet_files"])

            # local_act_files
            local_ids = row.pop("local_act_f_doc_hs") or []
            row["local_act_files"] = [
                {"f_doc_hs": did, "files": File(conn.engine.url.database).get_vsoko_local_act_files(did)}
                for did in local_ids
            ]
            row["has_local_act"] = any(bool(item["files"]) for item in row["local_act_files"])

            # Итоговая сумма флагов и результат
            sum_true = (
                    int(bool(row.get("has_doc"))) +
                    int(bool(row.get("has_internet_access"))) +
                    int(bool(row.get("has_local_act"))) +
                    int(bool(row.get("has_schedule"))) +
                    int(bool(row.get("has_library_access"))) +
                    int(bool(row.get("has_portfolio_formation"))) +
                    int(bool(row.get("has_up_rpd"))) +
                    int(bool(row.get("has_teacher_interactions")))
            )
            row["result"] = 10 if sum_true >= 4 else 0

        return rows

    @staticmethod
    def get_vsoko_stats(conn, year, mpid):
        if isinstance(conn, str):
            conn = current_app.ms.db(conn).connect()
            Dashboard.create_tmp_mpid_gid(conn, year)

        where = ""
        args = {"year": year}
        if mpid:
            where += " AND tmg.mpid = :mpid"
            args["mpid"] = mpid

        sql = """            
            SELECT 
                tmg.mpid,
                tmg.name,

                -- список документов для локального акта (c_type_doc = 8)
                COALESCE(local.docs, '{{}}') AS local_act_f_doc_hs,

                -- список документов для отчёта о самообследовании (c_type_doc = 9)
                COALESCE(selfrep.docs, '{{}}') AS self_exam_f_doc_hs,

                -- флаги по наличию документов
                (COALESCE(array_length(local.docs, 1), 0) > 0)    AS has_local_act,
                (COALESCE(array_length(selfrep.docs, 1), 0) > 0)  AS has_self_examination_report

            FROM tmp_mpid_gid tmg
            LEFT JOIN vw_schedule s 
                   ON s.main_gid = tmg.gid 
                  AND s.school_year = :year

            CROSS JOIN LATERAL (
                SELECT ARRAY_AGG(DISTINCT d.f_doc_hs) AS docs
                FROM f_doc_hs d
                WHERE d.c_type_doc = 8 AND d.stat = 1
                  AND (
                        (d.date_end IS NULL AND CURRENT_DATE > d.date)
                     OR (d.date_end IS NOT NULL AND d.date_end > CURRENT_DATE)
                  )
            ) AS local

            CROSS JOIN LATERAL (
                SELECT ARRAY_AGG(DISTINCT d.f_doc_hs) AS docs
                FROM f_doc_hs d
                WHERE d.c_type_doc = 9 AND d.stat = 1
                  AND (
                        (d.date_end IS NULL AND CURRENT_DATE > d.date)
                     OR (d.date_end IS NOT NULL AND d.date_end > CURRENT_DATE)
                  )
            ) AS selfrep

            WHERE 1=1 {where} 
            GROUP BY tmg.mpid, tmg.name, local.docs, selfrep.docs
        """.format(where=where)

        stmt = text(sql)
        query = conn.execute(stmt, args)
        rows = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

        # добавляем списки имён файлов по каждому f_doc_hs
        for row in rows:
            local_ids = row.pop("local_act_f_doc_hs") or []
            self_ids = row.pop("self_exam_f_doc_hs") or []

            row["local_act_files"] = [
                {"f_doc_hs": did, "files": File(conn.engine.url.database).get_vsoko_local_act_files(did)} for did in local_ids
            ]
            row["has_local_act"] = any(bool(item["files"]) for item in row["local_act_files"])

            row["self_exam_files"] = [
                {"f_doc_hs": did, "files": File(conn.engine.url.database).get_vsoko_local_act_files(did)} for did in self_ids
            ]
            row["has_self_examination_report"] = any(bool(item["files"]) for item in row["self_exam_files"])

            # Итоговая сумма флагов и результат
            sum_true = (
                    int(bool(row.get("has_local_act"))) +
                    int(bool(row.get("has_self_examination_report")))
            )
            row["result"] = 10 if sum_true == 2 else 0

        return rows

    @staticmethod
    def get_workers_stats(conn, year):
        if type(conn) == str:
            conn = current_app.ms.db(conn).connect()
            Dashboard.create_tmp_mpid_gid(conn, year)
        sql = """
            WITH RECURSIVE teacher_stag AS (
                SELECT 
                    ts.mid,
                    ts.datebeginstag::date AS datebegin,
                    COALESCE(ts.dateendstag::date, CURRENT_DATE) AS dateend
                FROM xp_teacher_stag ts
                WHERE isprofile != 0
            ),
            ordered AS (
                SELECT 
                    mid, datebegin, dateend,
                    ROW_NUMBER() OVER (PARTITION BY mid ORDER BY datebegin) AS rn
                FROM teacher_stag
            ),
            rec AS (
                SELECT mid, datebegin, dateend, rn
                FROM ordered
                WHERE rn = 1
                UNION ALL
                SELECT o.mid,
                       LEAST(r.datebegin, o.datebegin),
                       GREATEST(r.dateend, o.dateend),
                       o.rn
                FROM rec r
                JOIN ordered o 
                  ON o.mid = r.mid
                 AND o.rn = r.rn + 1
                 AND o.datebegin <= r.dateend
            ),
            merged AS (
                SELECT mid, MIN(datebegin) AS datebegin, MAX(dateend) AS dateend
                FROM rec
                GROUP BY mid
            ),
            calc AS (
                SELECT mid, SUM(dateend - datebegin) AS days
                FROM merged
                GROUP BY mid
            ),
            teachers AS (
                SELECT 
                    p.mid,
                    tmg.mpid,
                    COALESCE(ROUND(c.days / 365.0, 2), 0) AS years_stag,
                    tmg.experience_adj
                FROM tmp_mpid_gid tmg
                LEFT JOIN vw_schedule s ON s.main_gid = tmg.gid AND s.school_year = :year
                LEFT JOIN people p ON p.mid = ANY(s.teacher_mid)
                LEFT JOIN calc c ON c.mid = p.mid
                WHERE p.mid IS NOT NULL
                GROUP BY p.mid, c.days, tmg.experience_adj, tmg.mpid
            ),
            agg AS (
                SELECT 
                    COUNT(*) AS total_teachers,                      -- b
                    COUNT(*) FILTER (WHERE years_stag > 0) AS prof_teachers, -- a
                    COALESCE(MAX(experience_adj), 0) AS norm,
                    mpid
                FROM teachers
                GROUP BY mpid
            )
            SELECT 
                mpid,
                prof_teachers, 
                total_teachers,
                ROUND(prof_teachers::decimal / NULLIF(total_teachers,0) * 100, 2)::integer AS percent_value,
                norm::integer,
                CASE 
                    WHEN ROUND(prof_teachers::decimal / NULLIF(total_teachers,0) * 100, 2) >= GREATEST(COALESCE(norm, 0)::numeric, 1)  
                    THEN 20 ELSE 0 
                END AS result
            FROM agg;
            """
        stmt = text(sql)
        query = conn.execute(stmt, {"year": year})
        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    @staticmethod
    def get_college_workers_stats(conn, year, mpid):
        if type(conn) == str:
            conn = current_app.ms.db(conn).connect()
            Dashboard.create_tmp_mpid_gid(conn, year)
        sql = """
                WITH RECURSIVE teacher_stag AS (
                    SELECT 
                        ts.mid,
                        ts.datebeginstag::date AS datebegin,
                        COALESCE(ts.dateendstag::date, CURRENT_DATE) AS dateend
                    FROM xp_teacher_stag ts
                    WHERE isprofile != 0
                ),
                ordered AS (
                    SELECT 
                        mid, datebegin, dateend,
                        ROW_NUMBER() OVER (PARTITION BY mid ORDER BY datebegin) AS rn
                    FROM teacher_stag
                ),
                rec AS (
                    SELECT mid, datebegin, dateend, rn
                    FROM ordered
                    WHERE rn = 1
                    UNION ALL
                    SELECT o.mid,
                           LEAST(r.datebegin, o.datebegin),
                           GREATEST(r.dateend, o.dateend),
                           o.rn
                    FROM rec r
                    JOIN ordered o 
                      ON o.mid = r.mid
                     AND o.rn = r.rn + 1
                     AND o.datebegin <= r.dateend
                ),
                merged AS (
                    SELECT mid, MIN(datebegin) AS datebegin, MAX(dateend) AS dateend
                    FROM rec
                    GROUP BY mid
                ),
                calc AS (
                    SELECT mid, SUM(dateend - datebegin) AS days
                    FROM merged
                    GROUP BY mid
                ),
                teachers AS (
                    SELECT 
                        p.mid,
                        tmg.mpid,
                        COALESCE(ROUND(c.days / 365.0, 2), 0) AS years_stag,
                        tmg.experience_adj
                    FROM tmp_mpid_gid tmg
                    LEFT JOIN vw_schedule s ON s.main_gid = tmg.gid AND s.school_year = :year
                    LEFT JOIN people p ON p.mid = ANY(s.teacher_mid)
                    LEFT JOIN calc c ON c.mid = p.mid
                    WHERE p.mid IS NOT NULL AND tmg.mpid = :mpid
                    GROUP BY p.mid, c.days, tmg.experience_adj, tmg.mpid
                ),
                agg AS (
                    SELECT 
                        COUNT(*) AS total_teachers,                      -- b
                        COUNT(*) FILTER (WHERE years_stag > 0) AS prof_teachers, -- a
                        COALESCE(MAX(experience_adj), 0) AS norm,
                        mpid
                    FROM teachers
                    GROUP BY mpid
                )
                SELECT 
                    mpid,
                    prof_teachers, 
                    total_teachers,
                    ROUND(prof_teachers::decimal / NULLIF(total_teachers,0) * 100, 2)::integer AS percent_value,
                    norm::integer,
                    CASE 
                        WHEN ROUND(prof_teachers::decimal / NULLIF(total_teachers,0) * 100, 2) >= GREATEST(COALESCE(norm, 0)::numeric, 1)  
                        THEN 10 ELSE 0 -- Проверить сколько точно нужно баллов
                    END AS result
                FROM agg;
                """
        stmt = text(sql)
        query = conn.execute(stmt, {"year": year, "mpid": mpid})
        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    @staticmethod
    def get_workers_stage_count_stats(lname: str, mpid: int, year: int):
        conn = current_app.ms.db(lname).connect()
        Dashboard.create_tmp_mpid_gid(conn, year)
        sql = """
            WITH RECURSIVE teacher_stag AS (
                SELECT 
                    ts.mid,
                    ts.datebeginstag::date AS datebegin,
                    COALESCE(ts.dateendstag::date, CURRENT_DATE) AS dateend
                FROM xp_teacher_stag ts
                WHERE isprofile != 0
            ),
            ordered AS (
                SELECT 
                    mid, datebegin, dateend,
                    ROW_NUMBER() OVER (PARTITION BY mid ORDER BY datebegin) AS rn
                FROM teacher_stag
            ),
            rec AS (
                SELECT mid, datebegin, dateend, rn
                FROM ordered
                WHERE rn = 1
                UNION ALL
                SELECT o.mid,
                       LEAST(r.datebegin, o.datebegin),
                       GREATEST(r.dateend, o.dateend),
                       o.rn
                FROM rec r
                JOIN ordered o 
                  ON o.mid = r.mid
                 AND o.rn = r.rn + 1
                 AND o.datebegin <= r.dateend
            ),
            merged AS (
                SELECT mid, MIN(datebegin) AS datebegin, MAX(dateend) AS dateend
                FROM rec
                GROUP BY mid
            ),
            calc AS (
                SELECT mid, SUM(dateend - datebegin) AS days
                FROM merged
                GROUP BY mid
            ),
            teachers AS (
                SELECT 
                    p.mid,
                    FLOOR(COALESCE(c.days, 0) / 365.0) AS years_stag  -- целые годы стажа
                FROM tmp_mpid_gid tmg
                JOIN vw_schedule s ON s.main_gid = tmg.gid AND s.school_year = :year
                JOIN people p ON p.mid = ANY(s.teacher_mid)
                LEFT JOIN calc c ON c.mid = p.mid
                WHERE tmg.mpid = :mpid
                GROUP BY p.mid, c.days
            )
            SELECT 
                years_stag::integer,
                COUNT(*) AS teacher_count
            FROM teachers
            GROUP BY years_stag
            ORDER BY years_stag;
            """
        stmt = text(sql)
        query = conn.execute(stmt, {"mpid": mpid, "year": year})
        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    @staticmethod
    def get_students_stats(lname, conn, year):
        sql = """
            WITH groups_vo AS ( -- 6.2. Выпускные группы ВО (любая вышка) 
                SELECT DISTINCT tmg.mpid, tmg.name, tmg.gid
                FROM tmp_mpid_gid tmg
            ),
            diag_classes AS (
                SELECT DISTINCT g.gid, g.mpid, s.sheid, s.week_id 
                FROM groups_vo g 
                LEFT JOIN vw_schedule s ON s.main_gid = g.gid AND s.school_year = :year 
                LEFT JOIN nnz_schedule_modules sm ON sm.sheid = s.sheid 
                WHERE sm.isdiagnostic = TRUE
            ),
            
            question_sets AS (
                SELECT DISTINCT 
                    dc.gid, dc.mpid, 
                    COALESCE(hw.questionset, hwf.questionset, hws.questionset, hws1.questionset) AS questionset
                FROM diag_classes dc
                LEFT JOIN nnz_schedule sh ON sh.sheid = dc.sheid
                LEFT JOIN nnz_weeks w ON sh.sh_var_id = w.sh_var_id AND dc.week_id = w.week_id
                LEFT JOIN nnz_homework hw ON hw.sheid = sh.sheid AND hw.week_id = w.week_id
                LEFT JOIN nnz_homework_fact hwf ON hwf.sheid = sh.sheid AND hwf.week_id = w.week_id
                LEFT JOIN nnz_homework_student hws ON hws.hwid = hw.hwid
                LEFT JOIN nnz_homework_student hws1 ON hws1.hwid = hwf.hwid
            ),
            
            answers AS (
                SELECT DISTINCT qs.gid, qs.mpid, qa.mid, qs.questionset
                FROM question_sets qs
                LEFT JOIN question_answers qa ON qa.questionset = qs.questionset
                LEFT JOIN groupuser_period gp ON gp.mid = qa.mid AND gp.gid = qs.gid
            ),
            
            coverage AS (
                SELECT 
                    gv.mpid,
                    gv.gid,
                    COUNT(DISTINCT a.mid) AS passed_cnt,
                    COUNT(DISTINCT gp.mid) FILTER (WHERE gp.gid = gv.gid) AS total_cnt,
                    ROUND( (COUNT(DISTINCT a.mid)::numeric / NULLIF(COUNT(DISTINCT gp.mid),0)) * 100 ) AS percent_passed,
                    array_agg(DISTINCT a.mid) FILTER (WHERE a.mid IS NOT NULL) AS list_mid,
                    array_agg(DISTINCT a.questionset) FILTER (WHERE a.questionset IS NOT NULL) AS questionset_mids
                FROM groups_vo gv
                LEFT JOIN answers a ON a.gid = gv.gid
                LEFT JOIN groupuser_period gp ON gp.gid = gv.gid
                GROUP BY gv.mpid, gv.gid
            ),
            mpid_result AS (
                SELECT 
                    mpid,
                    MIN(
                        CASE 
                            WHEN total_cnt IS NULL OR total_cnt = 0 THEN 0
                            WHEN percent_passed IS NULL THEN 0
                            WHEN percent_passed < 70 THEN 0
                            ELSE 1
                        END
                    ) AS final_result
                FROM coverage
                GROUP BY mpid
            )
            SELECT 
                c.mpid,
                c.gid,
                COALESCE(c.list_mid, '{}') AS list_mid,
                COALESCE(c.questionset_mids, '{}') AS questionset_mids,
                mr.final_result
            FROM coverage c
            JOIN mpid_result mr ON mr.mpid = c.mpid;
            """
        stmt = text(sql)
        query = conn.execute(stmt, {"year": year})
        result = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

        stats_by_mpid = {}
        final_stats = []
        for item in result:
            if item["final_result"] != 1:
                final_stats.append({
                    "mpid": item["mpid"],
                    "result": 0
                })
                continue

            mpid = item["mpid"]
            if mpid not in stats_by_mpid:
                stats_by_mpid[mpid] = {
                    "total_students": set(),
                    "passed_students": set(),
                }

            questionsets = item["questionset_mids"]
            students = item["list_mid"]
            for mid in students:
                stats_by_mpid[mpid]["total_students"].add(mid)

                # Проверяем все тесты для одного студента
                all_tests_ok = True
                scipped = False
                for qs in questionsets:
                    if not QuestionAnswers().get_question_answers_all(lname, None, mid, qs, None):
                        scipped = True
                        continue

                    cq, tq, cqa, iqa, us, agb, ut, user_correct_percentage = \
                        QuestionSet().get_stats_by_questionset(lname, qs, mid)

                    if user_correct_percentage is None or user_correct_percentage < 70:
                        all_tests_ok = False
                        break

                if all_tests_ok and not scipped:
                    stats_by_mpid[mpid]["passed_students"].add(mid)

        # Преобразуем множества в числа
        for mpid, vals in stats_by_mpid.items():
            total = len(vals["total_students"])
            passed = len(vals["passed_students"])
            percent = (passed / total * 100) if total > 0 else 0

            # выставляем балл
            if percent < 55:
                score = 0
            elif percent < 65:
                score = 50
            else:
                score = 75

            final_stats.append({
                "mpid": mpid,
                "result": score
            })

        return final_stats

    @staticmethod
    def get_college_students_stats(lname, conn, year, mpid):
        sql = """
                WITH groups_vo AS ( -- 6.2. Выпускные группы ВО (любая вышка) 
                    SELECT DISTINCT tmg.mpid, tmg.name, tmg.gid
                    FROM tmp_mpid_gid tmg
                ),
                diag_classes AS (
                    SELECT DISTINCT g.gid, g.mpid, s.sheid, s.week_id 
                    FROM groups_vo g 
                    LEFT JOIN vw_schedule s ON s.main_gid = g.gid AND s.school_year = :year 
                    LEFT JOIN nnz_schedule_modules sm ON sm.sheid = s.sheid 
                    WHERE sm.isdiagnostic = TRUE
                ),

                question_sets AS (
                    SELECT DISTINCT 
                        dc.gid, dc.mpid, 
                        COALESCE(hw.questionset, hwf.questionset, hws.questionset, hws1.questionset) AS questionset
                    FROM diag_classes dc
                    LEFT JOIN nnz_schedule sh ON sh.sheid = dc.sheid
                    LEFT JOIN nnz_weeks w ON sh.sh_var_id = w.sh_var_id AND dc.week_id = w.week_id
                    LEFT JOIN nnz_homework hw ON hw.sheid = sh.sheid AND hw.week_id = w.week_id
                    LEFT JOIN nnz_homework_fact hwf ON hwf.sheid = sh.sheid AND hwf.week_id = w.week_id
                    LEFT JOIN nnz_homework_student hws ON hws.hwid = hw.hwid
                    LEFT JOIN nnz_homework_student hws1 ON hws1.hwid = hwf.hwid
                ),

                answers AS (
                    SELECT DISTINCT qs.gid, qs.mpid, qa.mid, qs.questionset
                    FROM question_sets qs
                    LEFT JOIN question_answers qa ON qa.questionset = qs.questionset
                    LEFT JOIN groupuser_period gp ON gp.mid = qa.mid AND gp.gid = qs.gid
                ),

                coverage AS (
                    SELECT 
                        gv.mpid,
                        gv.gid,
                        COUNT(DISTINCT a.mid) AS passed_cnt,
                        COUNT(DISTINCT gp.mid) FILTER (WHERE gp.gid = gv.gid) AS total_cnt,
                        ROUND( (COUNT(DISTINCT a.mid)::numeric / NULLIF(COUNT(DISTINCT gp.mid),0)) * 100 ) AS percent_passed,
                        array_agg(DISTINCT a.mid) FILTER (WHERE a.mid IS NOT NULL) AS list_mid,
                        array_agg(DISTINCT a.questionset) FILTER (WHERE a.questionset IS NOT NULL) AS questionset_mids
                    FROM groups_vo gv
                    LEFT JOIN answers a ON a.gid = gv.gid
                    LEFT JOIN groupuser_period gp ON gp.gid = gv.gid
                    GROUP BY gv.mpid, gv.gid
                ),
                mpid_result AS (
                    SELECT 
                        mpid,
                        MIN(
                            CASE 
                                WHEN total_cnt IS NULL OR total_cnt = 0 THEN 0
                                WHEN percent_passed IS NULL THEN 0
                                WHEN percent_passed < 70 THEN 0
                                ELSE 1
                            END
                        ) AS final_result
                    FROM coverage
                    GROUP BY mpid
                )
                SELECT 
                    c.mpid,
                    c.gid,
                    COALESCE(c.list_mid, '{}') AS list_mid,
                    COALESCE(c.questionset_mids, '{}') AS questionset_mids,
                    mr.final_result
                FROM coverage c
                JOIN mpid_result mr ON mr.mpid = c.mpid
                WHERE c.mpid = :mpid;
                """
        stmt = text(sql)
        query = conn.execute(stmt, {"year": year, "mpid": mpid})
        result = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

        stats_by_mpid = {}
        final_stats = []
        for item in result:
            if item["final_result"] != 1:
                final_stats.append({
                    "mpid": item["mpid"],
                    "result": 0
                })
                continue

            mpid = item["mpid"]
            if mpid not in stats_by_mpid:
                stats_by_mpid[mpid] = {
                    "total_students": set(),
                    "passed_students": set(),
                }

            questionsets = item["questionset_mids"]
            students = item["list_mid"]
            for mid in students:
                stats_by_mpid[mpid]["total_students"].add(mid)

                # Проверяем все тесты для одного студента
                all_tests_ok = True
                scipped = False
                for qs in questionsets:
                    if not QuestionAnswers().get_question_answers_all(lname, None, mid, qs, None):
                        scipped = True
                        continue

                    cq, tq, cqa, iqa, us, agb, ut, user_correct_percentage = \
                        QuestionSet().get_stats_by_questionset(lname, qs, mid)

                    if user_correct_percentage is None or user_correct_percentage < 70:
                        all_tests_ok = False
                        break

                if all_tests_ok and not scipped:
                    stats_by_mpid[mpid]["passed_students"].add(mid)

        # Преобразуем множества в числа
        for mpid, vals in stats_by_mpid.items():
            total = len(vals["total_students"])
            passed = len(vals["passed_students"])
            percent = (passed / total * 100) if total > 0 else 0

            # выставляем балл
            if percent < 55:
                score = 0
            elif percent < 65:
                score = 50
            else:
                score = 75

            final_stats.append({
                "mpid": mpid,
                "result": score
            })

        return final_stats

    @staticmethod
    def get_students_stats_by_mpid(conn, lname, mpid: int, year: int):
        sql = """
        WITH groups_vo AS (
            SELECT DISTINCT tmg.mpid, tmg.name, tmg.gid
            FROM tmp_mpid_gid tmg
            WHERE tmg.mpid = :mpid
        ),
        diag_classes AS (
            SELECT DISTINCT g.gid, g.mpid, s.sheid, s.week_id
            FROM groups_vo g
            LEFT JOIN vw_schedule s
              ON s.main_gid = g.gid AND s.school_year = :year
            LEFT JOIN nnz_schedule_modules sm
              ON sm.sheid = s.sheid
            WHERE sm.isdiagnostic = TRUE
        ),
        question_sets AS (
            SELECT DISTINCT 
                dc.gid, dc.mpid,
                COALESCE(hw.questionset, hwf.questionset, hws.questionset, hws1.questionset) AS questionset
            FROM diag_classes dc
            LEFT JOIN nnz_schedule sh ON sh.sheid = dc.sheid
            LEFT JOIN nnz_weeks w ON sh.sh_var_id = w.sh_var_id AND dc.week_id = w.week_id
            LEFT JOIN nnz_homework        hw  ON hw.sheid  = sh.sheid AND hw.week_id  = w.week_id
            LEFT JOIN nnz_homework_fact   hwf ON hwf.sheid = sh.sheid AND hwf.week_id = w.week_id
            LEFT JOIN nnz_homework_student hws  ON hws.hwid  = hw.hwid
            LEFT JOIN nnz_homework_student hws1 ON hws1.hwid = hwf.hwid
        ),
        answers AS (
            SELECT DISTINCT qs.gid, qs.mpid, qa.mid, qs.questionset
            FROM question_sets qs
            LEFT JOIN question_answers qa ON qa.questionset = qs.questionset
            LEFT JOIN groupuser_period gp ON gp.mid = qa.mid AND gp.gid = qs.gid
        ),
        coverage AS (
            SELECT 
                gv.mpid,
                gv.name,
                gv.gid,
                COUNT(DISTINCT a.mid) AS passed_cnt,
                COUNT(DISTINCT gp.mid) FILTER (WHERE gp.gid = gv.gid) AS total_cnt,
                ROUND( (COUNT(DISTINCT a.mid)::numeric / NULLIF(COUNT(DISTINCT gp.mid),0)) * 100 ) AS percent_passed,
                array_agg(DISTINCT a.mid)        FILTER (WHERE a.mid IS NOT NULL)        AS list_mid,
                array_agg(DISTINCT a.questionset) FILTER (WHERE a.questionset IS NOT NULL) AS questionset_mids
            FROM groups_vo gv
            LEFT JOIN answers a ON a.gid = gv.gid
            LEFT JOIN groupuser_period gp ON gp.gid = gv.gid
            GROUP BY gv.mpid, gv.gid, gv.name
        ),
        mpid_result AS (
            SELECT 
                mpid,
                MIN(
                    CASE 
                        WHEN total_cnt IS NULL OR total_cnt = 0 THEN 0
                        WHEN percent_passed IS NULL             THEN 0
                        WHEN percent_passed < 70                THEN 0
                        ELSE 1
                    END
                ) AS final_result
            FROM coverage
            GROUP BY mpid
        )
        SELECT 
            c.mpid,
            c.name,
            c.gid,
            COALESCE(c.list_mid, '{}')        AS list_mid,
            COALESCE(c.questionset_mids, '{}') AS questionset_mids,
            mr.final_result
        FROM coverage c
        JOIN mpid_result mr ON mr.mpid = c.mpid;
        """
        rows = conn.execute(text(sql), {"year": year, "mpid": mpid})
        result = [dict(zip(tuple(rows.keys()), r)) for r in rows.cursor]

        # агрегируем по mpid (тут ровно один mpid, но оставим универсально)
        agg = defaultdict(lambda: {"name": None, "total": set(), "passed": set()})
        for item in result:
            cur_mpid = item["mpid"]
            agg[cur_mpid]["name"] = item["name"] or agg[cur_mpid]["name"]

            # если общий результат по mpid = 0, сразу вернём 0% и 0 баллов
            if item["final_result"] != 1:
                agg[cur_mpid]["total"] |= set(item["list_mid"] or [])
                # passed не пополняем — процент будет 0
                continue

            questionsets = item["questionset_mids"] or []
            students = item["list_mid"] or []

            for mid in students:
                agg[cur_mpid]["total"].add(mid)

                # студент должен пройти каждый QS на >=70
                all_ok = True
                skipped_any = False
                for qs in questionsets:
                    if not QuestionAnswers().get_question_answers_all(lname, None, mid, qs, None):
                        skipped_any = True
                        continue
                    *_, user_correct_percentage = QuestionSet().get_stats_by_questionset(lname, qs, mid)
                    if user_correct_percentage is None or user_correct_percentage < 70:
                        all_ok = False
                        break

                if all_ok and not skipped_any:
                    agg[cur_mpid]["passed"].add(mid)

        # формируем вывод
        out = []
        for cur_mpid, val in agg.items():
            total = len(val["total"])
            passed = len(val["passed"])
            percent = round((passed / total * 100), 2) if total > 0 else 0.0

            # шкала пересчёта процента в score
            if percent < 60:
                score = 0
            elif percent < 65:
                score = 5
            else:
                score = 10

            out.append({
                "mpid": cur_mpid,
                "name": val["name"],
                "percent": percent,  # процент, по которому считался score
                "score": score
            })

        # если по mpid вообще нет строк, вернём заглушку
        if not out:
            out = [{"mpid": mpid, "name": None, "percent": 0.0, "score": 0}]

        return out

    @staticmethod
    def get_students_avg_percent(lname, mpid: int, year: int):
        conn = current_app.ms.db(lname).connect()
        Dashboard.create_tmp_mpid_gid(conn, year)
        sql = """
        WITH groups_vo AS (
            SELECT DISTINCT tmg.mpid, tmg.name, tmg.gid, tmg.gname
            FROM tmp_mpid_gid tmg
            WHERE tmg.mpid = :mpid
        ),
        diag_classes AS (
            SELECT DISTINCT g.gid, g.gname, g.mpid, s.sheid, s.week_id
            FROM groups_vo g
            LEFT JOIN vw_schedule s
              ON s.main_gid = g.gid
             AND s.school_year = :year
            LEFT JOIN nnz_schedule_modules sm
              ON sm.sheid = s.sheid
            WHERE sm.isdiagnostic = TRUE
        ),
        question_sets AS (
            SELECT DISTINCT 
                dc.gid, dc.gname, dc.mpid,
                COALESCE(hw.questionset, hwf.questionset, hws.questionset, hws1.questionset) AS questionset
            FROM diag_classes dc
            LEFT JOIN nnz_schedule sh ON sh.sheid = dc.sheid
            LEFT JOIN nnz_weeks w ON sh.sh_var_id = w.sh_var_id AND dc.week_id = w.week_id
            LEFT JOIN nnz_homework        hw  ON hw.sheid  = sh.sheid AND hw.week_id  = w.week_id
            LEFT JOIN nnz_homework_fact   hwf ON hwf.sheid = sh.sheid AND hwf.week_id = w.week_id
            LEFT JOIN nnz_homework_student hws  ON hws.hwid  = hw.hwid
            LEFT JOIN nnz_homework_student hws1 ON hws1.hwid = hwf.hwid
        ),
        answers AS (
            SELECT DISTINCT qs.mpid, qs.gid, qs.gname, qa.mid, qs.questionset
            FROM question_sets qs
            JOIN question_answers qa ON qa.questionset = qs.questionset
            -- ограничение по принадлежности к группе оставим простым,
            -- без доп. проверок и фильтров
        ),
        user_qsets AS (
            SELECT a.mpid, a.mid, a.gname, a.gid, array_agg(DISTINCT a.questionset) AS user_questionset_mids
            FROM answers a
            GROUP BY a.mpid, a.mid, a.gname, a.gid
        )
        SELECT
            uq.mpid,
            uq.mid,
            CONCAT(p.lastname,' ',p.firstname,' ',COALESCE(p.patronymic,'')) AS fio,
            uq.user_questionset_mids, 
            uq.gname, 
            uq.gid
        FROM user_qsets uq
        JOIN people p ON p.mid = uq.mid
        ORDER BY fio;
        """
        rows = conn.execute(text(sql), {"year": year, "mpid": mpid}).fetchall()

        results = []
        for mpid_val, mid, fio, qs_list, gid, gname in rows:
            qs_list = qs_list or []
            percents = []

            # считаем средний процент по всем вопросам, которые студент проходил
            for qs in qs_list:
                # если у тебя есть быстрый чек наличия попытки — можно пропустить,
                # здесь "без других проверок": сразу берём статистику
                cq, tq, cqa, iqa, us, agb, ut, user_correct_percentage = \
                    QuestionSet().get_stats_by_questionset(lname, qs, mid)
                if user_correct_percentage is not None:
                    percents.append(user_correct_percentage)

            avg_percent = round(sum(percents) / len(percents), 2) if percents else 0.0

            results.append({
                "mid": mid,
                "fio": fio,
                "avg_percent": avg_percent,
                "gid": gid,
                "gname": gname
            })

        return results

    @staticmethod
    def get_students_completion_histogram(lname: str, mpid: int, year: int):
        """
            Гистограмма: сколько студентов попало в интервалы процента выполнения тестов.
            Процент для студента = доля QS, по которым набрано >= 70%.
            Если final_result == 0, процент = 0.
            Возвращает:
            {
              "bins": [
                {"label":"0%-29%","count":12},
                {"label":"30%-49%","count":20},
                {"label":"50%-69%","count":35},
                {"label":"70%-89%","count":27},
                {"label":"90%-100%","count":12},
              ],
              "total": 106
            }
        """
        conn = current_app.ms.db(lname).connect()
        Dashboard.create_tmp_mpid_gid(conn, year)
        sql = """
        WITH groups_vo AS (
            SELECT DISTINCT tmg.mpid, tmg.name, tmg.gid, tmg.gname, tmg.mid
            FROM tmp_mpid_gid tmg
            WHERE tmg.mpid = :mpid
        ),
        diag_classes AS (
            SELECT DISTINCT g.gid, g.gname, g.mpid, s.sheid, s.week_id
            FROM groups_vo g
            LEFT JOIN vw_schedule s
              ON s.main_gid = g.gid
             AND s.school_year = :year
            LEFT JOIN nnz_schedule_modules sm
              ON sm.sheid = s.sheid
            WHERE sm.isdiagnostic = TRUE
        ),
        question_sets AS (
            SELECT DISTINCT 
                dc.gid, dc.gname, dc.mpid,
                COALESCE(hw.questionset, hwf.questionset, hws.questionset, hws1.questionset) AS questionset
            FROM diag_classes dc
            LEFT JOIN nnz_schedule sh ON sh.sheid = dc.sheid
            LEFT JOIN nnz_weeks w ON sh.sh_var_id = w.sh_var_id AND dc.week_id = w.week_id
            LEFT JOIN nnz_homework        hw  ON hw.sheid  = sh.sheid AND hw.week_id  = w.week_id
            LEFT JOIN nnz_homework_fact   hwf ON hwf.sheid = sh.sheid AND hwf.week_id = w.week_id
            LEFT JOIN nnz_homework_student hws  ON hws.hwid  = hw.hwid
            LEFT JOIN nnz_homework_student hws1 ON hws1.hwid = hwf.hwid
        ),
        answers AS (
            SELECT DISTINCT qs.mpid, qs.gid, qs.gname, qa.mid, qs.questionset
            FROM question_sets qs
            JOIN question_answers qa ON qa.questionset = qs.questionset
        ),
        user_qsets AS (
            SELECT a.mpid, a.mid, a.gname, a.gid,
                   array_agg(DISTINCT a.questionset) AS user_questionset_mids
            FROM answers a
            GROUP BY a.mpid, a.mid, a.gname, a.gid
        )
        SELECT
            gv.mpid,
            gv.mid,
            CONCAT(p.lastname,' ',p.firstname,' ',COALESCE(p.patronymic,'')) AS fio,
            uq.user_questionset_mids,
            gv.gname,
            gv.gid,
            f.idfaculty,
            f.faculty
        FROM groups_vo gv 
		LEFT JOIN user_qsets uq ON gv.mid = uq.mid
        JOIN people p ON p.mid = gv.mid
        JOIN groupname gn ON gv.gid = gn.gid
        LEFT OUTER JOIN cathedras c ON c.idcathedra = gn.idcathedra
        LEFT OUTER JOIN faculty f ON f.idfaculty = COALESCE(gn.idfaculty, c.faculty)
        ORDER BY fio;
        """
        rows = conn.execute(text(sql), {"year": year, "mpid": mpid}).fetchall()

        # корзины для графика
        bins = [
            {"label": "0%-29%", "count": 0, "lo": 0, "hi": 29},
            {"label": "30%-49%", "count": 0, "lo": 30, "hi": 49},
            {"label": "50%-69%", "count": 0, "lo": 50, "hi": 69},
            {"label": "70%-89%", "count": 0, "lo": 70, "hi": 89},
            {"label": "90%-100%", "count": 0, "lo": 90, "hi": 100},
        ]

        def bucket(pct: int):
            for b in bins:
                if b["lo"] <= pct <= b["hi"]:
                    b["count"] += 1
                    return

        details = []  # по желанию: список студентов для таблицы
        for mpid_val, mid, fio, qs_list, gname, gid, idfaculty, faculty in rows:
            qs_list = qs_list or []
            percents = []
            for qs in qs_list:
                # без доп. проверок: сразу берём статистику по набору
                *_rest, user_correct_percentage = QuestionSet().get_stats_by_questionset(lname, qs, mid)
                if user_correct_percentage is not None:
                    percents.append(float(user_correct_percentage))

            avg_percent = round(mean(percents), 2) if percents else 0.0
            bucket(int(avg_percent))

            details.append({
                "mid": mid,
                "fio": fio,
                "group_name": gname,
                "gid": gid,
                "avg_percent": avg_percent,
                "idfaculty": idfaculty,
                "faculty": faculty
            })

        # убрать служебные границы
        for b in bins:
            b.pop("lo", None);
            b.pop("hi", None)

        stud_info = Dashboard.get_students_stats_by_mpid(conn, lname, mpid, year)

        return {
            "bins": bins,
            "total": len(details),
            "students": details,  # можно не возвращать, если не нужно
            "info": stud_info[0]
        }

    @staticmethod
    def set_stats_to_redis(lname: str, result: list) -> None:
        # print(result)
        current_app.ms.redis(lname).setex(
        "STATS_RESULT", 5, pickle.dumps(result)
        )
        return None

    @staticmethod
    def get_stats_from_redis(lname: str):
        stats_result = current_app.ms.redis(lname).get("STATS_RESULT")
        # print("stats_result", stats_result)
        if stats_result is not None:
            res = pickle.loads(stats_result)
            return res
        return

    @classmethod
    def get_college_disciplines(cls, conn, school_year: int):
        cls.create_tmp_mpid_gid(conn, school_year)
        sql = """
            SELECT DISTINCT mp.mpid, mp.name, mp.shortname
                FROM vw_plan plan
            join militaryprofession mp on mp.mpid = plan.mpid 
            join edu_direction ed on ed.edu_direct_id = mp.edu_direct_id 
            join curriculum c on c.idcurriculum = plan.idcurriculum 
            JOIN preparation_structure ps ON ps.ps_id = ed.program_level
            JOIN education_levels el ON el.ideducationlevel = ps.education_level
            JOIN vw_schedule s ON s.school_year = plan.school_year AND s.cid = plan.cid
            WHERE el.ideducationlevel = 2 -- СПО
                AND plan.school_year = :school_year
        """
        stmt = text(sql)
        query = conn.execute(stmt, {"school_year": school_year})
        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]