# coding=UTF-8
# LMSAPI/API/Journal.py
import datetime

from flask import current_app, g
import pickle

from sqlalchemy import text


class QuestionAnswers:
    # Список возможных полей для таблицы EnrollQuestions кроме id
    possible_fields = [
        "hwid",
        "mid",
        "questionset",
        "question",
        "received",
        "answer_id",
        "answer_num",
        "answer_text",
        "solution",
    ]

    def get_question_answers_all(self, lname, hwid, mid, questionset, question):
        conn = current_app.ms.db(lname).connect()
        where = ""
        if hwid:
            where += " and hwid = :hwid "
        if mid:
            where += " and mid = :mid "
        if questionset:
            where += " and questionset = :questionset "
        if question:
            where += " and question = :question "

        sql = """
            select * from question_answers
            where 1=1
            {where}
            """.format(
            where=where
        )

        stmt = text(sql)
        if hwid:
            stmt = stmt.bindparams(hwid=hwid)
        if mid:
            stmt = stmt.bindparams(mid=mid)
        if questionset:
            stmt = stmt.bindparams(questionset=questionset)
        if question:
            stmt = stmt.bindparams(question=question)

        query = conn.execute(stmt)
        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    def get_question_answers_by_id(self, lname, id):
        conn = current_app.ms.db(lname).connect()

        sql = """
            select * from question_answers
            where id = :id
            """
        stmt = text(sql)
        stmt = stmt.bindparams(id=id)
        query = conn.execute(stmt)
        result = query.fetchone()

        if result is None:
            return None

        result_dict = dict(zip(query.keys(), result))
        return result_dict

    def get_max_traying(self, lname, mid, questionset):
        conn = current_app.ms.db(lname).connect()
        max_trying = conn.execute(
            text("""
                    SELECT COALESCE(MAX(trying), 0) AS new_trying 
                    FROM question_answers 
                    WHERE mid = :mid AND questionset = :questionset
                """), {"mid": mid, "questionset": questionset}
        ).scalar()
        return max_trying

    def create_question_answers(self, lname, data):
        conn = current_app.ms.db(lname).connect()
        mid = data["mid"]
        questionset = data["questionset"]
        question = data["question"]

        # Получаем текущую максимальную попытку
        max_trying = conn.execute(
            text("""
                SELECT COALESCE(MAX(trying), 0) + 1 AS new_trying 
                FROM question_answers 
                WHERE mid = :mid AND questionset = :questionset AND question = :question
            """), {"mid": mid, "questionset": questionset, "question": question}
        ).scalar()

        # Удаляем старые ответы
        conn.execute(
            text("""
                DELETE FROM question_answers 
                WHERE mid = :mid AND questionset = :questionset AND question = :question
            """), {"mid": mid, "questionset": questionset, "question": question}
        )

        # Добавляем новые
        params = {}
        for field in self.possible_fields:
            params[field] = data.get(field)
        params["received"] = datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        params["trying"] = max_trying

        sql = """
            INSERT INTO question_answers(hwid, mid, questionset, question, received,
                                         answer_id, answer_num, answer_text, solution, trying)
            VALUES (:hwid, :mid, :questionset, :question, :received,
                    :answer_id, :answer_num, :answer_text, :solution, :trying)
            RETURNING id
        """
        stmt = text(sql)
        query = conn.execute(stmt, params)
        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    def update_question_answers(self, lname, id, data):
        conn = current_app.ms.db(lname).connect()

        # Формируем SET выражение и параметры
        set_clauses = []
        params = {"id": id}

        for field in self.possible_fields:
            if data.get(field) is not None:
                set_clauses.append("{field} = :{field}".format(field=field))
                params[field] = data.get(field)

        # Проверяем, есть ли поля для обновления
        if not set_clauses:
            raise ValueError("No fields to update")

        # Формируем SQL запрос
        set_clause = ", ".join(set_clauses)
        sql = """
            UPDATE question_answers
            SET {set_clause}
            WHERE id = :id
            RETURNING id
        """.format(
            set_clause=set_clause
        )

        stmt = text(sql).bindparams(**params)
        query = conn.execute(stmt)

        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    def delete_question_answers(self, lname, id):
        conn = current_app.ms.db(lname).connect()
        sql = """
            DELETE FROM question_answers
            WHERE id = :id
            RETURNING id
        """

        stmt = text(sql).bindparams(id=id)
        query = conn.execute(stmt)

        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    def get_availability_answers_test_questions(self, lname, mid, questionset):
        conn = current_app.ms.db(lname).connect()

        sql = """
            SELECT COALESCE(MAX(trying), 0) AS trying
            FROM question_answers qa
            WHERE qa.mid = :mid
                AND qa.questionset = :questionset
            """

        stmt = text(sql)
        stmt = stmt.bindparams(mid=mid, questionset=questionset)

        query = conn.execute(stmt)
        result = query.fetchone()

        if result is None:
            return None
        return dict(zip(query.keys(), result))
