from flask import Blueprint, g
from flask_cors import CORS
from LMSAPI.api.Models.ScholarshipTree import GrantLeafToPersonnelIndicatorService, GrantIndicatorTreeService
from flask import request, jsonify
from LMSAPI.api.Views.TokenAPI import auth

scholarship_tree = Blueprint("scholarship_tree", __name__, url_prefix="/lms/api/v1.0/<lname>")
CORS(scholarship_tree)


@scholarship_tree.route("/scholarship_tree", methods=["POST"])
@auth.login_required
def create_grant_indicator_tree(lname):
    data = request.json
    title = data.get("title")
    parent_id = data.get("parent_id", 0)
    level = data.get("level", 1)
    ordering = data.get("ordering", 0)
    xp_key = data.get("xp_key", None)
    personnel_indicator_ids = data.get("personnel_indicator_ids", [])

    # Проверяем parent_id != 0 и наличие связей в grant_leaf_to_personnel_indicator
    if parent_id != 0:
        existing_link = GrantLeafToPersonnelIndicatorService(lname).get_links_by_grant_leaf_id(parent_id)
        if existing_link:
            return jsonify({"error": "У родительской ветки уже есть связи с достижениями"}), 404

    try:
        result = GrantIndicatorTreeService(lname).create_grant_indicator_tree(
            title=title,
            parent_id=parent_id,
            level=level,
            ordering=ordering,
            xp_key=xp_key,
            personnel_indicator_ids=personnel_indicator_ids,
        )
        return jsonify(result), 201
    except Exception as e:
        return jsonify({"error": str(e)}), 400


@scholarship_tree.route("/scholarship_tree/<int:id>", methods=["PUT"])
@auth.login_required
def update_grant_indicator_tree(lname, id):
    data = request.json
    title = data.get("title")
    parent_id = data.get("parent_id", 0)
    level = data.get("level", 1)
    ordering = data.get("ordering", 0)
    xp_key = data.get("xp_key", None)
    personnel_indicator_ids = data.get("personnel_indicator_ids", [])

    node = GrantIndicatorTreeService(lname).get_by_id(id)
    if not node:
        return jsonify({"error": "Узел не найден"}), 404

    # Проверка, если parent_id != 0 и у него есть связи
    if parent_id != 0 and parent_id != node.parent_id:
        existing_link = GrantLeafToPersonnelIndicatorService(lname).get_links_by_grant_leaf_id(parent_id)
        if existing_link:
            return jsonify({"error": "Невозможно сменить parent_id — у него уже есть связи"}), 400

    # Можно будет убрать эту проверку
    father = GrantIndicatorTreeService(lname).get_by_parent_id(id)
    if father and parent_id != node.parent_id:
        return jsonify({"error": "Невозможно сменить parent_id, так как эта запись уже является родительской"}), 400

    try:
        result = GrantIndicatorTreeService(lname).update_grant_indicator_tree(
            id_=id,
            title=title,
            parent_id=parent_id,
            level=level,
            ordering=ordering,
            xp_key=xp_key,
            personnel_indicator_ids=personnel_indicator_ids,
        )
        return jsonify(result), 201
    except Exception as e:
        return jsonify({"error": str(e)}), 400


@scholarship_tree.route("/scholarship_tree/<int:id>", methods=["DELETE"])
@auth.login_required
def delete_grant_indicator_tree(lname, id):
    approve = request.args.get("approve", "false").lower() == "true"

    node = GrantIndicatorTreeService(lname).get_by_id(id)
    if not node:
        return jsonify({"error": "Узел не найден"}), 404

    # Получить всех потомков узла
    descendants = GrantIndicatorTreeService(lname).get_descendants(id)

    if descendants and not approve:
        return jsonify({
            "error": "У узла есть дочерние элементы. Удаление возможно только с параметром ?approve=true"
        }), 400

    try:
        all_ids_to_delete = [id] + [desc["id"] for desc in descendants]
        # Удаление узлов дерева и их связей с достижениями
        GrantIndicatorTreeService(lname).delete_grant_indicator_tree(all_ids_to_delete)
        return jsonify({"message": "Удалено {all_ids_to_delete} узлов дерева.".format(all_ids_to_delete=len(all_ids_to_delete))}), 200

    except Exception as e:
        return jsonify({"error": str(e)}), 400


@scholarship_tree.route("/scholarship_tree", methods=["GET"])
@auth.login_required
def get_grant_indicator_tree(lname):
    xp_key = request.args.get("xp_key", type=int, default=41)
    tree = GrantIndicatorTreeService(lname).get_tree(xp_key)
    return jsonify(tree), 200


@scholarship_tree.route("/scholarship_tree/tree_rated", methods=["GET"])
@auth.login_required
def get_rated_tree(lname):
    try:
        mid = request.args.get("mid", type=int)
        xp_key = request.args.get("xp_key", type=int)
        education_level = request.args.get("education_level", type=int, default=1)
        prep_struc_category_id = request.args.get("prep_struc_category_id", type=int, default=1)
        scholarship = request.args.get("scholarship", type=int, default=1)

        if not mid or not xp_key:
            return jsonify({"error": "Параметры 'mid' и 'xp_key' обязательны"}), 400

        tree = GrantIndicatorTreeService(lname).get_tree_by_mid(mid, scholarship, xp_key, education_level, prep_struc_category_id)
        return jsonify(tree), 200

    except Exception as e:
        return jsonify({"error": str(e)}), 400


@scholarship_tree.route("/scholarship_tree/list_grants", methods=["GET"])
@auth.login_required
def get_list_stipend_grants(lname):
    try:
        grants = GrantIndicatorTreeService(lname).get_list_stipend_grants()
        return jsonify(grants), 200

    except Exception as e:
        return jsonify({"error": str(e)}), 400


