import textwrap

from LMSAPI.api.utils.swagger_params import Param
from LMSAPI.api.utils.swagger_utils import SwaggerUtils


class SwaggerComments:
    orders_api_getSRW_comment = {
        "tags": ["Научно-исследовательские работы"],
        "summary": "Получить список Научно-исследовательских работ (НИР, SRW)",
        "description": textwrap.dedent("""
            **Работает для пользователей с правами:**
            **- Режимы → Научные работы → Чтение**
            **- Режимы → Научные работы → Полный**

            Данный запрос возвращает список **научно-исследовательских работ**,
            фильтруя результаты по дате, факультету, типу работы, статусу, номеру приказа и
            ряду дополнительных условий.

            🔹 Логика работы:
            - По умолчанию выбираются приказы с типом документа `S` (НИР).
            - Фильтрация:
              - `idorderdate` (год приказа). Если указан, возвращаются приказы только за этот год.
              - `idfaculty` — фильтр по факультету (0 = все факультеты).
              - `order_type_id` — фильтр по виду работы (0 = все типы).
              - `order_state`:
                - `0` — все работы,
                - `1` — только действующие,
                - `2` — только закрытые.
              - `ordernum` — поиск по номеру приказа (`all` = все).
              - `by_user=true` — ограничение по пользователю: выбираются только те приказы, где пользователь:
                - инициатор (`ord_init_mid`),
                - научный руководитель (`nir_instructor_id`),
                - заместитель руководителя (`nir_subinstructor_id`),
                - ответственный исполнитель (`nir_main_executor`),
                - включён в таблицу `order_people`.
              - `from_date` и `to_date` — фильтрация по диапазону дат приказа.
              - `paginateFrom`, `maxResults` — поддержка пагинации.
        """),
        "security": [SwaggerUtils.security],
        "parameters": [
            Param.lname(in_="path", required=True),
            Param.from_date(in_="query", required=False),
            Param.to_date(in_="query", required=False),
            Param.idfaculty(in_="path", required=True),
            Param.xp_key(in_="query", required=False, example=42),
            {
                "name": "idorderdate",
                "in": "path",
                "type": "integer",
                "required": True,
                "description": "Год приказа (0 = все годы)",
                "example": 2024,
            },
            {
                "name": "order_type_id",
                "in": "path",
                "type": "integer",
                "required": True,
                "description": "ID вида работы (0 = все типы)",
                "example": 2,
            },
            {
                "name": "order_state",
                "in": "path",
                "type": "integer",
                "required": True,
                "description": "0=все, 1=действующие, 2=закрытые",
                "example": 1,
            },
            {
                "name": "ordernum",
                "in": "path",
                "type": "string",
                "required": True,
                "description": "Номер приказа (all = все)",
                "example": "all",
            },
            {
                "name": "by_user",
                "in": "query",
                "type": "boolean",
                "required": False,
                "description": "true = только приказы пользователя",
                "example": True,
            },
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — список НИР",
                "schema": {
                    "type": "object",
                    "properties": {
                        "count": {
                            "type": "integer",
                            "description": "Общее количество найденных записей",
                            "example": 25
                        },
                        "scientific_research_work_table": {
                            "type": "array",
                            "items": {
                                "type": "object",
                                "properties": {
                                    "idorder": {
                                        "type": "integer",
                                        "description": "ID приказа",
                                        "example": 123
                                    },
                                    "order_type": {
                                        "type": "string",
                                        "description": "Вид работы",
                                        "example": "НИР (Научная работа)"
                                    },
                                    "ordernum": {
                                        "type": "string",
                                        "description": "Номер приказа",
                                        "example": "45/Н"
                                    },
                                    "orderdate": {
                                        "type": "string",
                                        "description": "Дата приказа",
                                        "example": "2024-03-15"
                                    },
                                    "order_state": {
                                        "type": "string",
                                        "description": "Статус приказа",
                                        "example": "Действующий"
                                    },
                                    "ord_close_date": {
                                        "type": "string",
                                        "description": "Дата закрытия приказа (если есть)",
                                        "example": "2024-07-01"
                                    },
                                    "executor": {
                                        "type": "string",
                                        "description": "Исполнители (ФИО)",
                                        "example": "Иванов И.И., Петров П.П."
                                    },
                                    "nir_instructor_id": {
                                        "type": "string",
                                        "description": "Научный руководитель (ФИО)",
                                        "example": "Сидоров С.С."
                                    },
                                    "nir_subinstructor_id": {
                                        "type": "string",
                                        "description": "Заместитель научного руководителя (ФИО)",
                                        "example": "Кузнецов А.А."
                                    },
                                    "nir_main_executor": {
                                        "type": "string",
                                        "description": "Ответственный исполнитель (ФИО)",
                                        "example": "Васильев В.В."
                                    },
                                    "ord_content": {
                                        "type": "string",
                                        "description": "Содержание приказа",
                                        "example": "Исследование по направлению искусственного интеллекта."
                                    },
                                    "coalesce": {
                                        "type": "string",
                                        "description": "Отдел",
                                        "example": "11 отдел"
                                    },
                                }
                            }
                        }
                    }
                }
            },
            401: SwaggerUtils.defoult_401,
            404: SwaggerUtils.defoult_404,
        }
    }

    division_api_get_divisions_permanent_staff = {
        "tags": ["Постоянный состав", "Научно-исследовательские работы"],
        "summary": "Получить список подразделений (режим 'Постоянный состав')",
        "description": textwrap.dedent("""
            Если пользователь имеет **права администратора (Справочники → Постоянный состав → Администратор → Полный)**:
            - возвращается список всех подразделений

            Если прав **нет**:
            - возвращаются только подразделения, к которым он принадлежит
        """),
        "security": [SwaggerUtils.security],
        "parameters": [
            Param.lname(in_="path", required=True),
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — список подразделений.",
                "schema": {
                    "type": "array",
                    "items": {
                        "type": "object",
                        "properties": {
                            "did": {
                                "type": "integer",
                                "description": "ID подразделения",
                                "example": 101,
                            },
                            "owner_dep": {
                                "type": "integer",
                                "nullable": True,
                                "description": "ID родительского подразделения",
                                "example": 10,
                            },
                            "name": {
                                "type": "string",
                                "description": "Название подразделения",
                                "example": "1 Курс",
                            },
                        },
                    },
                },
            },
            401: SwaggerUtils.defoult_401,
            404: SwaggerUtils.defoult_404,
        },
    }

    orders_api_getOrderTypes = {
        "tags": ["Научно-исследовательские работы"],
        "summary": "Получить список видов приказов (работ)",
        "description": textwrap.dedent("""
                **Запрос возвращает ответ даже если у пользователя нет никаких прав.**

                Данный запрос возвращает список возможных **видов приказов/работ**, 
                которые могут оформляться в системе (например: НИР, дипломные работы, практики и т.д.).

                🔹 Логика работы:
                - Выполняется запрос к таблице `order_types`.
                - Отбираются записи только с `doc_type = 'S'`.
                - Список сортируется по названию (`order_type`).
                - Результат возвращается в виде JSON-объекта с ключом `"order_type"`.
                - Если список пуст → возвращается `404` и объект `{"success": false}`.
            """),
        "security": [SwaggerUtils.security],
        "parameters": [
            Param.lname(in_="path", required=True),
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — список видов приказов",
                "schema": {
                    "type": "object",
                    "properties": {
                        "order_type": {
                            "type": "array",
                            "items": {
                                "type": "object",
                                "properties": {
                                    "order_type_id": {
                                        "type": "integer",
                                        "description": "ID вида работ",
                                        "example": 5
                                    },
                                    "order_type": {
                                        "type": "string",
                                        "description": "Наименование вида работ",
                                        "example": "НИР"
                                    }
                                }
                            }
                        }
                    }
                }
            },
            401: SwaggerUtils.defoult_401,
            404: SwaggerUtils.defoult_404,
        }
    }

    year_api_getSchoolYearList = {
        "tags": ["Переменный состав", "Расписание занятий"],
        "summary": "Получить список учебных годов",
        "description": textwrap.dedent("""
                **Запрос возвращает ответ даже если у него нет никаких прав.**

                Возвращает список учебных лет с датами начала/окончания и доп. атрибутами.
            """),
        "parameters": [
            Param.lname(in_="path", required=True),
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — список учебных лет",
                "schema": {
                    "type": "object",
                    "properties": {
                        "schoolYear": {
                            "type": "array",
                            "items": {
                                "type": "object",
                                "properties": {
                                    "xp_key": {
                                        "type": "integer",
                                        "description": "ID учебного года",
                                        "example": 37,
                                    },
                                    "name": {
                                        "type": "string",
                                        "description": "Название учебного года",
                                        "example": "2024/2025 учебный год",
                                    },
                                    "begdate": {
                                        "type": "string",
                                        "description": "Дата начала",
                                        "example": "01-09-2024",
                                    },
                                    "enddate": {
                                        "type": "string",
                                        "description": "Дата окончания",
                                        "example": "31-05-2025",
                                    },
                                    "year_grade_types": {
                                        "type": "string",
                                        "description": "Не известно",
                                        "example": None,
                                    },
                                },
                            },
                        }
                    }
                },
            },
            404: SwaggerUtils.defoult_404,
        },
    }

    year_api_getEducationYear = {
        "tags": ["Переменный состав", "Расписание занятий"],
        "summary": "Получить список возможных курсов обучения",
        "description": textwrap.dedent("""
            **Запрос возвращает ответ даже если у пользователя нет никаких прав.**
        """),
        "parameters": [
            Param.lname(in_="path", required=True),
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — список курсов обучения",
                "schema": {
                    "type": "object",
                    "properties": {
                        "educationYear": {
                            "type": "array",
                            "items": {
                                "type": "object",
                                "properties": {
                                    "eyid": {
                                        "type": "integer",
                                        "description": "ID курса обучения",
                                        "example": 1
                                    },
                                    "name": {
                                        "type": "string",
                                        "description": "Название курса обучения",
                                        "example": "1 курс"
                                    }
                                }
                            }
                        }
                    }
                }
            },
            404: SwaggerUtils.defoult_404,
        }
    }

    schedule_api_getScheduleFilter = {
        "tags": ["Расписание занятий"],
        "summary": "Получить расписание занятий по фильтрам",
        "description": textwrap.dedent("""
                **Запрос возвращает ответ даже если у пользователя нет прав.**

                Данный запрос формирует и возвращает расписание занятий для заданного факультета,
                курса и учебного года.

                🔹 Основные шаги:
                1. На основе параметров (`schoolyear`, `month`) вычисляется **начальная и конечная дата календарного периода**.
                2. Через функцию `nnz_fill_period_schedule` в БД подготавливаются временные таблицы с расписанием.
                3. Если указан `gid` (ID группы) в query-параметрах — расписание фильтруется только для этой группы.
                4. В итоговом JSON-ответе расписание группируется по **дате** и **номеру пары**.

                🔹 Особенности:
                - Если задан `gid=0` (по умолчанию), то возвращаются все группы факультета.
                - Для каждой записи дополнительно формируется "очищенное" поле `cell`, где из текста убраны HTML-теги.
            """),
        "security": [SwaggerUtils.security],
        "parameters": [
            Param.lname(in_="path", required=True),
            Param.idfaculty(in_="path", required=True),
            Param.educationyear(in_="path", required=True),
            Param.xp_key(in_="path", required=True),
            Param.gid(in_="query", required=False),
            {
                "name": "month",
                "in": "path",
                "type": "integer",
                "required": True,
                "description": "Номер месяца (1–12)",
                "example": 9,
            }
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — расписание занятий",
                "schema": {
                    "type": "object",
                    "additionalProperties": {
                        "type": "object",
                        "description": "Ключ = дата занятия",
                        "additionalProperties": {
                            "type": "array",
                            "description": "Ключ = период (номер пары), значение = список занятий",
                            "items": {
                                "type": "object",
                                "properties": {
                                    "cathedra": {"type": "string", "description": "Название кафедры",
                                                 "example": "12 кафедра"},
                                    "cell": {"type": "string", "description": "Не известно", "example": "Не известно"},
                                    "cellraw": {"type": "string", "description": "Не известно",
                                                "example": "Не известно"},
                                    "course_alias": {"type": "string", "description": "Сокращенное название курса",
                                                     "example": "ВТС1"},
                                    "course_index": {"type": "string", "description": "Индекс курса", "example": "1"},
                                    "course_title": {"type": "string", "description": "Название курса",
                                                     "example": "Введение в теорию систем"},
                                    "educationyear": {"type": "string", "description": "Курс обучения",
                                                      "example": "2 курс"},
                                    "faculty": {"type": "string", "description": "Название факультета",
                                                "example": "Факультет информатики"},
                                    "group_id": {"type": "integer", "description": "ID группы", "example": 1},
                                    "groupname": {"type": "string", "description": "Название группы",
                                                  "example": "ИСТ-21"},
                                    "idcathedra": {"type": "integer", "description": "ID кафедры", "example": 12},
                                    "idfaculty": {"type": "integer", "description": "ID факультета", "example": 1},
                                    "kurs": {"type": "integer", "description": "Курс обучения", "example": 1},
                                    "lessondate": {"type": "string", "description": "Дата занятия",
                                                   "example": "Thu, 04 Sep 2025 00:00:00 GMT"},
                                    "lessondate2": {"type": "string", "description": "Не известно",
                                                    "example": "Thu, 04 Sep 2025 00:00:00 GMT"},
                                    "lessontype_alias": {"type": "string",
                                                         "description": "Сокращенное название типа занятия",
                                                         "example": "Л"},
                                    "lessontype_name": {"type": "string", "description": "Тип занятия",
                                                        "example": "Лекция"},
                                    "meta_course_name": {"type": "string", "description": "Не известно",
                                                         "example": "Не известно"},
                                    "period_id": {"type": "integer", "description": "ID пары занятий", "example": 1},
                                    "period_name": {"type": "string", "description": "Пара/период занятия",
                                                    "example": "2-3 пара"},
                                    "period_time": {"type": "string", "description": "Время занятия",
                                                    "example": "10:00-11:30"},
                                    "rooms": {"type": "string", "description": "Аудитории", "example": "Ауд. 305"},
                                    "schedule_id": {"type": "integer", "description": "ID расписания", "example": 1},
                                    "subject_index": {"type": "string", "description": "Не известно",
                                                      "example": "Не известно"},
                                    "subject_title": {"type": "string", "description": "Не известно",
                                                      "example": "Не известно"},
                                    "teacher_mids": {"type": "string", "description": "ID преподавателей",
                                                     "example": "1,2,3"},
                                    "teachers": {"type": "string", "description": "Преподаватели",
                                                 "example": "Иванов И.И., Петров П.П., Сидоров С.С."},
                                    "tema_num": {"type": "integer", "description": "Не известно",
                                                 "example": "Не известно"},
                                    "wday": {"type": "integer", "description": "ID День недели", "example": 6},
                                    "wday_name": {"type": "string", "description": "День недели", "example": "Сб"},
                                    "week_id": {"type": "integer", "description": "ID недели", "example": 1},
                                }
                            }
                        }
                    }
                }
            },
            401: SwaggerUtils.defoult_401,
            404: SwaggerUtils.defoult_404,
        }
    }

    faculty_api_getFacultyList = {
        "tags": ["Переменный состав", "Расписание занятий"],
        "summary": "Получить список факультетов (только действующих)",
        "description": textwrap.dedent("""
            **Запрос возвращает ответ даже если у пользователя нет никаких прав.**
        """),
        "parameters": [
            Param.lname(in_="path", required=True),
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — список факультетов",
                "schema": {
                    "type": "array",
                    "items": {
                        "type": "object",
                        "properties": {
                            "idfaculty": {
                                "type": "integer",
                                "description": "ID факультета",
                                "example": 1
                            },
                            "faculty": {
                                "type": "string",
                                "description": "Название факультета",
                                "example": "Факультет информатики"
                            }
                        }
                    }
                }
            },
            404: SwaggerUtils.defoult_404,
        }
    }

    group_api_get_groups_to_schedule_classes = {
        "tags": ["Расписание занятий"],
        "summary": "Получить список групп для составления расписания занятий",
        "description": textwrap.dedent("""
            **Запрос возвращает ответ даже если у него нет никаких прав.**

            Данный запрос используется для получения списка учебных групп, 
            к которым можно привязать расписание.  

            Результат зависит от:
            - выбранного учебного года (`xp_key`)
            - факультета (`idfaculty`)
            - курса обучения (`eyid`).

            🔹 Логика работы:
            - Всегда выбираются группы, связанные с указанным учебным годом (`xp_key` или его "главный").
            - Если указан `idfaculty != 0`, то фильтрация ограничивается только группами указанного факультета.
            - Если указан `eyid != 0`, то дополнительно применяется фильтрация по курсу обучения.
            - Если параметры `idfaculty=0` и `eyid=0`, то возвращаются все группы за указанный учебный год.
        """),
        "security": [SwaggerUtils.security],
        "parameters": [
            Param.lname(in_="path", required=True),
            Param.xp_key(in_="query", required=False),
            Param.idfaculty(in_="query", required=False),
            Param.eyid(in_="query", required=False),
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — список групп для расписания",
                "schema": {
                    "type": "array",
                    "items": {
                        "type": "object",
                        "properties": {
                            "gid": {
                                "type": "integer",
                                "description": "ID группы",
                                "example": 1012
                            },
                            "name": {
                                "type": "string",
                                "description": "Название группы",
                                "example": "ИСТ-21"
                            }
                        }
                    }
                }
            },
            401: SwaggerUtils.defoult_401,
            404: SwaggerUtils.defoult_404,
        }
    }

    group_api_get_groups_by_faculty_cathedra_year_eyid = {
        "tags": ["Переменный состав"],
        "summary": "Получить список учебных групп (режим «Переменный состав»)",
        "description": textwrap.dedent("""
            🔹 Логика работы:
            - Если пользователь имеет право **Справочники → Переменный состав → Администратор → Полный**,
              то ему доступны все группы.
            - Если прав администратора нет:
              - то применяются дополнительные фильтры:
                - ограничение по факультетам, кафедрам и подразделениям, доступным пользователю через `permission_groups`;
                - учёт того, что пользователь может быть **классным руководителем** (`class_teacher` / `class_teacher2`).

            🔹 Параметры фильтрации:
            - `idfaculty=0` → фильтр по факультету отключён.
            - `idcathedra=0` → фильтр по кафедре отключён.
            - `eyid=-1` → фильтр по курсу обучения отключён.
            - Если `xp_key > 0`, то отображается только название группы.
            - Если `xp_key = 0`, то к названию группы добавляется учебный год.
        """),
        "security": [SwaggerUtils.security],
        "parameters": [
            Param.lname(in_="path", required=True),
            Param.xp_key(in_="path", required=True),
            Param.idfaculty(in_="query", required=False),
            Param.eyid(in_="query", required=False),
            Param.idcathedra(in_="query", required=False),
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — список групп",
                "schema": {
                    "type": "array",
                    "items": {
                        "type": "object",
                        "properties": {
                            "gid": {
                                "type": "integer",
                                "description": "ID группы",
                                "example": 1012
                            },
                            "group_name": {
                                "type": "string",
                                "description": "Название группы (может содержать год обучения, если xp_key=0)",
                                "example": "ИСТ-21 (2024/2025)"
                            }
                        }
                    }
                }
            },
            401: SwaggerUtils.defoult_401,
            404: SwaggerUtils.defoult_404,
        }
    }

    user_api_getUserDetailsPhotoId = {
        "tags": ["Постоянный состав", "Переменный состав"],
        "summary": "Получить фотографию пользователя по его ID",
        "description": textwrap.dedent("""
            **Запрос возвращает фотографию пользователя даже если у него нет никаких прав.**
        """),
        "produces": ["image/jpeg"],
        "parameters": [
            Param.lname(in_="path", required=True),
            Param.mid(in_="path", required=True),
        ],
        "responses": {
            200: {
                "description": "✅ Фотография пользователя (image/jpeg)",
                "schema": {
                    "type": "string",
                    "format": "binary"
                },
                "headers": {
                    "Content-Disposition": {
                        "type": "string",
                        "description": "Имя файла в формате \"<id>.jpg\""
                    },
                    "Cache-control": {
                        "type": "string",
                        "description": "Заголовок кэширования (max-age=86400)"
                    }
                }
            },
            404: SwaggerUtils.defoult_404
        }
    }

    user_api_getUserList = {
        "tags": ["Постоянный состав"],
        "summary": "Получить список преподавателей (режим 'Постоянный состав')",
        "description": textwrap.dedent("""
                Если пользователь имеет **права администратора (Справочники → Постоянный состав → Администратор → Полный)**:
                  - возвращается полный список преподавателей.

                Если права **(Отчётность → Постоянный состав → Чтение) или (Отчётность → Постоянный состав → Полный)**:
                  - возвращается список только доступных преподавателей (фильтрация по подразделениям, к которым принадлежит запрашиваемый).

                Поддерживает:
                  - фильтрацию по статусу (`status`: works, dismissed, relocated, external, archieved, all)
                  - фильтрацию по подразделению (`division`: id)
                  - пагинацию (`paginateFrom`, `maxResults`)
                  - возврат только количества (`count=true`)
            """),
        "security": [SwaggerUtils.security],
        "parameters": [
            Param.lname(in_="path", required=True),
            Param.division(in_="query", required=False),
            Param.limit(in_="query", required=False),
            Param.offset(in_="query", required=False),
            Param.count(in_="query", required=False),
            {
                "name": "status",
                "in": "query",
                "type": "string",
                "required": False,
                "enum": ["works", "dismissed", "relocated", "external", "archieved", "all"],
                "description": "Фильтр по статусу сотрудника"
            },
        ],
        "responses": {
            200: {
                "description": "**Вариант 1 — список преподавателей:** Возвращается массив объектов Teacher",
                "schema": {
                    "type": "array",
                    "items": {"$ref": "#/definitions/Teacher"}
                }
            },
            201: {
                "description": "**Вариант 2 (возвращается 200) — количество записей:** Возвращается массив с одним объектом Count (если count=true)",
                "schema": {
                    "type": "array",
                    "items": {"$ref": "#/definitions/Count"}
                }
            },
            401: SwaggerUtils.defoult_401,
            404: SwaggerUtils.defoult_404,
        },
        "definitions": {
            "Teacher": {
                "type": "object",
                "properties": {
                    "id": {"type": "integer", "description": "ID преподавателя", "example": 602},
                    "gid": {"type": "integer", "description": "ID группы", "example": 123},
                    "lastname": {"type": "string", "description": "Фамилия", "example": "Иванов"},
                    "firstname": {"type": "string", "description": "Имя", "example": "Иван"},
                    "patronymic": {"type": "string", "description": "Отчество", "example": "Иванович"},
                    "birthday": {"type": "string", "description": "Дата рождения", "example": "12.05.1980"},
                    "email": {"type": "string", "description": "Электронная почта", "example": "ivanov@gmail.com"},
                    "photosize": {"type": "integer", "description": "Размер фото", "example": 12345},
                    "contact": {"type": "string", "description": "Контактные данные", "example": "+79999999999"},
                    "employername": {
                        "type": "string",
                        "description": "Подразделение в структуре",
                        "example": "Факультет информатики / Кафедра математики"
                    },
                    "position": {"type": "string", "description": "Должность",
                                 "example": "Заместитель начальника отдела"},
                    "type": {"type": "string", "description": "Тип персонала", "example": "Военнослужащие"},
                    "militaryrank": {"type": "string", "description": "Воинское звание",
                                     "example": "Старший лейтенант"},
                    "division": {"type": "string", "description": "Подразделение", "example": "Кафедра математики"},
                }
            },
            "Count": {
                "type": "object",
                "properties": {
                    "count": {"type": "integer", "description": "Количество найденных записей", "example": 152}
                }
            }
        }
    }

    cathedra_api_getCathedras = {
        "tags": ["Переменный состав"],
        "summary": "Получить список кафедр по ID факультета",
        "description": textwrap.dedent("""
            **Запрос возвращает ответ даже если у пользователя нет никаких прав.**

            Если `id=0` → возвращаются все действующие кафедры.
        """),
        "parameters": [
            Param.lname(in_="path", required=True),
            Param.idfaculty(in_="path", required=True),
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — список кафедр",
                "schema": {
                    "type": "array",
                    "items": {
                        "type": "object",
                        "properties": {
                            "idcathedra": {
                                "type": "integer",
                                "description": "ID кафедры",
                                "example": 12
                            },
                            "cathedra": {
                                "type": "string",
                                "description": "Название кафедры",
                                "example": "Кафедра информатики"
                            },
                            "faculty": {
                                "type": "integer",
                                "description": "ID факультета, к которому относится кафедра",
                                "example": 1
                            }
                        }
                    }
                }
            },
            404: SwaggerUtils.defoult_404
        }
    }

    user_api_getStudentList = {
        "summary": "Получить список студентов (режим 'Переменный состав')",
        "description": textwrap.dedent("""
            Администратор имеет права: 
            **Справочники → Переменный состав. Администратор → Полный**
        
            Данный запрос возвращает список студентов с учётом:
                - выбранного учебного года
                - факультета, кафедры, курса обучения
                - принадлежности к конкретной группе
                - статуса студента (текущий, выпускник, отчисленный и т. д.)
                - прав доступа пользователя (администратор или нет)
            
            🔹 Логика работы:
                - Если передан параметр `group`, то поиск выполняется только по указанной группе.
                - Если `group` не указан:
                  - выполняется выборка групп через метод `get_groups_by_faculty_cathedra_year_eyid`
                    с фильтрацией по факультету (`faculty`), кафедре (`cathedra`), курсу (`kurs`) и году (`year`).
                  - если список групп пуст, используется фиктивное значение `gid=0`, чтобы запрос не упал.
                - В зависимости от параметра `status`:
                  - `current` → текущие студенты (без отчисленных/выпускников/архивных).
                  - `deducted` → отчисленные в рамках указанного учебного года.
                  - `graduates` → выпускники (при наличии `year` — фильтрация по дате выпуска).
                  - `archieved` → архивные записи.
                  - `new` → вновь поступившие (без группы или только поступление).
                - Если параметр `count=true` → возвращается только количество записей.
                - **Если пользователь не администратор**:
                  - **То есть (Справочники -> Переменный состав. Прочие сведения -> Чтение) или (Справочники -> Переменный состав. Прочие сведения -> Полный)**
                  - применяется проверка прав доступа на основании `permission_groups` и автора записи (`whocreate_mid`).
            
            🔹 Поддерживает:
                - Поиск по ФИО (`search`)
                - Фильтр по категории (`catname`)
                - Пагинацию (`paginateFrom`, `maxResults`)
        """),
        "tags": ["Переменный состав"],
        "security": [SwaggerUtils.security],
        "parameters": [
            Param.lname(in_="path", required=True),
            Param.xp_key(in_="query", required=True, example=37),
            Param.idfaculty(in_="query", required=False, example=0),
            Param.idcathedra(in_="query", required=False, example=0),
            Param.eyid(in_="query", required=False, example=-1),  # курс
            Param.gid(in_="query", required=False),  # группа
            Param.search(in_="query", required=False, description="Поиск по ФИО (частичное совпадение)"),
            Param.count(in_="query", required=False, example=False),
            Param.limit(in_="query", required=False, example=50),
            Param.offset(in_="query", required=False, example=0),
            {
                "name": "status",
                "in": "query",
                "type": "string",
                "required": False,
                "enum": ["current", "deducted", "graduates", "archieved", "new"],
                "description": "Статус студента"
            },
            {
                "name": "catname",
                "in": "query",
                "type": "string",
                "required": False,
                "description": "Фильтр по категории (prep_struc_category.name)"
            }
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — список студентов или количество записей",
                "schema": {
                    "type": "array",
                    "items": {
                        "type": "object",
                        "properties": {
                            "mid": {"type": "integer", "description": "ID студента", "example": 10234},
                            "Фамилия Имя Отчество": {"type": "string", "example": "Иванов Иван Иванович"},
                            "Пол": {"type": "string", "example": "Мужской"},
                            "birthday": {"type": "string", "example": "2003-05-12"},
                            "groupname": {"type": "string", "example": "ИСТ-21"},
                            "Логин": {"type": "string", "example": "ivanov21"},
                            "Статус": {"type": "string", "example": "Включен в состав группы"},
                            "Категория": {"type": "string", "example": "Курсант"},
                            "Фото": {"type": "boolean", "example": True},
                            "Фамилия": {"type": "string", "example": "Иванов"},
                            "Имя": {"type": "string", "example": "Иван"},
                            "Субъект РФ": {"type": "string", "example": "Москва"},
                            "Отчество": {"type": "string"},
                            "Страна": {"type": "string", "example": "Россия"},
                            "Род войск (отец)": {"type": "string", "example": "Род войск"},
                            "Род войск (мать)": {"type": "string", "example": "Род войск"},
                            "Наставник": {"type": "string", "example": "Иванов Иван Иванович"},
                            "Льготы": {"type": "string", "example": "Льготы"},
                            "Куратор": {"type": "string", "example": "Иванов Иван Иванович"},
                            "Военный округ прибытия": {"type": "string", "example": ""},
                            "xp_pf_id": {"type": "integer", "example": 12345},
                            "patronymic": {"type": "string", "example": "Иванович"},
                            "lastname": {"type": "string", "example": "Иванов"},
                            "firstname": {"type": "string", "example": "Иван"},
                        }
                    }
                }
            },
            400: SwaggerUtils.defoult_400,
            401: SwaggerUtils.defoult_401,
            404: SwaggerUtils.defoult_404,
        }
    }

