import datetime
from decimal import Decimal

from sqlalchemy.sql import text
from flask import current_app, g

from LMSAPI.api.Models.File import File
from LMSAPI.api.Models.User import User
from LMSAPI.api.utils.request_utils import RequestUtils


class Homework:
    def get_home_work(self, lname, sheid, week_id, gid):
        conn = current_app.ms.db(lname).connect()
        sql = """
            SELECT hw.*, 
                   CASE 
                       WHEN hw.questionset = 0 OR hw.questionset IS NULL THEN NULL 
                       ELSE qs.name 
                   END AS questionset_name
            FROM nnz_homework hw
            LEFT JOIN questionset qs ON qs.id = hw.questionset
            WHERE sheid = :sheid 
            AND week_id = :week_id
            AND gid = :gid
        """

        stmt = text(sql).bindparams(
            sheid=sheid,
            week_id=week_id,
            gid=gid,
        )
        query = conn.execute(stmt)
        result = query.fetchone()
        if result:
            result = RequestUtils.convert_decimal_to_float(dict(result))
            check_files = File(lname).getGroupHomeworkFiles(sheid, week_id)
            if check_files is None:
                result["group_hw_files"] = None
            else:
                path = File(lname).getGroupHomeworkDirectory(sheid, week_id)
                result["group_hw_files"] = File.check_for_links_in_files(
                    path, check_files
                )
            return dict(result)
        else:
            return result

    def get_home_work_student(self, lname, trmid, sheid, week_id, gid):
        conn = current_app.ms.db(lname).connect()
        sql = """
            DO $$ 
            DECLARE
                term_begin DATE;
                term_end DATE;
            BEGIN
                -- Извлекаем даты начала и окончания семестра
                SELECT t.term_begin, t.term_end INTO term_begin, term_end FROM terms t WHERE trmid = :trmid;
                
                -- Вызываем функцию с этими датами
                PERFORM xp_fill_groupuser_period(term_begin, term_end, :gid);
            END $$;
            
            -- Основной запрос для получения информации о студентах и домашнем задании
            SELECT 
                gp.*, 
                p.mid AS student_mid, 
                xp_format_fio(p.LastName, p.FirstName, p.Patronymic, 0) AS fio,
                hw.hwid,
                hws.text,
				hws.created,
				hws.type,
				hws.indhwid,
				hws.hwid,
				hws.deadline,
				hws.questionset,
                qs.name AS questionset_name
            FROM groupuser_period gp
            JOIN people p ON p.mid = gp.mid
            LEFT JOIN nnz_homework hw ON gp.gid = hw.gid 
                AND hw.sheid = :sheid
                AND hw.week_id = :week_id
                AND hw.gid = :gid
            LEFT JOIN nnz_homework_student hws ON hws.hwid = hw.hwid 
                AND hws.student_mid = gp.mid
                AND hws.type = 2
            LEFT JOIN questionset qs ON qs.id = hws.questionset
            WHERE gp.cid = -1
			ORDER BY p.LastName, p.FirstName, p.Patronymic;
        """

        stmt = text(sql).bindparams(
            trmid=trmid,
            sheid=sheid,
            week_id=week_id,
            gid=gid,
        )
        query = conn.execute(stmt)
        result = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]
        for r in result:
            r["ind_hw_files"] = File(lname).getHomeworkFiles(r["mid"], sheid, week_id)
            r["ind_hw_result_files"] = File(lname).getHomeworkResultFiles(
                r["mid"], sheid, week_id
            )
            r["group_hw_result_files"] = File(lname).getGroupHomeworkResultFiles(
                r["mid"], sheid, week_id
            )
        return result

    def create_home_work(self, lname, data):
        conn = current_app.ms.db(lname).connect()

        sql = "select hwid from nnz_homework where sheid = :sheid and week_id = :week_id and gid = :gid;"
        stmt = text(sql)
        stmt = stmt.bindparams(sheid=data["sheid"])
        stmt = stmt.bindparams(week_id=data["week_id"])
        stmt = stmt.bindparams(gid=data["gid"])
        query = conn.execute(stmt)
        check_homework = query.fetchone()

        trans = conn.begin()
        try:
            if check_homework:
                sql = (
                    "UPDATE nnz_homework SET text = :text, teacher_mid = :teacher_mid, deadline = :deadline, "
                    "work_hours = :work_hours, filescount = :filescount, questionset = :questionset "
                    "WHERE hwid = :hwid;"
                )
                stmt = text(sql).bindparams(
                    hwid=check_homework[0],
                    text=(data.get("text", ""),),
                    teacher_mid=(g.user.mid if g.user.isTeacher else 0,),
                    deadline=(data.get("deadline", None),),
                    work_hours=(data.get("work_hours", None),),
                    filescount=(data.get("filescount", None),),
                    questionset=(data.get("questionset", None),),
                )
                conn.execute(stmt)
                trans.commit()
                return True
            else:
                sql = """
                    INSERT INTO nnz_homework (
                        gid, text, sheid, week_id, teacher_mid, deadline, work_hours, filescount, questionset
                    )
                    VALUES (:gid, :text, :sheid, :week_id, :teacher_mid, :deadline, :work_hours, :filescount, :questionset); 
                    """

                stmt = text(sql).bindparams(
                    sheid=data["sheid"],
                    week_id=data["week_id"],
                    gid=data["gid"],
                    text=data.get("text", ""),
                    teacher_mid=g.user.mid if g.user.isTeacher else 0,
                    deadline=data.get("deadline", None),
                    work_hours=data.get("work_hours", None),
                    filescount=data.get("filescount", None),
                    questionset=data.get("questionset", None),
                )
                conn.execute(stmt)
                trans.commit()
                return True
        except Exception as e:
            print(e)
            trans.rollback()
            return False

    def delete_home_work(self, lname, hwid):
        conn = current_app.ms.db(lname).connect()
        trans = conn.begin()
        try:
            sql = "DELETE FROM nnz_homework WHERE hwid = :hwid "
            stmt = text(sql).bindparams(
                hwid=hwid,
            )
            conn.execute(stmt)
            trans.commit()
            return True
        except Exception as e:
            print(e)
            trans.rollback()
            return False

    def create_homework_student(self, lname, data):
        conn = current_app.ms.db(lname).connect()

        sql = "select indhwid from nnz_homework_student where hwid = :hwid and student_mid = :mid;"
        stmt = text(sql)
        stmt = stmt.bindparams(hwid=data["hwid"], mid=data["student_mid"])
        query = conn.execute(stmt)
        check_homework = query.fetchone()

        trans = conn.begin()
        try:
            if check_homework:
                sql = """
                    UPDATE nnz_homework_student SET 
                        text=:text, student_mid=:student_mid, type=:type, hwid=:hwid, deadline=:deadline, 
                        questionset=:questionset
                    WHERE indhwid=:indhwid
                    """

                stmt = text(sql).bindparams(
                    indhwid=check_homework[0],
                    text=data.get("text", ""),
                    student_mid=data["student_mid"],
                    type=data.get("type", 2),
                    hwid=data["hwid"],
                    deadline=data.get("deadline", None),
                    questionset=data.get("questionset", None),
                )
                User(lname).prepare_logger(conn)
                conn.execute(stmt)
                trans.commit()
                return True
            else:
                sql = """
                    INSERT INTO nnz_homework_student (
                        text, student_mid, created, type, hwid, deadline, questionset
                    )
                    VALUES (:text, :student_mid, :created, :type, :hwid, :deadline, :questionset); 
                    """

                stmt = text(sql).bindparams(
                    text=data.get("text", ""),
                    student_mid=data["student_mid"],
                    created=datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S"),
                    type=data.get("type", 2),
                    hwid=data["hwid"],
                    deadline=data.get("deadline", None),
                    questionset=data.get("questionset", None),
                )
                User(lname).prepare_logger(conn)
                conn.execute(stmt)
                trans.commit()
                return True
        except Exception as e:
            print(e)
            trans.rollback()
            return False

    def delete_homework_student(self, lname, indhwid):
        conn = current_app.ms.db(lname).connect()
        trans = conn.begin()
        try:
            sql = "DELETE FROM nnz_homework_student WHERE indhwid = :indhwid "
            stmt = text(sql).bindparams(
                indhwid=indhwid,
            )
            conn.execute(stmt)
            trans.commit()
            return True
        except Exception as e:
            print(e)
            trans.rollback()
            return False

    def get_questionset_by_home_work(self, lname, hwid):
        conn = current_app.ms.db(lname).connect()
        sql = """
            SELECT q.* FROM questionset q
            JOIN nnz_homework hw ON hw.questionset = q.id
            WHERE hw.hwid = :hwid
        """

        stmt = text(sql).bindparams(
            hwid=hwid,
        )
        query = conn.execute(stmt)
        result = query.fetchone()

        if result is None:
            return None

        result_dict = dict(zip(query.keys(), result))
        return result_dict

    def get_questionset_by_home_work_student(self, lname, indhwid):
        conn = current_app.ms.db(lname).connect()
        sql = """
            SELECT q.* FROM questionset q
            JOIN nnz_homework_student hw ON hw.questionset = q.id
            WHERE hw.indhwid = :indhwid
        """

        stmt = text(sql).bindparams(
            indhwid=indhwid,
        )
        query = conn.execute(stmt)
        result = query.fetchone()

        if result is None:
            return None

        result_dict = dict(zip(query.keys(), result))
        return result_dict

    def get_lesson_check(self, lname, sheid, instance):
        # Проверка стоит ли галочка о том проведено ли занятие или нет
        conn = current_app.ms.db(lname).connect()
        nnz_sh_module_fact_sql = """
            select * from nnz_sh_module_fact where sheid = :sheid and week_id = :instance
        """
        nnz_homework_fact_sql = """
            select * from nnz_homework_fact where sheid = :sheid and week_id = :instance
        """

        nnz_sh_module_fact_stmt = text(nnz_sh_module_fact_sql).bindparams(
            sheid=sheid,
            instance=instance,
        )
        query = conn.execute(nnz_sh_module_fact_stmt)
        nnz_sh_module_fact_result = query.fetchone()

        nnz_homework_fact_stmt = text(nnz_homework_fact_sql).bindparams(
            sheid=sheid,
            instance=instance,
        )
        query = conn.execute(nnz_homework_fact_stmt)
        nnz_homework_fact_result = query.fetchone()

        if nnz_sh_module_fact_result is None and nnz_homework_fact_result is None:
            return False
        else:
            return True

    def get_homwork_info(self, lname, termid, mid, cid):

        conn = current_app.ms.db(lname).connect()
        q1 = """
            SELECT DISTINCT
                COALESCE(hwf.hwid, hw.hwid) AS hwid,
                COALESCE(hwf.text, hw.text) AS homeworktext,
                s.sheid,
                s.week_id AS instance,
                COALESCE(hwf.questionset, hw.questionset) AS group_questionset,
                COALESCE(qs1.name, qs.name) AS group_questionset_name,
                COALESCE(qs1.trying, qs.trying) AS group_questionset_trying,
                (
                    SELECT MAX(qa.trying)
                    FROM question_answers qa
                    WHERE qa.mid = :mid
                      AND qa.questionset = COALESCE(hwf.questionset, hw.questionset)
                ) AS group_questionset_user_trying,
                COALESCE(hws1.indhwid, hws.indhwid) as indhwid,
                COALESCE(hws1.text, hws.text) AS individualhomeworktext,
                COALESCE(hws1.questionset, hws.questionset) AS student_questionset,
                COALESCE(qs3.name, qs2.name) AS student_questionset_name,
                COALESCE(qs3.trying, qs2.trying) AS student_questionset_trying,
                (
                    SELECT MAX(qa.trying)
                    FROM question_answers qa
                    WHERE qa.mid = :mid
                      AND qa.questionset = COALESCE(hws1.questionset, hws.questionset)
                ) AS student_questionset_user_trying,
                coalesce(sg.grade, '') AS grade,
                COALESCE(NULLIF(smf.tema, ''), NULLIF(sm.tema, ''), 'тема занятия не задана') AS otitle,  -- Название темы занятия
                COALESCE(et1.typename, et.typename) AS lessontype,  -- Тип занятия
                COALESCE(et1.typeid, et.typeid) AS lessontypeid,  -- Идентификатор типа занятия
                s.lesson_date AS lessonfulldate,
                CASE 
                    WHEN EXISTS (
                        SELECT 1 
                        FROM question_answers qa
                        WHERE qa.mid = :mid
                          AND qa.questionset = COALESCE(hwf.questionset, hw.questionset)
                    ) 
                    THEN TRUE 
                    ELSE FALSE 
                END AS group_question_answers,
                CASE 
                    WHEN EXISTS (
                        SELECT 1 
                        FROM question_answers qa
                        WHERE qa.mid = :mid
                          AND qa.questionset = COALESCE(hws1.questionset, hws.questionset)
                    ) 
                    THEN TRUE 
                    ELSE FALSE 
                END AS student_question_answers
            FROM vw_schedule s
            JOIN nnz_schedule sh ON sh.sheid = s.sheid 
            JOIN nnz_weeks w ON sh.sh_var_id = w.sh_var_id and s.week_id = w.week_id
            LEFT JOIN nnz_homework hw ON hw.sheid = sh.sheid AND hw.week_id = w.week_id
            LEFT JOIN nnz_homework_fact hwf ON hwf.sheid = sh.sheid AND hwf.week_id = w.week_id
            LEFT JOIN questionset qs ON qs.id = hw.questionset
            LEFT JOIN questionset qs1 ON qs1.id = hwf.questionset
            LEFT JOIN nnz_homework_student hws ON hws.hwid = hw.hwid AND hws.student_mid = :mid
            LEFT JOIN nnz_homework_student hws1 ON hws1.hwid = hwf.hwid AND hws1.student_mid = :mid
            LEFT JOIN questionset qs2 ON qs2.id = hws.questionset
            LEFT JOIN questionset qs3 ON qs3.id = hws1.questionset
            LEFT JOIN nnz_schedule_modules sm ON sm.sheid = sh.sheid AND sm.week_id = w.week_id
            LEFT JOIN nnz_sh_module_fact smf ON smf.sheid = sh.sheid AND smf.week_id = w.week_id
            LEFT JOIN eventtools et ON et.typeid = sh.pair_type_id
            LEFT JOIN eventtools et1 ON et1.typeid = sh.pair_type_id
			LEFT JOIN processes pcs ON pcs.pid = et.processtypeid or pcs.pid = et1.processtypeid
            left outer join nnz_sh_grades sg on sg.sheid = sh.sheid and sg.week_id = w.week_id and sg.student_mid = :mid
            WHERE sh.cid = :cid
              and s.trmid = :termid  
              and case when pcs.pid = 3 and pcs.is_exam = 1 THEN FALSE ELSE TRUE END     
              AND sh.gid = (
                        SELECT 
                            g.gid groupname
                        FROM people p
                        LEFT OUTER JOIN groupuser gu ON p.mid = gu.mid AND gu.cid = -1
                        LEFT OUTER JOIN groupname g ON gu.gid = g.gid AND g.cid = -1
                        WHERE p.mid = :mid
                    )
            ORDER BY 
                s.lesson_date;
        """

        stmt = text(q1)
        stmt = stmt.bindparams(termid=termid, cid=cid, mid=mid)
        query = conn.execute(stmt)

        cp = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]
        af = File(lname)
        for s in cp:
            s['filesExist'] = af.homeworkFilesExist(mid, s['sheid'], s['instance']) or af.homework_student_fact_files_exist(mid, s['sheid'], s['instance'])
            s['resultFilesExist'] = af.homeworkResultFilesExist(mid, s['sheid'], s['instance'])
            s['groupFilesExist'] = af.groupHomeworkFilesExist(s['sheid'], s['instance']) or af.homework_fact_files_exist(s['sheid'], s['instance'])
            s['groupResultFilesExist'] = af.groupHomeworkResultFilesExist(mid, s['sheid'], s['instance'])
        return cp
