# coding=UTF-8
# LMSAPI/API/Faculty.py
import json
from datetime import datetime

from sqlalchemy.sql import text
from flask import current_app


class ImportApplicant:
    def __init__(self, cname):
        self.lname = cname

    def validate_date(self, date_str):
        """
        Проверяет, является ли строка допустимой датой в формате YYYY-MM-DD.
        Возвращает None, если дата недопустима.
        """
        if not date_str or date_str == "0000-00-00":
            return None  # Заменить на None для базы данных
        try:
            return datetime.strptime(date_str, "%Y-%m-%d").date()
        except ValueError:
            return None  # Если формат неверен, также возвращаем None

    def create_xp_applicant(self, data):
        conn = current_app.ms.db(self.lname).connect()
        trans = conn.begin()
        try:
            sql = """
                INSERT INTO xp_applicant (date,
                    child_lastname, child_firstname, child_midname, body, parents_addr, child_birth, tel_mobile, email, 
                    post_addr, documenttype, documentseria, documentnumber, documentwhogive, documentwhengive, rfidid, 
                    school2enter, tel_home, inn, healthstate, socialstatus, i_familystatus, birthplace, 
                    sex, father_country, category, studyyear, info_json
                )
                VALUES (now(),:child_lastname, :child_firstname, :child_midname, :body, :parents_addr, :child_birth, :tel_mobile, :email, 
                    :post_addr, :documenttype, :documentseria, :documentnumber, :documentwhogive, :documentwhengive, :rfidid, 
                    :school2enter, :tel_home, :inn, :healthstate, :socialstatus, :i_familystatus, :birthplace, 
                    :sex, :father_country, :category, :studyyear, :info_json)
                RETURNING xp_key;
                """

            stmt = text(sql).bindparams(
                child_lastname=data.get("LastName"),
                child_firstname=data.get("FirstName"),
                child_midname=data.get("Patronymic"),
                body=data.get("Information", None),
                parents_addr=data.get("Address", None),
                child_birth=self.validate_date(data.get("BirthDate")),
                tel_mobile=data.get("CellularNumber", None),
                email=data.get("Email", None),
                post_addr=data.get("RegistrationAddress", None),
                documenttype=data.get("DocumentType", None),
                documentseria=data.get("DocumentSeria", None),
                documentnumber=data.get("DocumentNumber", None),
                documentwhogive=data.get("DocumentWhoGive", None),
                documentwhengive=self.validate_date(data.get("DocumentWhenGive")),
                rfidid=data.get("RFIDID", None),
                school2enter=data.get("School2Enter", None),
                tel_home=data.get("Phone home", None),
                inn=data.get("INN", None),
                healthstate=data.get("HealthGroup", None),
                socialstatus=data.get("FamilyStatus", None),
                i_familystatus=data.get("FamilyStatus (family_status)", None),
                birthplace=data.get("Birthplace", None),
                sex=data.get("Sex", None),
                father_country=data.get("Citizenship", None),
                category=data.get("category", None),
                studyyear=0,
                info_json=json.dumps(data),
            )

            query = conn.execute(stmt)
            trans.commit()
            for row in query:
                return row.xp_key
        except Exception as e:
            print(e)
            trans.rollback()
            return None

    def check_exist_xp_applicant_v1(self, data):
        conn = current_app.ms.db(self.lname).connect()
        stmt = text(
            """
            select xp_key from xp_applicant 
            where info_json = :info_json
            """
        )

        # Преобразуем dict в JSON-строку
        serialized_data = json.dumps(data, ensure_ascii=False)

        stmt = stmt.bindparams(info_json=serialized_data)

        query = conn.execute(stmt)
        result = query.first()
        return result

    def create_import_applicant_task(self, task_id, mid, status):
        try:
            conn = current_app.ms.db(self.lname).connect()
            sql = """
                insert into import_applicant_task (task_id, mid, created_at, status) 
                values (:task_id, :mid, NOW(),:status)
                """
            stmt = text(sql)
            stmt = stmt.bindparams(task_id=task_id)
            stmt = stmt.bindparams(mid=mid)
            stmt = stmt.bindparams(status=status)
            query = conn.execute(stmt)
            return True
        except Exception as e:
            print(e)
            return False

    def create_import_applicant_task_info(self, task_id, status, path):
        conn = current_app.ms.db(self.lname).connect()
        try:
            sql = """
                insert into import_applicant_task_info (task_id, status, path) 
                values (:task_id, :status, :path)
                returning id;
                """
            stmt = text(sql)
            stmt = stmt.bindparams(task_id=task_id, status=status, path=path)
            query = conn.execute(stmt)
            for row in query:
                return row.id
        except Exception as e:
            print(e)
            return None

    def create_import_applicant_task_info_full(self, task_id, status, path, data):
        conn = current_app.ms.db(self.lname).connect()
        try:
            sql = """
                insert into import_applicant_task_info (task_id, status, path, data) 
                values (:task_id, :status, :path, :data)
                returning id;
                """
            stmt = text(sql)
            stmt = stmt.bindparams(
                task_id=task_id,
                status=status,
                path=path,
                data=json.dumps(data, ensure_ascii=False),
            )
            query = conn.execute(stmt)
            for row in query:
                return row.id
        except Exception as e:
            print(e)
            return None

    def update_status_import_applicant_task(self, task_id, status):
        conn = current_app.ms.db(self.lname).connect()
        sql = """
            UPDATE import_applicant_task
            SET status = :status
            WHERE task_id = :task_id
            """

        stmt = text(sql)
        stmt = stmt.bindparams(task_id=task_id, status=status)
        conn.execute(stmt)

    def update_status_import_applicant_task_info(self, id, status, xp_key):
        conn = current_app.ms.db(self.lname).connect()
        sql = """
            UPDATE import_applicant_task_info
            SET status = :status, xp_key = :xp_key
            WHERE id = :id
            """

        stmt = text(sql)
        stmt = stmt.bindparams(
            id=id,
            status=status,
            xp_key=xp_key,
        )
        conn.execute(stmt)

    def get_import_applicant_task(self, task_id):
        conn = current_app.ms.db(self.lname).connect()
        sql = """
            SELECT * FROM import_applicant_task
            WHERE task_id = :task_id
            """
        stmt = text(sql)
        stmt = stmt.bindparams(task_id=task_id)
        query = conn.execute(stmt)
        result = query.first()
        return result

    def get_import_applicant_task_info(self, task_id, limit, offset):
        conn = current_app.ms.db(self.lname).connect()
        sql = """
            SELECT 
                task_id, 
                status, 
                path, 
                data ->> 'LastName' AS LastName,
                data ->> 'FirstName' AS FirstName,
                data ->> 'Patronymic' AS Patronymic,
                data ->> 'BirthDate' AS BirthDate,
                xp_key
            FROM import_applicant_task_info
            WHERE task_id = :task_id
            LIMIT :limit
            OFFSET :offset
            """
        stmt = text(sql)
        stmt = stmt.bindparams(task_id=task_id, limit=limit, offset=offset)
        query = conn.execute(stmt)
        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]
