# coding=UTF-8
# LMSAPI/API/Schedule.py
from dateutil.parser import parse
from sqlalchemy import create_engine
from sqlalchemy.exc import DataError
from sqlalchemy.sql import text
from flask import current_app
import pickle

from LMSAPI.api.Models.Schoolyear import Schoolyear
from LMSAPI.api.Models.User import User
from datetime import datetime, date, timedelta


class Schedule:
    def __init__(self, cname):
        self.lname = cname

    def getListFromRedis(self):
        cachedClist = current_app.ms.redis(self.lname).get("SCDL")
        if cachedClist is not None:
            clist = pickle.loads(cachedClist)
            return clist
        return

    def setListToRedis(self, ylist):
        current_app.ms.redis(self.lname).setex("SCDL", 5, pickle.dumps(ylist))
        return

    def getSchedule(self, mid, scheduledate):
        conn = current_app.ms.db(self.lname).connect()

        stmt = text(
            """
        SELECT MAX(p.short_name) period, 
            to_char(to_timestamp(cast(div(MAX(p.starttime), 60) as char(2)) || ':' || cast(mod(MAX(p.starttime), 60) as char(2)), 'HH24:MI'), 'HH24:MI') "start",
            to_char(to_timestamp(cast(div(MAX(p.stoptime), 60) as char(2)) || ':' || cast(mod(MAX(p.stoptime), 60) as char(2)), 'HH24:MI'), 'HH24:MI') stop,
            MAX(courses.title) course,
            MAX(meta_course.name) meta_course_name,
            STRING_AGG(DISTINCT COALESCE(rooms.short_name, :nd), ', ') AS rooms,
            STRING_AGG(DISTINCT COALESCE(CASE WHEN pf."ShedulingName" > '' THEN pf."ShedulingName" ELSE xp_f_get_mid_fio(teacher.mid, 1) END, :nd), ', ') AS teachers,
            '' as grade,
            '' as homework
        FROM groupuser gu
        JOIN vw_schedule schedule ON schedule.gid = gu.gid
        JOIN courses ON courses.cid = schedule.cid
        LEFT JOIN meta_course ON meta_course.id = courses.meta_course
        JOIN rooms ON rooms.rid = ANY(schedule.rid)
        JOIN people teacher ON teacher.mid = ANY(schedule.teacher_mid)
        JOIN periods p ON p.lid = schedule.period
        LEFT OUTER JOIN xp_personal_file pf ON teacher.mid = pf.mid
        WHERE gu.mid = :id and schedule.lesson_date = :scdate
        GROUP BY schedule.sheid ORDER BY PERIOD

        """
        )

        stmt = stmt.bindparams(id=mid)
        stmt = stmt.bindparams(nd="н/д")
        stmt = stmt.bindparams(scdate=scheduledate)
        query = conn.execute(stmt)
        # current_app.logger.debug(stmt)
        scheduleList = {
            "schedule": [dict(zip(tuple(query.keys()), i)) for i in query.cursor]
        }

        return scheduleList

    def getScheduleGroup(self, gid, scheduledate):
        conn = current_app.ms.db(self.lname).connect()

        stmt = text(
            """
        SELECT MAX(p.short_name) period, 
            to_char(to_timestamp(cast(div(MAX(p.starttime), 60) as char(2)) || ':' || cast(mod(MAX(p.starttime), 60) as char(2)), 'HH24:MI'), 'HH24:MI') "start",
            to_char(to_timestamp(cast(div(MAX(p.stoptime), 60) as char(2)) || ':' || cast(mod(MAX(p.stoptime), 60) as char(2)), 'HH24:MI'), 'HH24:MI') stop,
            MAX(courses.title) course, 
            STRING_AGG(DISTINCT COALESCE(rooms.short_name,:nd), ', ') AS rooms,
            STRING_AGG(DISTINCT COALESCE(CASE WHEN pf."ShedulingName" > '' THEN pf."ShedulingName" ELSE xp_f_get_mid_fio(teacher.mid, 1) END, :nd), ', ') AS teachers,
            '' as grade,
            '' as homework
        FROM vw_schedule schedule
        INNER JOIN courses ON courses.cid = schedule.cid
        INNER JOIN rooms ON rooms.rid = ANY(schedule.rid)
        INNER JOIN people teacher ON teacher.mid = ANY(schedule.teacher_mid)
        INNER JOIN periods p ON p.lid = schedule.period
        LEFT OUTER JOIN xp_personal_file pf ON teacher.mid = pf.mid
        WHERE schedule.gid = :gid AND schedule.lesson_date = :scdate
        GROUP BY schedule.sheid ORDER BY Period

        """
        )

        stmt = stmt.bindparams(gid=gid)
        stmt = stmt.bindparams(nd="н/д")
        stmt = stmt.bindparams(scdate=scheduledate)
        query = conn.execute(stmt)
        # current_app.logger.debug(query)
        scheduleList = {
            "schedule": [dict(zip(tuple(query.keys()), i)) for i in query.cursor]
        }

        return scheduleList

    def getStartAndEnddate(self, schoolyear, month):
        conn = current_app.ms.db(self.lname).connect()

        sql = """
        SELECT 
        CASE 
          WHEN (str_to_date(:month||CAST(EXTRACT(YEAR FROM begdate) AS VARCHAR(4)), '%d.%m.%Y') >= begdate AND 
                str_to_date(:month||CAST(EXTRACT(YEAR FROM begdate) AS VARCHAR(4)), '%d.%m.%Y') < enddate ) THEN
            str_to_date(:month||CAST(EXTRACT(YEAR FROM begdate) AS VARCHAR(4)), '%d.%m.%Y')
          ELSE
            str_to_date(:month||CAST(EXTRACT(YEAR FROM enddate) AS VARCHAR(4)), '%d.%m.%Y')
           
        END startdate, 
        
        CASE 
          WHEN (str_to_date(:month||CAST(EXTRACT(YEAR FROM begdate) AS VARCHAR(4)), '%d.%m.%Y') >= begdate AND 
                str_to_date(:month||CAST(EXTRACT(YEAR FROM begdate) AS VARCHAR(4)), '%d.%m.%Y') < enddate ) THEN
            str_to_date(:month||CAST(EXTRACT(YEAR FROM begdate) AS VARCHAR(4)), '%d.%m.%Y')
          ELSE
            str_to_date(:month||CAST(EXTRACT(YEAR FROM enddate) AS VARCHAR(4)), '%d.%m.%Y')
         
        END  + INTERVAL '1 MONTH - 1 day' enddate
        FROM school_year 
        WHERE xp_key = :schoolyear 
        """

        mq = "01." + str(month) + "."
        stmt = text(sql)
        stmt = stmt.bindparams(month=mq)
        stmt = stmt.bindparams(schoolyear=schoolyear)
        try:
            query = conn.execute(stmt)
        except DataError as e:
            return None

        result = {}
        for row in query:
            result["startdate"] = datetime.strftime(row.startdate, "%d.%m.%Y")
            result["enddate"] = datetime.strftime(row.enddate, "%d.%m.%Y")
            return result

    def getTerms(self, startdate, enddate):
        conn = current_app.ms.db(self.lname).connect()

        sql = """
        select * from terms
        where (:startdate BETWEEN  str_to_date(term_begin, '%d.%m.%Y') AND str_to_date(term_end, '%d.%m.%Y')) or
        (:enddate BETWEEN  str_to_date(term_begin, '%d.%m.%Y') AND str_to_date(term_end, '%d.%m.%Y'));
        """

        stmt = text(sql)
        stmt = stmt.bindparams(startdate=startdate)
        stmt = stmt.bindparams(enddate=enddate)
        query = conn.execute(stmt)

        result = []
        for row in query:
            result.append(row.trmid)

        return result

    def getListFromRedisFilter(self, faculty, educationyear, schoolyear, month):
        cname = (
            "SCDLF-"
            + str(faculty)
            + "-"
            + str(educationyear)
            + "-"
            + str(schoolyear)
            + "-"
            + str(month)
        )
        cachedClist = current_app.ms.redis(self.lname).get(cname)
        if cachedClist is not None:
            clist = pickle.loads(cachedClist)
            return clist
        return

    def setListToRedisFilter(self, ylist, faculty, educationyear, schoolyear, month):
        cname = (
            "SCDLF-"
            + str(faculty)
            + "-"
            + str(educationyear)
            + "-"
            + str(schoolyear)
            + "-"
            + str(month)
        )
        current_app.ms.redis(self.lname).setex(cname, 120, pickle.dumps(ylist))
        return

    def get_calendar_start_end(self, year_id: int, month: int):
        # Get the school year information
        school_year = Schoolyear(self.lname).getSchoolYear(year_id)["schoolYear"][0]

        # Parse dates from string (handling DD-MM-YYYY format)
        def parse_date(date_str):
            if isinstance(date_str, date):
                return date_str

            return parse(date_str, dayfirst=True).date()

        begdate = parse_date(school_year['begdate'])
        enddate = parse_date(school_year['enddate'])

        # Determine the actual year for the month
        if month >= begdate.month:
            year = begdate.year
        else:
            year = enddate.year

        # Create first and last day of the month
        try:
            first_day = date(year, month, 1)
        except ValueError:
            first_day = begdate

        # Calculate last day of month
        last_day = date(year, month + 1, 1) - timedelta(days=1) if month < 12 else date(year, 12, 31)

        # Adjust to stay within school year boundaries
        first_day = max(first_day, begdate)
        last_day = min(last_day, enddate)

        # Calculate calendar view start (previous Sunday) and end (next Saturday)
        start_date = first_day - timedelta(days=first_day.weekday())
        end_date = last_day + timedelta(days=(6 - last_day.weekday()))

        # Adjust calendar view to stay within school year if needed
        start_date = max(start_date, begdate)
        end_date = min(end_date, enddate)

        return start_date, end_date

    def getScheduleFilter(self, faculty, educationyear, schoolyear, month, gid):
        start_date, end_date = self.get_calendar_start_end(schoolyear, month)
        if start_date is None or end_date is None:
            return None

        conn = current_app.ms.db(self.lname).connect()
        stmt = text(
            "select nnz_fill_period_schedule(:schoolyear, :faculty, :educationyear, 0, 0, :startdate, :enddate);"
        )

        stmt = stmt.bindparams(schoolyear=schoolyear)
        stmt = stmt.bindparams(faculty=faculty)
        stmt = stmt.bindparams(educationyear=educationyear)
        stmt = stmt.bindparams(startdate=start_date.strftime("%Y-%m-%d"))
        stmt = stmt.bindparams(enddate=end_date.strftime("%Y-%m-%d"))

        query = conn.execute(stmt)

        sqlargs = {}
        where = ""
        if gid != 0:
            where += "and g.gid = :gid"
            sqlargs["gid"] = gid

        stmt = text(
            """
            select lessondate::date, period_name as p 
            from tmp_schedule_period s 
            join groupname g on s.group_id=g.gid 
            group by lessondate, period_name 
            order by lessondate, period_name
            """.format(where=where)
        )
        query = conn.execute(stmt, sqlargs)
        scheduleList = {}
        for row in query:
            if str(row.lessondate) in scheduleList:
                scheduleList[str(row.lessondate)][row.p] = []
            else:
                scheduleList[str(row.lessondate)] = {row.p: []}

        sql = """
            select s.*, s.lessondate:: date as lessondate2, s.groupname as groupname,ey.name educationyear, faculty.faculty,
            c.course_index, c.title course_title, mc.name meta_course_name, c.alias course_alias,
            o.title_index subject_index, o.title subject_title, 
            pt.alias lessontype_alias, pt.typename lessontype_name,
            get_peoples_name(sh.teacher_mid, ' ') teachers,
            sh.teacher_mid as teacher_mids,
            get_rooms_short_name(sh.rid, ' ') rooms,
            g.idcathedra,
            s.cathedra
            from tmp_schedule_period s 
            left join nnz_schedule sh on sh.sheid = s.schedule_id 
            left join courses c on sh.cid = c.cid
            left join meta_course mc on mc.id = c.meta_course
            left join organizations o on o.oid = sh.f_organizations        
            left join eventtools pt ON sh.pair_type_id = pt.typeid  
            join groupname g on s.group_id=g.gid 
            left join cathedras on cathedras.idcathedra = g.idcathedra
            left join faculty  on faculty.idfaculty = coalesce(g.idfaculty, cathedras.faculty)
            left join educationyears ey on ey.number = g.ideducationlevel
            where True {where}
            order by  s.lessondate, g.name
        """.format(where=where)

        stmt = text(sql)
        query = conn.execute(stmt, sqlargs)
        for row in query:
            scheduleList[str(row.lessondate2)][row.period_name].append(
                [dict(zip(tuple(query.keys()), row))]
            )
            for s in scheduleList[str(row.lessondate2)][row.period_name]:
                backup = s[0]["cell"]
                workcopy = s[0]["cell"]
                s[0]["cellraw"] = backup
                if backup is not None:
                    s[0]["cell"] = (
                        str(workcopy)
                        .replace("<B>", "")
                        .replace("</B>", "")
                        .replace("\r\n", " ")
                    )
                else:
                    s[0]["cell"] = ""

        return scheduleList

    def getScheduleTeacher(self, mid, weekid):
        conn = current_app.ms.db(self.lname).connect()

        stmt = text(
            """
            select nnz_fill_schedule_teachers(:mid,:weekid);
        
            select 
                CONCAT(CAST(date_part('day' , lesson_date) AS CHAR(2)), ' ',xp_monthname_genitive(cast(date_part('month', lesson_date) as integer)), ', ', xp_WeekDays(ScheduleTeachersT.day_of_week))  xDate, 
                ScheduleTeachersT.*, 
                et.alias, 
                o.title_index, 
                sh.lesson_num, 
                CASE 
                    WHEN COALESCE(et.alias, '') <> '' AND COALESCE(o.title_index, '') <> '' AND sh.lesson_num IS NOT NULL 
                        THEN CONCAT(et.alias, ' / ', o.title_index, '.', sh.lesson_num)
                    WHEN COALESCE(et.alias, '') <> '' AND COALESCE(o.title_index, '') <> ''
                        THEN CONCAT(et.alias, ' / ', o.title_index)
                    WHEN COALESCE(et.alias, '') <> ''
                        THEN et.alias
                    ELSE ''
                END AS lesson_format,
            CONCAT_WS(' - ', CONCAT_WS('.', NULLIF(o.title_index, ''), ScheduleTeachersT.lesson_num), o.title)::varchar(254) as theme_name
            from ScheduleTeachersT 
            left join nnz_schedule sh on sh.sheid = ScheduleTeachersT.sheid
            left join eventtools et on et.typeid = sh.pair_type_id
            left join organizations o on o.oid = sh.f_organizations
            left join course_tema_detail ctd on ctd.id = sh.course_tema_detail_id
            order by day_of_week, starttime, part, gid1;
            """
        )

        stmt = stmt.bindparams(mid=mid)
        stmt = stmt.bindparams(weekid=weekid)
        query = conn.execute(stmt)

        scheduleList = {
            "teacherSchedule": [dict(zip(tuple(query.keys()), i)) for i in query.cursor]
        }

        return scheduleList

    def getScheduleRoom(self, roomid, scheduledate):
        conn = current_app.ms.db(self.lname).connect()

        stmt = text(
            "select week_id from nnz_weeks where :scheduledate between wstart_date and wend_date"
        )
        stmt = stmt.bindparams(scheduledate=scheduledate)
        query = conn.execute(stmt)

        weekid = None
        for row in query:
            weekid = row.week_id

        if weekid is None:
            return None

        stmt = text(
            """
        select nnz_fill_schedule_print(:weekid);
        select periodname, periodtime, 
        c.cid, c.title coursename, 
        g.gid, g.name groupname,
        r.rid, r.name as roomname,
        r.short_name as short_name,
        '' subject,
        string_to_array(teachermids, ',') teachers
        from ScheduleTP
        left outer join courses c on sh_cid = c.cid
        left outer join groupname g on sgid = g.gid
        left outer join rooms r on r.rid = :roomid
        where cast(:roomid  as varchar(10)) = ANY(string_to_array(roomids, ',')) and day_of_week = EXTRACT(DOW FROM cast(:scheduledate as date ))
        order by starttime;
        """
        )
        stmt = stmt.bindparams(scheduledate=scheduledate)
        stmt = stmt.bindparams(roomid=roomid)
        stmt = stmt.bindparams(weekid=weekid)
        query = conn.execute(stmt)

        scheduleList = {
            "roomschedule": [dict(zip(tuple(query.keys()), i)) for i in query.cursor]
        }

        for a in scheduleList["roomschedule"]:
            newTeachers = []
            for b in a["teachers"]:
                teacher = User(self.lname)
                # teacher = teacher.get_user_from_sql_id(b)
                # if teacher is not None:
                #    newTeachers.append(teacher.serialize())

                newTeachers.append(teacher.get_user_from_sql_detailed(b))

            a["teachers"] = newTeachers

        return scheduleList

    def get_schedule_teacher_vypiska_per_semester(self, mid, trmid):
        # select xp_fill_report_vypiska(Преподаватель, Семестр, Месяц (-1 - отчет по неделе, 0 - отчет по семестру), начало-периода, конец-периода);
        conn = current_app.ms.db(self.lname).connect()

        stmt = text(
            """
            select xp_fill_report_vypiska(:mid, :trmid, 0, null, null);
            
            SELECT 
                l.wd, 
				l.sheid,
                l.periodname, 
                l.groupname,
                l.weekstart || E' ' || 
                substring(
                    xp_monthname(
                        CASE 
                            WHEN (l.d - l.weekstart + 1 <> l.wd) OR (l.m <> l.weekmonth) 
                            THEN COALESCE(NULLIF((l.weekmonth - 1) % 12, 0), 12) 
                            ELSE l.weekmonth 
                        END
                    ) 
                    FROM 1 FOR 3
                ) || '.' AS weekstart,
                'нед.' || E' ' || '№ ' || l.week::text AS week,
                INITCAP(xp_monthname(l.weekmonth)) AS monthname,
                INITCAP(xp_weekdays(l.wd)) AS WeekDayName,
				et.alias, 
                o.title_index, 
                sh.lesson_num, 
                CASE 
                    WHEN COALESCE(et.alias, '') <> '' AND COALESCE(o.title_index, '') <> '' AND sh.lesson_num IS NOT NULL 
                        THEN CONCAT(et.alias, ' / ', o.title_index, '.', sh.lesson_num)
                    WHEN COALESCE(et.alias, '') <> '' AND COALESCE(o.title_index, '') <> ''
                        THEN CONCAT(et.alias, ' / ', o.title_index)
                    WHEN COALESCE(et.alias, '') <> ''
                        THEN et.alias
                    ELSE ''
                END AS lesson_format,
                rooms.rid AS rid,
                string_agg(rooms.short_name, '/') roomname,
                c.cid AS cid,
                string_agg((case when coalesce(c.alias, '') <> '' then c.alias when m.shortname<>'' then m.shortname else xp_Subject_Without_Year(c.title) end), '/') coursename
            FROM 
                tmp_teacher_load l 
			left join nnz_schedule sh on sh.sheid = l.sheid
            left join eventtools et on et.typeid = sh.pair_type_id
            left join organizations o on o.oid = sh.f_organizations
            left join course_tema_detail ctd on ctd.id = sh.course_tema_detail_id
            left outer join rooms on (rooms.rid = ANY(sh.rid))
            left outer join courses c on (sh.cid = c.cid)
            left outer join meta_course m on m.id=c.meta_course
            WHERE 
                -- Условие для вывода всех значений за Воскресенье, если есть хотя бы одна запись с groupname
                (
                    INITCAP(xp_weekdays(l.wd)) <> 'Воскресенье'
                    OR EXISTS (
                        SELECT 1 
                        FROM tmp_teacher_load sub_l 
                        WHERE 
                            INITCAP(xp_weekdays(sub_l.wd)) = 'Воскресенье' 
                            AND sub_l.groupname IS NOT NULL
                            AND sub_l.full_date = l.full_date
                    )
                )
            GROUP BY l.wd, l.sheid, l.periodname, l.groupname, et.alias, o.title_index, sh.lesson_num, l.weekstart, l.week, l.weekmonth, l.d, l.m, l.week, lesson_format, rooms.rid, rooms.short_name, l.full_date, l.starttime, c.cid, c.alias, m.shortname, c.title
            ORDER BY 
                full_date, wd, starttime;
            """
        )

        stmt_info = text(
            """
            select xp_fill_report_vypiska(:mid, :trmid, 0, null, null);
            SELECT * FROM tmp_vypiska_totals;
            """
        )

        stmt = stmt.bindparams(mid=mid)
        stmt = stmt.bindparams(trmid=trmid)
        query = conn.execute(stmt)
        schedule_list = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

        stmt_info = stmt_info.bindparams(mid=mid)
        stmt_info = stmt_info.bindparams(trmid=trmid)
        query_info = conn.execute(stmt_info)
        schedule_info = [
            dict(zip(tuple(query_info.keys()), i)) for i in query_info.cursor
        ]
        return schedule_list, schedule_info

    def get_schedule_teacher_vypiska_per_week(self, mid, week_id):
        conn = current_app.ms.db(self.lname).connect()

        # Получаем Term, wstart_date и wend_date
        term_stmt = text("SELECT trmid FROM term_weeks WHERE week_id = :week_id ORDER BY trmid desc")
        date_stmt = text(
            "SELECT wstart_date, wend_date FROM nnz_weeks WHERE week_id = :week_id"
        )

        term_result = conn.execute(term_stmt, week_id=week_id).fetchone()
        date_result = conn.execute(date_stmt, week_id=week_id).fetchone()

        if not term_result or not date_result:
            return []

        stmt = text(
            """
            select xp_fill_report_vypiska(:mid, :trmid, 0, :wstart_date, :wend_date);

            SELECT 
                l.wd, 
                l.periodname, 
                l.sheid, 
                l.groupname,
                l.weekstart || E' ' || 
                substring(
                    xp_monthname(
                        CASE 
                            WHEN (l.d - l.weekstart + 1 <> l.wd) OR (l.m <> l.weekmonth) 
                            THEN COALESCE(NULLIF((l.weekmonth - 1) % 12, 0), 12) 
                            ELSE l.weekmonth 
                        END
                    ) 
                    FROM 1 FOR 3
                ) || '.' AS weekstart,
                'нед.' || E' ' || '№ ' || l.week::text AS week,
                INITCAP(xp_monthname(l.weekmonth)) AS monthname,
                INITCAP(xp_weekdays(l.wd)) AS WeekDayName,
				et.alias, 
                o.title_index, 
                sh.lesson_num, 
                CASE 
                    WHEN COALESCE(et.alias, '') <> '' AND COALESCE(o.title_index, '') <> '' AND sh.lesson_num IS NOT NULL 
                        THEN CONCAT(et.alias, ' / ', o.title_index, '.', sh.lesson_num)
                    WHEN COALESCE(et.alias, '') <> '' AND COALESCE(o.title_index, '') <> ''
                        THEN CONCAT(et.alias, ' / ', o.title_index)
                    WHEN COALESCE(et.alias, '') <> ''
                        THEN et.alias
                    ELSE ''
                END AS lesson_format,
                rooms.rid AS rid,
                string_agg(rooms.short_name, '/') roomname,
                c.cid AS cid,
                string_agg((case when coalesce(c.alias, '') <> '' then c.alias when m.shortname<>'' then m.shortname else xp_Subject_Without_Year(c.title) end), '/') coursename
            FROM 
                tmp_teacher_load l 
            left join nnz_schedule sh on sh.sheid = l.sheid
            left join eventtools et on et.typeid = sh.pair_type_id
            left join organizations o on o.oid = sh.f_organizations
            left join course_tema_detail ctd on ctd.id = sh.course_tema_detail_id
            left outer join rooms on (rooms.rid = ANY(sh.rid))
            left outer join courses c on (sh.cid = c.cid)
            left outer join meta_course m on m.id=c.meta_course
            GROUP BY l.wd, l.sheid, l.periodname, l.groupname, et.alias, o.title_index, sh.lesson_num, l.weekstart, l.week, l.weekmonth, l.d, l.m, l.week, lesson_format, rooms.rid, rooms.short_name, l.full_date, l.starttime, c.cid, c.alias, m.shortname, c.title
            ORDER BY 
                full_date, wd, starttime;
            """
        )

        stmt_info = text(
            """
            select xp_fill_report_vypiska(:mid, :trmid, -1, :wstart_date, :wend_date);
            SELECT * FROM tmp_vypiska_totals;
            """
        )

        stmt = stmt.bindparams(
            mid=mid,
            trmid=term_result.trmid,
            wstart_date=date_result.wstart_date,
            wend_date=date_result.wend_date,
        )
        query = conn.execute(stmt)
        schedule_list = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

        stmt_info = stmt_info.bindparams(
            mid=mid,
            trmid=term_result.trmid,
            wstart_date=date_result.wstart_date,
            wend_date=date_result.wend_date,
        )
        query_info = conn.execute(stmt_info)
        schedule_info = [
            dict(zip(tuple(query_info.keys()), i)) for i in query_info.cursor
        ]
        return schedule_list, schedule_info
