import shutil
from urllib.parse import urlparse

from flask import jsonify, request, current_app, send_from_directory, g
from flask import Blueprint

from LMSAPI.api.Models.Homework import Homework
from LMSAPI.api.Models.UserActionLog import UserActionLog
from LMSAPI.api.Views.TokenAPI import auth
from LMSAPI.api.Models.File import File
from flask_cors import CORS
import os

homework_api = Blueprint("homework_api", __name__)

CORS(homework_api)


@homework_api.route(
    "/lms/api/v1.0/<lname>/homework/user/<int:id>/<int:sheid>/<int:instance>",
    methods=["GET"],
)
@auth.login_required
def getHomework(lname, id, sheid, instance):
    if g.user.mid != id and g.user.isTeacher != True:
        return jsonify(success=False), 403

    result = File(lname).getHomeworkFiles(id, sheid, instance)
    if result is None:
        return jsonify(success=False), 404
    path = File(lname).getHomeworkDirectory(id, sheid, instance)
    result = File.check_for_links(path, result)
    return jsonify(result)


@homework_api.route(
    "/lms/api/v1.0/<lname>/homework/user/<int:userId>/<int:sheid>/<int:instance>/<filename>",
    methods=["DELETE"],
)
@auth.login_required
def deleteHomeworkStudentFile(lname, userId, sheid, instance, filename):
    if g.user.mid != userId and g.user.isTeacher != True:
        return jsonify(success=False), 403

    try:
        os.remove(
            os.path.join(
                File(lname).getHomeworkDirectory(userId, sheid, instance), filename
            )
        )
        return jsonify(success=True)

    except:
        current_app.logger.error("getUserFile(" + str(userId) + "," + filename + ")")
        return jsonify(success=False), 404


@homework_api.route(
    "/lms/api/v1.0/<lname>/homework/user/<int:mid>/<int:sheid>/<int:instance>",
    methods=["POST"],
)
@auth.login_required
def postHomeworkStudent(lname, mid, sheid, instance):
    if g.user.isTeacher == False:
        return jsonify(success=False), 403

    ip_address = (
        request.headers.get("X-Forwarded-For")
        or request.headers.get("X-Real-IP")
        or request.environ.get("REMOTE_ADDR")
    )

    submitted_file = request.files["file"]
    if submitted_file:
        File(lname).createHomeworkStudentFileDirectory(mid, sheid, instance)
        file_path = File(lname).getHomeworkDirectory(mid, sheid, instance)
        return File.save_file_to_path(
            lname=lname,
            submitted_file=submitted_file,
            file_path=file_path,
            action_source=request.environ.get("HTTP_USER_AGENT"),
            login=g.user.id,
            user_mid=g.user.mid,
            ip_address=ip_address,
        )
    return jsonify({"success": False, "error": "upload file with key file"})


@homework_api.route(
    "/lms/api/v1.0/<lname>/homework/user/<int:userId>/<int:sheid>/<int:instance>/<filename>",
    methods=["GET"],
)
@auth.login_required
def getHomeworkFile(lname, userId, sheid, instance, filename):
    if g.user.mid != userId and g.user.isTeacher != True:
        return jsonify(success=False), 403
    file_path = File(lname).getHomeworkDirectory(userId, sheid, instance)
    ip_address = (
        request.headers.get("X-Forwarded-For")
        or request.headers.get("X-Real-IP")
        or request.environ.get("REMOTE_ADDR")
    )
    return File.send_from_directory(
        lname=lname,
        file_path=file_path,
        file_name=filename,
        action_source=request.environ.get("HTTP_USER_AGENT"),
        login=g.user.id,
        user_mid=g.user.mid,
        ip_address=ip_address,
    )


# загрузить на сервер результат индивидуального д/з
@homework_api.route(
    "/lms/api/v1.0/<lname>/homework/userresult/<int:mid>/<int:sheid>/<int:instance>",
    methods=["POST"],
)
@auth.login_required
def postHomeworkResult(lname, mid, sheid, instance):
    ip_address = (
        request.headers.get("X-Forwarded-For")
        or request.headers.get("X-Real-IP")
        or request.environ.get("REMOTE_ADDR")
    )

    submitted_file = request.files["file"]
    if submitted_file:
        File(lname).createHomeworkResultFileDirectory(mid, sheid, instance)
        file_path = File(lname).getHomeworkResultDirectory(mid, sheid, instance)
        return File.save_file_to_path(
            lname=lname,
            submitted_file=submitted_file,
            file_path=file_path,
            action_source=request.environ.get("HTTP_USER_AGENT"),
            login=g.user.id,
            user_mid=g.user.mid,
            ip_address=ip_address,
        )

    return jsonify({"success": False, "error": "upload file with key file"})


@homework_api.route(
    "/lms/api/v1.0/<lname>/homework/userresult/<int:userId>/<int:sheid>/<int:instance>/<filename>",
    methods=["DELETE"],
)
@auth.login_required
def deleteHomeworkResultFile(lname, userId, sheid, instance, filename):
    try:
        os.remove(
            os.path.join(
                File(lname).getHomeworkResultDirectory(userId, sheid, instance),
                filename,
            )
        )
        return jsonify(success=True)

    except:
        current_app.logger.error("getUserFile(" + str(userId) + "," + filename + ")")
        return jsonify(success=False), 404


@homework_api.route(
    "/lms/api/v1.0/<lname>/homework/userresult/<int:id>/<int:sheid>/<int:instance>",
    methods=["GET"],
)
@auth.login_required
def getHomeworkResult(lname, id, sheid, instance):
    result = File(lname).getHomeworkResultFiles(id, sheid, instance)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@homework_api.route(
    "/lms/api/v1.0/<lname>/homework/userresult/<int:userId>/<int:sheid>/<int:instance>/<filename>",
    methods=["GET"],
)
@auth.login_required
def getHomeworkResultFile(lname, userId, sheid, instance, filename):
    file_path = File(lname).getHomeworkResultDirectory(userId, sheid, instance)
    ip_address = (
        request.headers.get("X-Forwarded-For")
        or request.headers.get("X-Real-IP")
        or request.environ.get("REMOTE_ADDR")
    )
    return File.send_from_directory(
        lname=lname,
        file_path=file_path,
        file_name=filename,
        action_source=request.environ.get("HTTP_USER_AGENT"),
        login=g.user.id,
        user_mid=g.user.mid,
        ip_address=ip_address,
    )


@homework_api.route(
    "/lms/api/v1.0/<lname>/homework/group/<int:sheid>/<int:instance>", methods=["GET"]
)
@auth.login_required
def getGroupHomework(lname, sheid, instance):
    group_homework = File(lname).getGroupHomeworkFiles(sheid, instance)
    if group_homework is None:
        return jsonify(success=False), 404

    group_homework_path = File(lname).getGroupHomeworkDirectory(sheid, instance)
    group_homework_result = File.check_for_links_in_files(
        group_homework_path, group_homework
    )

    journal_lesson_detail = File(lname).get_journal_lesson_detail_files(sheid, instance)
    if journal_lesson_detail is None:
        return jsonify(success=False), 404

    journal_lesson_detail_path = File(lname).get_journal_lesson_detail_directory(
        sheid, instance
    )
    journal_lesson_detail_result = File.check_for_links_in_files(
        journal_lesson_detail_path, journal_lesson_detail
    )
    return jsonify(
        {"teacher": journal_lesson_detail_result, "students": group_homework_result}
    )


@homework_api.route(
    "/lms/api/v1.0/<lname>/homework/group/<int:sheid>/<int:instance>/<filename>",
    methods=["GET"],
)
@auth.login_required
def getGroupHomeworkFile(lname, sheid, instance, filename):
    file_path = File(lname).getGroupHomeworkDirectory(sheid, instance)
    ip_address = (
        request.headers.get("X-Forwarded-For")
        or request.headers.get("X-Real-IP")
        or request.environ.get("REMOTE_ADDR")
    )
    return File.send_from_directory(
        lname=lname,
        file_path=file_path,
        file_name=filename,
        action_source=request.environ.get("HTTP_USER_AGENT"),
        login=g.user.id,
        user_mid=g.user.mid,
        ip_address=ip_address,
    )


@homework_api.route(
    "/lms/api/v1.0/<lname>/homework/group/link/<int:sheid>/<int:instance>",
    methods=["POST"],
)
@auth.login_required
def postGroupHomeworkReference(lname, sheid, instance):
    if g.user.isTeacher == False:
        return jsonify(success=False), 403

    submitted_link = request.form.get("link")  # Получаем ссылку из запроса
    if submitted_link:
        File(lname).createGroupHomeworkDirectory(sheid, instance)
        file_path = File(lname).getGroupHomeworkDirectory(sheid, instance)

        # Извлекаем имя файла из ссылки
        parsed_url = urlparse(submitted_link)
        file_name = os.path.basename(parsed_url.path) + ".url"

        # Создаем файл с расширением .url и записываем в него ссылку
        file_path = os.path.join(file_path, file_name)
        with open(file_path, "w") as f:
            f.write("[InternetShortcut]\n")
            f.write("URL=" + submitted_link + "\n")

        return jsonify(
            success=True,
            message="Link saved successfully as '{file_name}'".format(
                file_name=file_name
            ),
        )

    return jsonify({"success": False, "error": "link not provided in the request"})


@homework_api.route(
    "/lms/api/v1.0/<lname>/homework/group/<int:sheid>/<int:instance>", methods=["POST"]
)
@auth.login_required
def postGroupHomework(lname, sheid, instance):
    if g.user.isTeacher == False:
        return jsonify(success=False), 403

    ip_address = (
        request.headers.get("X-Forwarded-For")
        or request.headers.get("X-Real-IP")
        or request.environ.get("REMOTE_ADDR")
    )

    submitted_file = request.files["file"]
    if submitted_file:
        File(lname).createGroupHomeworkDirectory(sheid, instance)
        file_path = File(lname).getGroupHomeworkDirectory(sheid, instance)
        return File.save_file_to_path(
            lname=lname,
            submitted_file=submitted_file,
            file_path=file_path,
            action_source=request.environ.get("HTTP_USER_AGENT"),
            login=g.user.id,
            user_mid=g.user.mid,
            ip_address=ip_address,
        )
    return jsonify({"success": False, "error": "upload file with key file"})


@homework_api.route(
    "/lms/api/v1.0/<lname>/homework/group/<int:sheid>/<int:instance>/<filename>",
    methods=["DELETE"],
)
@auth.login_required
def deleteGroupHomeworkFile(lname, sheid, instance, filename):
    try:
        os.remove(
            os.path.join(
                File(lname).getGroupHomeworkDirectory(sheid, instance), filename
            )
        )
        return jsonify(success=True)
    except:
        current_app.logger.error("getUserFile(" + filename + ")")
        return jsonify(success=False), 404


# загрузить на сервер результат группового д/з
@homework_api.route(
    "/lms/api/v1.0/<lname>/homework/groupresult/<int:id>/<int:sheid>/<int:instance>",
    methods=["POST"],
)
@auth.login_required
def postGroupHomeworkResult(lname, id, sheid, instance):
    ip_address = (
        request.headers.get("X-Forwarded-For")
        or request.headers.get("X-Real-IP")
        or request.environ.get("REMOTE_ADDR")
    )

    submitted_file = request.files["file"]
    if submitted_file:
        File(lname).createGroupHomeworkResultFileDirectory(id, sheid, instance)
        file_path = File(lname).getGroupHomeworkResultDirectory(id, sheid, instance)
        return File.save_file_to_path(
            lname=lname,
            submitted_file=submitted_file,
            file_path=file_path,
            action_source=request.environ.get("HTTP_USER_AGENT"),
            login=g.user.id,
            user_mid=g.user.mid,
            ip_address=ip_address,
        )

    return jsonify({"success": False, "error": "upload file with key file"})


@homework_api.route(
    "/lms/api/v1.0/<lname>/homework/groupresult/<int:id>/<int:sheid>/<int:instance>/<filename>",
    methods=["DELETE"],
)
@auth.login_required
def deleteGroupHomeworkResultFile(lname, id, sheid, instance, filename):
    try:
        os.remove(
            os.path.join(
                File(lname).getGroupHomeworkResultDirectory(id, sheid, instance),
                filename,
            )
        )
        return jsonify(success=True)
    except:
        current_app.logger.error("getUserFile(" + filename + ")")
        return jsonify(success=False), 404


@homework_api.route(
    "/lms/api/v1.0/<lname>/homework/groupresult/<int:id>/<int:sheid>/<int:instance>",
    methods=["GET"],
)
@auth.login_required
def getGroupHomeworkResult(lname, id, sheid, instance):
    result = File(lname).getGroupHomeworkResultFiles(id, sheid, instance)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@homework_api.route(
    "/lms/api/v1.0/<lname>/homework/groupresult/<int:userId>/<int:sheid>/<int:instance>/<filename>",
    methods=["GET"],
)
@auth.login_required
def getGroupHomeworkResultFile(lname, userId, sheid, instance, filename):
    file_path = File(lname).getGroupHomeworkResultDirectory(userId, sheid, instance)
    ip_address = (
        request.headers.get("X-Forwarded-For")
        or request.headers.get("X-Real-IP")
        or request.environ.get("REMOTE_ADDR")
    )
    return File.send_from_directory(
        lname=lname,
        file_path=file_path,
        file_name=filename,
        action_source=request.environ.get("HTTP_USER_AGENT"),
        login=g.user.id,
        user_mid=g.user.mid,
        ip_address=ip_address,
    )


@homework_api.route(
    "/lms/api/v1.0/<lname>/homework/<int:trmid>/<int:sheid>/<int:week_id>/<int:gid>",
    methods=["GET"],
)
@auth.login_required
def get_home_work(lname, trmid, sheid, week_id, gid):
    result = {
        "group_homework": Homework().get_home_work(lname, sheid, week_id, gid),
        "student_homework": Homework().get_home_work_student(
            lname, trmid, sheid, week_id, gid
        ),
    }
    if result is None:
        return jsonify(success=False), 404
    return jsonify(result)


@homework_api.route(
    "/lms/api/v1.0/<lname>/homework",
    methods=["POST"],
)
@auth.login_required
def create_home_work(lname):
    data = request.json

    if not data:
        return jsonify({"error": "No data provided"}), 400

    # Проверяем наличие обязательных полей
    required_fields = ["sheid", "week_id", "gid"]
    missing_fields = [field for field in required_fields if field not in data]

    if missing_fields:
        return (
            jsonify(
                {
                    "error": "Missing required fields: {missing_fields}".format(
                        missing_fields=", ".join(missing_fields)
                    )
                }
            ),
            400,
        )

    result = Homework().create_home_work(lname, data)

    if result is False:
        return jsonify(success=False), 404

    return jsonify(success=True), 200


@homework_api.route(
    "/lms/api/v1.0/<lname>/homework/<int:hwid>",
    methods=["DELETE"],
)
@auth.login_required
def delete_home_work(lname, hwid):
    result = Homework().delete_home_work(lname, hwid)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(success=True), 200


@homework_api.route(
    "/lms/api/v1.0/<lname>/homework/student",
    methods=["POST"],
)
@auth.login_required
def create_homework_student(lname):
    data = request.json

    if not data:
        return jsonify({"error": "No data provided"}), 400

    # Проверяем наличие обязательных полей
    required_fields = ["student_mid", "hwid"]
    missing_fields = [field for field in required_fields if field not in data]

    if missing_fields:
        return (
            jsonify(
                {
                    "error": "Missing required fields: {missing_fields}".format(
                        missing_fields=", ".join(missing_fields)
                    )
                }
            ),
            400,
        )

    result = Homework().create_homework_student(lname, data)

    if result is False:
        return jsonify(success=False), 404

    return jsonify(success=True), 200


@homework_api.route(
    "/lms/api/v1.0/<lname>/homework/student/<int:indhwid>",
    methods=["DELETE"],
)
@auth.login_required
def delete_homework_student(lname, indhwid):
    result = Homework().delete_homework_student(lname, indhwid)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(success=True), 200


@homework_api.route(
    "/lms/api/v1.0/<lname>/homework/<int:hwid>/questionset",
    methods=["GET"],
)
@auth.login_required
def get_questionset_by_home_work(lname, hwid):
    result = Homework().get_questionset_by_home_work(lname, hwid)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@homework_api.route(
    "/lms/api/v1.0/<lname>/homework/student/<int:indhwid>/questionset",
    methods=["GET"],
)
@auth.login_required
def get_questionset_by_home_work_student(lname, indhwid):
    result = Homework().get_questionset_by_home_work_student(lname, indhwid)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@homework_api.route(
    "/lms/api/v1.0/<lname>/homework/fact/<int:sheid>/<int:instance>", methods=["GET"]
)
@auth.login_required
def get_homework_fact(lname, sheid, instance):
    result = File(lname).get_homework_fact_files(sheid, instance)
    if result is None:
        return jsonify(success=False), 404
    return jsonify(result)


@homework_api.route(
    "/lms/api/v1.0/<lname>/homework/fact/<int:sheid>/<int:instance>/<filename>",
    methods=["GET"],
)
@auth.login_required
def get_homework_fact_file(lname, sheid, instance, filename):
    file_path = File(lname).get_homework_fact_directory(sheid, instance)
    ip_address = (
        request.headers.get("X-Forwarded-For")
        or request.headers.get("X-Real-IP")
        or request.environ.get("REMOTE_ADDR")
    )
    return File.send_from_directory(
        lname=lname,
        file_path=file_path,
        file_name=filename,
        action_source=request.environ.get("HTTP_USER_AGENT"),
        login=g.user.id,
        user_mid=g.user.mid,
        ip_address=ip_address,
    )


@homework_api.route(
    "/lms/api/v1.0/<lname>/homework/fact/<int:sheid>/<int:instance>", methods=["POST"]
)
@auth.login_required
def post_homework_fact(lname, sheid, instance):
    if g.user.isTeacher == False:
        return jsonify(success=False), 403

    ip_address = (
        request.headers.get("X-Forwarded-For")
        or request.headers.get("X-Real-IP")
        or request.environ.get("REMOTE_ADDR")
    )

    submitted_file = request.files["file"]
    if submitted_file:
        File(lname).create_homework_fact_directory(sheid, instance)
        file_path = File(lname).get_homework_fact_directory(sheid, instance)
        return File.save_file_to_path(
            lname=lname,
            submitted_file=submitted_file,
            file_path=file_path,
            action_source=request.environ.get("HTTP_USER_AGENT"),
            login=g.user.id,
            user_mid=g.user.mid,
            ip_address=ip_address,
        )
    return jsonify({"success": False, "error": "upload file with key file"})


@homework_api.route(
    "/lms/api/v1.0/<lname>/homework/fact/<int:sheid>/<int:instance>/<filename>",
    methods=["DELETE"],
)
@auth.login_required
def delete_homework_fact_file(lname, sheid, instance, filename):
    try:
        os.remove(
            os.path.join(
                File(lname).get_homework_fact_directory(sheid, instance), filename
            )
        )
        return jsonify(success=True)
    except:
        current_app.logger.error("getUserFile(" + filename + ")")
        return jsonify(success=False), 404


@homework_api.route(
    "/lms/api/v1.0/<lname>/journal/homework/fact/<int:sheid>/<int:instance>",
    methods=["GET"],
)
@auth.login_required
def get_journal_homework_fact(lname, sheid, instance):
    result = []
    lesson_check = Homework().get_lesson_check(lname, sheid, instance)
    if lesson_check:
        result = File(lname).get_homework_fact_files(sheid, instance)

    group_homework_files = File(lname).getGroupHomeworkFiles(sheid, instance)
    path = File(lname).getGroupHomeworkDirectory(sheid, instance)
    group_homework = File.check_for_links_in_files(path, group_homework_files)
    for homework in group_homework:
        result.append(homework)

    if result is None:
        return jsonify(success=False), 404
    return jsonify(result)


@homework_api.route(
    "/lms/api/v1.0/<lname>/journal/homework/fact/<int:sheid>/<int:instance>/<filename>",
    methods=["GET"],
)
@auth.login_required
def get_journal_homework_fact_file(lname, sheid, instance, filename):
    ip_address = (
        request.headers.get("X-Forwarded-For")
        or request.headers.get("X-Real-IP")
        or request.environ.get("REMOTE_ADDR")
    )
    if File(lname).homework_fact_file_by_filename_exists(sheid, instance, filename):
        file_path = File(lname).get_homework_fact_directory(sheid, instance)
    else:
        file_path = File(lname).getGroupHomeworkDirectory(sheid, instance)

    return File.send_from_directory(
        lname=lname,
        file_path=file_path,
        file_name=filename,
        action_source=request.environ.get("HTTP_USER_AGENT"),
        login=g.user.id,
        user_mid=g.user.mid,
        ip_address=ip_address,
    )


@homework_api.route(
    "/lms/api/v1.0/<lname>/journal/student/homework/fact/<int:sheid>/<int:instance>",
    methods=["GET"],
)
@auth.login_required
def get_journal_student_homework_fact(lname, sheid, instance):
    result = []
    lesson_check = Homework().get_lesson_check(lname, sheid, instance)
    if lesson_check:
        result = File(lname).get_homework_student_fact_files(
            g.user.mid, sheid, instance
        )
    student_homework_files = File(lname).getHomeworkFiles(g.user.mid, sheid, instance)
    path = File(lname).getHomeworkDirectory(g.user.mid, sheid, instance)
    student_homework = File.check_for_links(path, student_homework_files)
    for homework in student_homework:
        result.append(homework)

    if result is None:
        return jsonify(success=False), 404
    return jsonify(result)


@homework_api.route(
    "/lms/api/v1.0/<lname>/journal/student/homework/fact/<int:sheid>/<int:instance>/<filename>",
    methods=["GET"],
)
@auth.login_required
def get_journal_student_homework_fact_file(lname, sheid, instance, filename):
    ip_address = (
        request.headers.get("X-Forwarded-For")
        or request.headers.get("X-Real-IP")
        or request.environ.get("REMOTE_ADDR")
    )

    if File(lname).homework_student_fact_file_by_filename_exists(g.user.mid, sheid, instance, filename):
        file_path = File(lname).get_homework_student_fact_directory(
            g.user.mid, sheid, instance
        )
    else:
        file_path = File(lname).getHomeworkDirectory(g.user.mid, sheid, instance)

    return File.send_from_directory(
        lname=lname,
        file_path=file_path,
        file_name=filename,
        action_source=request.environ.get("HTTP_USER_AGENT"),
        login=g.user.id,
        user_mid=g.user.mid,
        ip_address=ip_address,
    )


@homework_api.route(
    "/lms/api/v1.0/<lname>/journal/student/homework/info/<int:termid>/<int:cid>",
    methods=["GET"],
)
@auth.login_required
def get_journal_student_homework_info(lname, termid: int, cid: int):
    result = Homework().get_homwork_info(lname, termid, g.user.mid, cid)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)
