import json
import shutil
import tempfile
import threading
import uuid
from collections import OrderedDict, defaultdict
from decimal import Decimal
from typing import List, Dict, Any

from flask import send_file, Response

from flask import Flask, jsonify, request, current_app, g
from flask import Blueprint
from sqlalchemy import text

from LMSAPI.api.Models.Company import Company
from LMSAPI.api.Models.File import File
from LMSAPI.api.Models.Group import Group
from flask_cors import CORS

from LMSAPI.api.Models.Rating import Rating
from LMSAPI.api.Models.ScholarshipCandidatesFilesStorage import ScholarshipCandidatesFilesStorage
from LMSAPI.api.Models.ScholarshipListOptions import (
    ScholarshipListOptions,
    DegreeCandidatesTask,
)
from LMSAPI.api.Views.TokenAPI import auth

scholarship = Blueprint("scholarship", __name__, url_prefix="/lms/api/v1.0/<lname>")

CORS(scholarship)


def degree_candidates_task(app, lname, task_id, xp_key):
    with app.app_context():
        try:
            result = {}

            options = ScholarshipListOptions(
                lname
            ).get_education_level_and_prep_struc_category_id_option(
                lname=lname,
                xp_key=xp_key,
            )

            for option in options:
                if (
                    option["education_level"] == 1
                    and option["prep_struc_category_id"] == 1
                ):
                    result[
                        "scholarship_higher_education_cadet"
                    ] = ScholarshipListOptions(lname).get_scholarship_list(
                        lname, xp_key, 1, 1, (1, 2, 4, 5)
                    )
                elif (
                    option["education_level"] == 1
                    and option["prep_struc_category_id"] == 2
                ):
                    result[
                        "scholarship_higher_education_listener"
                    ] = ScholarshipListOptions(lname).get_scholarship_list(
                        lname, xp_key, 1, 2, (1, 2, 4, 5)
                    )
                elif (
                    option["education_level"] == 1
                    and option["prep_struc_category_id"] == 3
                ):
                    result[
                        "scholarship_higher_education_adjunct"
                    ] = ScholarshipListOptions(lname).get_scholarship_list(
                        lname, xp_key, 1, 3, (1, 2, 4, 5)
                    )
                elif (
                    option["education_level"] == 2
                    and option["prep_struc_category_id"] == 1
                ):
                    result[
                        "scholarship_secondary_education"
                    ] = ScholarshipListOptions(lname).get_scholarship_list(
                        lname, xp_key, 2, 1, (3,)
                    )
                else:
                    pass
            result["columns_honours"] = ScholarshipListOptions(
                lname
            ).get_columns_honours(lname)
            DegreeCandidatesTask(lname).update_data_degree_candidates_task(
                task_id, "completed", result
            )

        except Exception as e:
            print(e)
            DegreeCandidatesTask(lname).update_status_degree_candidates_task(
                task_id, "failed"
            )


@scholarship.route(
    "/degree_candidates/<int:xp_key>",
    methods=["GET"],
)
@auth.login_required
def get_degree_candidates(lname, xp_key):
    if not Rating(lname).has_true_stipend():
        return (
            jsonify({"error": "Нет схем с полем stipend для подсчета достижений"}),
            404,
        )

    task_id = str(uuid.uuid4())  # Генерируем уникальный идентификатор задачи
    DegreeCandidatesTask(lname).create_degree_candidates_task(
        task_id, g.user.mid, "in_progress"
    )
    # Запускаем задачу в отдельном потоке
    app = current_app._get_current_object()
    thread = threading.Thread(
        target=degree_candidates_task,
        args=(app, lname, task_id, xp_key),
    )
    thread.start()
    return jsonify({"task_id": task_id}), 202


@scholarship.route("/degree_candidates_task/<string:task_id>", methods=["GET"])
@auth.login_required
def get_task_details(lname, task_id):
    task_id = uuid.UUID(task_id).hex
    degree_candidates_task = DegreeCandidatesTask(lname).get_degree_candidates_task(
        task_id
    )

    if not degree_candidates_task:
        return jsonify({"error": "Task not found"}), 404

    if degree_candidates_task and degree_candidates_task.mid != g.user.mid:
        return (
            jsonify({"error": "У данного пользователя нет доступа к этой информации"}),
            404,
        )

    if degree_candidates_task.status == "completed":
        # Объединяем данные перед возвратом
        data = degree_candidates_task.data or {}

        cadet_data = data.get("scholarship_higher_education_cadet", [])
        listener_data = data.get("scholarship_higher_education_listener", [])

        # Создаём новое объединённое поле
        combined_list = sorted(cadet_data + listener_data, key=lambda x: x.get("student_name", ""))
        data["scholarship_higher_education_cadet_listener"] = combined_list

        # Убираем старые поля
        data.pop("scholarship_higher_education_cadet", None)
        data.pop("scholarship_higher_education_listener", None)

        return jsonify({"status": degree_candidates_task.status, "data": data}), 200

    elif degree_candidates_task.status == "failed":
        return jsonify({"status": degree_candidates_task.status}), 500

    return jsonify({"status": degree_candidates_task.status}), 200


@scholarship.route(
    "/degree_candidates/info/<int:mid>/<int:gid>/<int:xp_key>/<int:education_level>/<int:prep_struc_category_id>",
    methods=["GET"],
)
@auth.login_required
def get_candidates_info(
    lname, mid, gid, xp_key, education_level, prep_struc_category_id
):
    result = ScholarshipListOptions(lname).get_candidates_info(
        lname, mid, gid, xp_key, education_level, prep_struc_category_id
    )
    if result:
        result["ovu_name"] = Company().getSetting(lname, "OVU_Name")
        result["gou_short"] = Company().getSetting(lname, "GOU_Short")
        result["honours"] = ScholarshipListOptions(lname).get_candidates_honours(
            lname, mid, xp_key, education_level, prep_struc_category_id
        )
    return jsonify(result)


@scholarship.route("/scholarship_list_options", methods=["GET"])
@auth.login_required
def get_scholarship_list(lname):
    try:
        # Получаем параметры запроса
        id = request.args.get("id", type=int)
        trmid = request.args.get("trmid", type=int)
        xp_key = request.args.get("xp_key", type=int)
        education_level = request.args.get("education_level", type=int)

        final_result = {}
        if xp_key:
            education_levels = ScholarshipListOptions(lname).check_levels_of_education(
                lname=lname, xp_key=xp_key
            )
            final_result["education_levels"] = education_levels

        # Вызываем метод для получения данных
        result = ScholarshipListOptions(lname).get_scholarship_list_options(
            lname=lname,
            id=id,
            trmid=trmid,
            xp_key=xp_key,
            education_level=education_level,
        )

        # Если данные не найдены
        if not result:
            final_result["data"] = []
        else:
            final_result["data"] = result

        # Возвращаем результат в формате JSON
        return jsonify(final_result), 200
    except Exception as e:
        print("Error in GET request: {e}".format(e=e))
        return jsonify({"error": str(e)}), 500


@scholarship.route(
    "/scholarship_list_options",
    methods=["POST"],
)
@auth.login_required
def insert_scholarship_list(lname):
    try:
        # Получаем данные из запроса
        data = request.json

        # Проверяем, что данные являются массивом
        if not isinstance(data, list):
            return jsonify({"error": "Invalid input format. Expected a list."}), 400

        inserted_ids = []

        # Обрабатываем каждый элемент массива
        for entry in data:
            if (
                "xp_key" in entry
                and "trmids" in entry
                and "education_level" in entry
                and "date_to" in entry
                and "date_from" in entry
                and "prep_struc_category_id" in entry
            ):
                ScholarshipListOptions(lname).delete_scholarship_list_option(
                    lname,
                    entry["xp_key"],
                    entry["education_level"],
                    entry["prep_struc_category_id"],
                )

                new_id = ScholarshipListOptions(lname).insert_scholarship_list_option(
                    lname, entry
                )
                if new_id:
                    inserted_ids.extend(new_id)
                else:
                    return jsonify({"error": "Failed to insert data."}), 500
            else:
                return jsonify({"error": "Invalid input data structure."}), 400

        return jsonify({"inserted_ids": inserted_ids}), 200
    except Exception as e:
        print("Error in POST request: {e}".format(e=e))
        return jsonify({"error": str(e)}), 500


@scholarship.route("/scholarship_candidates", methods=["GET"])
@auth.login_required
def get_scholarship_candidates(lname):
    try:
        xp_key = request.args.get("xp_key", type=int)
        mid = request.args.get("mid", type=int)
        gid = request.args.get("gid", type=int)
        education_level = request.args.get("education_level", type=int)
        scholarship_type = request.args.get("scholarship_type", type=str)

        result = ScholarshipListOptions(lname).get_scholarship_candidates(
            lname=lname,
            xp_key=xp_key,
            mid=mid,
            gid=gid,
            education_level=education_level,
            scholarship_type=scholarship_type,
        )

        if not result:
            return jsonify({"error": "No records found"}), 404
        return jsonify(result), 200
    except Exception as e:
        print("Error in GET request: {e}".format(e=e))
        return jsonify({"error": str(e)}), 500


@scholarship.route("/scholarship_candidates", methods=["POST"])
@auth.login_required
def create_scholarship_candidate(lname):
    data = request.json

    # Проверяем наличие обязательных полей
    required_fields = ["xp_key", "mid", "gid", "education_level", "scholarship_type"]
    if not all(k in data for k in required_fields):
        return jsonify({"message": "Missing required fields"}), 400

    # Проверяем, что scholarship_type является массивом
    if not isinstance(data["scholarship_type"], list):
        return jsonify({"message": "scholarship_type must be a list"}), 400

    # Проверка на допустимые значения в scholarship_type
    allowed_scholarship_types = [
        "Президента РФ",
        "Правительства РФ",
        "Министерства обороны РФ",
        "Исключить",
    ]
    if not all(st in allowed_scholarship_types for st in data["scholarship_type"]):
        return jsonify({"message": "Invalid scholarship_type value"}), 400

    # Удаление значений для этого пользователя
    ScholarshipListOptions(lname).delete_scholarship_candidate(lname, data["xp_key"], data["mid"], data["gid"], data["education_level"])

    # Обрабатываем каждое значение из scholarship_type
    results = []
    for scholarship in data["scholarship_type"]:
        single_entry = {
            "xp_key": data["xp_key"],
            "mid": data["mid"],
            "gid": data["gid"],
            "education_level": data["education_level"],
            "scholarship_type": scholarship,
        }
        try:
            scholarship_candidate = ScholarshipListOptions(lname).create_scholarship_candidate(
                lname, single_entry
            )
            results.append(dict(scholarship_candidate))
        except Exception as e:
            return jsonify({"message": "Failed to create candidate: {e}".format(e=str(e))}), 500

    return jsonify(results), 201


@scholarship.route(
    "/scholarship_candidates/<type>/<string:task_id>/<int:xp_key>",
    methods=["GET"],
)
@auth.login_required
def get_scholarship_candidates_mo_list(lname, type, task_id, xp_key):
    if type == "ksmo":
        result = {
            "scholarship_candidates_higher_cadet": ScholarshipListOptions(
                lname
            ).get_scholarship_candidates_list(
                lname, task_id, xp_key, 1, "Министерства обороны РФ", 1, (1, 2, 4, 5)
            ),
            "scholarship_candidates_higher_listener": ScholarshipListOptions(
                lname
            ).get_scholarship_candidates_list(
                lname, task_id, xp_key, 1, "Министерства обороны РФ", 2, (1, 2, 4, 5)
            ),
        }
    elif type == "amo":
        result = {
            "scholarship_candidates_higher_adjunct": ScholarshipListOptions(
                lname
            ).get_scholarship_candidates_list(
                lname, task_id, xp_key, 1, "Министерства обороны РФ", 3, (1, 2, 4, 5)
            ),
        }
    elif type == "government":
        president_scholarship_candidates_higher_cadet_listener = ScholarshipListOptions(
            lname
        ).get_scholarship_candidates_list(
            lname, task_id, xp_key, 1, "Президента РФ", 1, (1, 2, 4, 5)
        )
        for i in ScholarshipListOptions(lname).get_scholarship_candidates_list(
            lname, task_id, xp_key, 1, "Президента РФ", 2, (1, 2, 4, 5)
        ):
            president_scholarship_candidates_higher_cadet_listener.append(i)

        government_scholarship_candidates_higher_cadet_listener = (
            ScholarshipListOptions(lname).get_scholarship_candidates_list(
                lname, task_id, xp_key, 1, "Правительства РФ", 1, (1, 2, 4, 5)
            )
        )
        for i in ScholarshipListOptions(lname).get_scholarship_candidates_list(
            lname, task_id, xp_key, 1, "Правительства РФ", 2, (1, 2, 4, 5)
        ):
            government_scholarship_candidates_higher_cadet_listener.append(i)

        result = {
            "president_scholarship_candidates_higher_cadet_listener": president_scholarship_candidates_higher_cadet_listener,
            "government_scholarship_candidates_higher_cadet_listener": government_scholarship_candidates_higher_cadet_listener,
            "government_scholarship_candidates_secondary_cadet": ScholarshipListOptions(
                lname
            ).get_scholarship_candidates_list(
                lname, task_id, xp_key, 2, "Правительства РФ", 1, (3,)
            ),
        }
    else:
        return jsonify(success=False, message="Incorrect document type passed"), 400
    return jsonify(result)


@scholarship.route(
    "/scholarship_candidates/excel/<type>/<string:task_id>/<int:xp_key>",
    methods=["GET"],
)
@auth.login_required
def get_excel_scholarship_candidates_mo_list(lname, type, task_id, xp_key):
    result, files, unique_mids_files = ScholarshipListOptions(lname).get_excel_data(lname, type, task_id, xp_key)
    tmpdir = tempfile.mkdtemp()
    try:
        zip_file = DegreeCandidatesTask(lname).create_zip_files_v3(lname, type, result, files, unique_mids_files, tmpdir)
        if isinstance(zip_file, tuple):
            return jsonify(message="Нет данных для экспорта"), 204
        return send_file(
            zip_file,
            as_attachment=True,
            attachment_filename="{type}_package.zip".format(type=type),
            mimetype="application/zip"
        )
    finally:
        shutil.rmtree(tmpdir)
