# coding=UTF-8
# LMSAPI/API/Journal.py

from sqlalchemy.sql import text
from flask import current_app


class IndividualPlans:
    @staticmethod
    def get_individual_plans(lname, did, full_access, full_podr, full_main, edit_access, user_id, school_year):
        conn = current_app.ms.db(lname).connect()
        filter_podr = " AND vs.depid = :did" if did > 0 else ""
        filter_access = ""
        if full_access > 0:
            filter_access = ""
        elif full_podr > 0 and (filter_podr is None or len(filter_podr) == 0):
            filter_access = (
                """ 
                AND ip.mid IN (SELECT DISTINCT pwd1.mid FROM vw_personnel_with_dep pwd 
                JOIN vw_personnel_with_dep pwd1 ON pwd.depid = pwd1.depid 
                WHERE pwd.mid = :user_id)
                """
            )
        elif full_podr > 0 and filter_podr:
            filter_access = ""
        else:
            filter_podr = " AND ip.mid = :user_id"

        edit_field = "0"
        delete_field = "0"

        if full_access == 2:
            edit_field = "1"
            delete_field = "1"
        elif full_podr == 2:
            if did > 0 or full_access == 0:
                edit_field = "1"
                delete_field = "1"
            else:
                edit_field = (
                    """
                    (SELECT CAST(count(*) > 0 AS int) FROM vw_personnel_with_dep pwd 
                    JOIN vw_personnel_with_dep pwd1 ON pwd.depid = pwd1.depid 
                    WHERE pwd.mid = :user_id AND pwd1.mid = t.mid)
                    """
                )
                delete_field = edit_field
        elif full_main == 2:
            edit_field = "CASE WHEN t.mid = :user_id THEN 1 ELSE 0 END"

        if edit_access == 1:
            edit_field = "1"

        sql_query = """
        SELECT t.id,
               t.fio AS fio,
               t.position_name AS position,
               t.degree AS academic_rank,
               (SELECT SUM(plan_1) FROM individual_plan_budget WHERE ip_id=t.id) AS one_semester,
               (SELECT SUM(plan_2) FROM individual_plan_budget WHERE ip_id=t.id) AS two_semester,
               (SELECT NULLIF(SUM(COALESCE(plan_1,0) + COALESCE(plan_2,0)),0) FROM individual_plan_budget WHERE ip_id=t.id) AS budget,
               t.status AS status,
               {edit_field}::int AS index_editing,
               {delete_field}::int AS index_deleting
        FROM (
            SELECT ip.id, 
                   xp_f_get_mid_fio(vs.mid, 0) AS fio,
                   CONCAT(pos.name, ' (', TRIM(TO_CHAR(vs.rate/100.0, '0.00')), ')') AS position_name,
                   TRIM(COALESCE(at.name, '') || COALESCE(' (' || ad.name || ')', ''))::character varying(254) AS degree,
                   ip.status, ip.mid
            FROM individual_plans ip
            JOIN vw_staff vs ON ip.mid = vs.mid AND vs.id = ip.cpid
            JOIN positions pos ON pos.pid = vs.positionid
            JOIN xp_personal_file pf ON pf.mid = ip.mid
            LEFT JOIN academicdegree ad ON ad.agid = pf."AcademicDegree"
            LEFT JOIN academictitle at ON at.atid = pf."AcademicTitle"
            WHERE ip.school_year = :school_year
            {filter_podr}
            {filter_access}
        ) AS t
        ORDER BY 2,3,4;
        """.format(filter_podr=filter_podr, filter_access=filter_access, edit_field=edit_field, delete_field=delete_field)

        stmt = text(sql_query)
        query = conn.execute(stmt, {"user_id": user_id, "school_year": school_year, "did": did})
        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    @staticmethod
    def get_individual_plans_filters_admin(lname, mid):
        conn = current_app.ms.db(lname).connect()

        base_sql = """
            SELECT 
                vwd.id AS did, 
                vwd.owner_dep, 
                vwd.name,
                CASE WHEN vwpwd.depid IS NOT NULL THEN TRUE ELSE FALSE END AS has_relation
            FROM vw_divisions vwd
            LEFT JOIN vw_personnel_with_dep vwpwd 
                ON vwpwd.depid = vwd.id AND vwpwd.mid = :mid
            ORDER BY COALESCE(vwd.owner_dep, 0), vwd.name;
            """

        stmt = text(base_sql)
        query = conn.execute(stmt, {"mid": mid})

        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    @staticmethod
    def get_individual_plans_filters(lname, mid):
        conn = current_app.ms.db(lname).connect()

        base_sql = """
            DROP TABLE IF EXISTS tmp_dep_list;
            CREATE TEMPORARY TABLE tmp_dep_list AS
            SELECT 
                  id AS did,  
                  owner_dep, 
                  "name", 
                  id IN (SELECT depid FROM vw_staff WHERE is_main AND mid = :mid) AS user_dep 
                FROM vw_divisions 
                WHERE id IN ( 
                              SELECT DISTINCT pd.depid 
                              FROM vw_personnel_with_dep pd 
                              WHERE pd.mid = :mid 
                            ) 
                ORDER BY COALESCE(owner_dep, 0), "name"; 
                UPDATE tmp_dep_list tdl 
                SET owner_dep = NULL 
               WHERE owner_dep NOT IN (SELECT did FROM tmp_dep_list WHERE did = tdl.owner_dep); 
               SELECT * 
               FROM tmp_dep_list;
        """

        stmt = text(base_sql)
        query = conn.execute(stmt, {"mid": mid})

        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    @staticmethod
    def insert_individual_plans(lname, xp_key, cid):
        conn = current_app.ms.db(lname).connect()

        insert_sql = """
        INSERT INTO individual_plans (
            school_year, mid, cpid, status, discussed, protocol_number, protocol_date, 
            performed, verified, approved, verified_mid, approved_mid
        ) 
        SELECT 
            :xp_key, cp.mid, cp.cpid, 'Новый', 0, '', NULL, 0, 0, 0, 0, 0  
        FROM 
            cathedra_personnel cp 
        WHERE 
            cid = :cid 
            AND cp.mid IS NOT NULL              
            AND NOT EXISTS (
                SELECT ip.mid 
                FROM individual_plans ip 
                WHERE school_year = :xp_key 
                AND ip.mid = cp.mid
            );
        """

        try:
            stmt = text(insert_sql).bindparams(xp_key=xp_key, cid=cid)
            conn.execute(stmt)
            return True
        except Exception as e:
            current_app.logger.error(
                "Error inserting individual plans: {e}".format(e=e)
            )
            return None

    @staticmethod
    def update_individual_plan(lname, plan_id, data):
        conn = current_app.ms.db(lname).connect()

        fields_to_update = {}
        if "discussed" in data:
            fields_to_update["discussed"] = data["discussed"]
        if "protocol_number" in data:
            fields_to_update["protocol_number"] = data["protocol_number"]
        if "protocol_date" in data:
            fields_to_update["protocol_date"] = data["protocol_date"]
        if "sign_date" in data:
            fields_to_update["sign_date"] = data["sign_date"]

        if not fields_to_update:
            return None

        set_clause = ", ".join(
            ["{key} = :{key}".format(key=key) for key in fields_to_update.keys()]
        )
        update_sql = """
        UPDATE individual_plans
        SET {set_clause}
        WHERE id = :plan_id
        RETURNING id, discussed, protocol_number, protocol_date, sign_date
        """.format(
            set_clause=set_clause
        )

        stmt = text(update_sql).bindparams(**fields_to_update, plan_id=plan_id)

        result = conn.execute(stmt)
        updated_plan = result.fetchone()

        if not updated_plan:
            return None

        return dict(zip(result.keys(), updated_plan))

    @staticmethod
    def create_individual_plans(lname, lessontypeid: int, image_bytes: bytes):
        conn = current_app.ms.db(lname).connect()

        sql = """
            UPDATE eventtools
            SET bmp = :image_bytes
            WHERE typeid = :lessontypeid
            """

        stmt = text(sql)
        stmt = stmt.bindparams(lessontypeid=lessontypeid)
        stmt = stmt.bindparams(image_bytes=image_bytes)
        conn.execute(stmt)

    @staticmethod
    def get_main(lname, ip_id):
        """ Вкладка 'Главная' """
        conn = current_app.ms.db(lname).connect()

        sql = """
            SELECT
                xp_format_fio(p.LastName, p.FirstName, p.Patronymic,0)::varchar(90) fio,
                sy.name as syname,
                ip.status,
                cath.cathedra,
                xp_format_fio(vp.LastName, vp.FirstName, vp.Patronymic,0)::varchar(90) ver_fio,
                xp_format_fio(ap.LastName, ap.FirstName, ap.Patronymic,0)::varchar(90) appr_fio,
                ps.name AS "position",
                ip.discussed,
                ip.protocol_number,
                ip.protocol_date,
                ip.status,
                ip.performed,
                ip.verified,
                ip.verified_mid,
                ip.approved,
                ip.approved_mid,
                ip.sign_date
                FROM individual_plans ip
                LEFT OUTER JOIN people             p ON p.mid = ip.mid
                LEFT OUTER JOIN xp_personal_file   pf on pf.mid = p.mid
                LEFT OUTER JOIN positions          ps on ps.pid = pf."Position"
                LEFT OUTER JOIN school_year        sy ON sy.xp_key = ip.school_year
                LEFT OUTER JOIN cathedra_personnel cp ON cp.mid = ip.mid
                LEFT OUTER JOIN cathedras        cath ON cath.idcathedra = cp.cid
                LEFT OUTER JOIN people             vp ON vp.mid = ip.verified_mid
                LEFT OUTER JOIN people             ap ON ap.mid = ip.approved_mid
            WHERE ip.id = :ip_id
        """
        stmt = text(sql)
        stmt = stmt.bindparams(ip_id=ip_id)
        query = conn.execute(stmt)

        result = [dict(i) for i in query]

        return result

    @staticmethod
    def get_tasks_for_the_academic_year(lname, ip_id):
        """ Вкладка 'I. Задачи на учебный год' """
        conn = current_app.ms.db(lname).connect()

        sql = "SELECT * FROM individual_plan_tasks WHERE ip_id = :ip_id"
        stmt = text(sql)
        stmt = stmt.bindparams(ip_id=ip_id)
        query = conn.execute(stmt)

        result = [dict(i) for i in query]

        return result

    @staticmethod
    def create_task_for_the_academic_year(lname, ip_id, data):
        """ Создание задачи на вкладке 'I. Задачи на учебный год' """
        conn = current_app.ms.db(lname).connect()

        sql = """
            INSERT INTO individual_plan_tasks (ip_id, task, term, performed, mode)
            VALUES (:ip_id, :task, :term, :performed, :mode)
            RETURNING id
        """

        stmt = text(sql).bindparams(
            ip_id=ip_id,
            task=data.get("task"),
            term=data.get("term"),
            performed=data.get("performed"),
            mode=data.get("mode")
        )

        result = conn.execute(stmt)
        new_id = result.fetchone()[0]

        return new_id

    @staticmethod
    def update_tasks_for_the_academic_year(lname, task_id, updates):
        """ Вкладка 'I. Задачи на учебный год' """
        conn = current_app.ms.db(lname).connect()

        set_clause = ", ".join(["{0} = :{0}".format(key) for key in updates.keys()])
        updates['task_id'] = task_id

        sql = "UPDATE individual_plan_tasks SET {set_clause} WHERE id = :task_id".format(set_clause=set_clause)
        stmt = text(sql).bindparams(**updates)

        conn.execute(stmt)

    @staticmethod
    def delete_tasks_for_the_academic_year(lname, task_id):
        """ Вкладка 'I. Задачи на учебный год' """
        conn = current_app.ms.db(lname).connect()

        sql = "DELETE FROM individual_plan_tasks WHERE id = :task_id"
        stmt = text(sql).bindparams(task_id=task_id)

        conn.execute(stmt)

    @staticmethod
    def get_annual_budget_of_office_time(lname, ip_id):
        """ Вкладка 'II. Годовой бюджет служебного (рабочего) времени' """
        conn = current_app.ms.db(lname).connect()

        sql = """
        SELECT nnz_fill_ind_budget(:ip_id);
        SELECT DISTINCT
            id,
            parent,
            name,
            plan_1,
            fact_1,
            plan_2,
            fact_2,
            s_plan,
            s_fact,
            count, sort1 as num, sort2
            FROM tmp_budget
            ORDER BY sort1, sort2
            """
        stmt = text(sql)
        stmt = stmt.bindparams(ip_id=ip_id)
        query = conn.execute(stmt)

        result = [dict(i) for i in query]

        return result

    @staticmethod
    def get_educational_work_times(lname, ip_id, mid, xp_key):
        """ Вкладка 'III. Учебная работа' """
        conn = current_app.ms.db(lname).connect()

        result = {}

        # 1 строка
        sql = """
        SELECT nnz_fill_ind_budget(:ip_id);
        SELECT sum(s_plan) 
        FROM tmp_budget 
        WHERE parent IS NOT NULL AND TRIM(parent) != ''
        """
        stmt = text(sql)
        stmt = stmt.bindparams(ip_id=ip_id)
        query = conn.execute(stmt)
        result["1"] = [dict(i) for i in query]


        # 2 строка
        sql = """
        SELECT SUM(b.plan_1 + b.plan_2) AS plan
        FROM methodical_work_kinds w
        LEFT JOIN courses c ON w.bycourses 
            AND c.cid IN (SELECT cid FROM teachers WHERE mid=:mid)   
            AND c.cid in (SELECT cid FROM individual_plan_budget bc WHERE bc.ip_id=:ip_id AND bc.meth_id=w.kid)  
        JOIN (
                SELECT meth_id FROM individual_plan_budget 
                WHERE ip_id=:ip_id 
                GROUP BY meth_id
            ) bc ON bc.meth_id=w.kid  
        JOIN individual_plan_budget b ON b.meth_id=w.kid AND b.ip_id=:ip_id AND (b.cid=c.cid OR c.cid IS null)  
        LEFT JOIN individual_plan_budget br ON br.meth_id=w.kid AND br.ip_id=:ip_id AND br.cid IS null 
        WHERE w.mode IN (0, 1, 2, 3)
            AND (
                NULLIF(w.status, 0) IS null 
                OR COALESCE(b.plan_1, 0) + COALESCE(b.plan_2, 0) + COALESCE(b.fact_1, 0) + COALESCE(b.fact_2, 0) > 0
                )
        """
        stmt = text(sql)
        stmt = stmt.bindparams(ip_id=ip_id, mid=mid)
        query = conn.execute(stmt)
        result["2"] = [dict(i) for i in query]

        # 3 строка
        sql = """
        SELECT 
            prikaz, 
            datetostr(orderdate, 0) AS orderdate,
            total,
            lesson 
        FROM year_load 
        WHERE school_year=:xp_key
        """
        stmt = text(sql)
        stmt = stmt.bindparams(xp_key=xp_key)
        query = conn.execute(stmt)
        result["3"] = [dict(i) for i in query]

        # 4 строка
        sql = """
        SELECT SUM(b.plan_1 + b.plan_2) AS plan
        FROM methodical_work_kinds w
        LEFT JOIN courses c ON w.bycourses 
            AND c.cid IN (SELECT cid FROM teachers WHERE mid=:mid)   
            AND c.cid in (SELECT cid FROM individual_plan_budget bc WHERE bc.ip_id=:ip_id AND bc.meth_id=w.kid)  
        JOIN (
                SELECT meth_id FROM individual_plan_budget 
                WHERE ip_id=:ip_id 
                GROUP BY meth_id
            ) bc ON bc.meth_id=w.kid  
        JOIN individual_plan_budget b ON b.meth_id=w.kid AND b.ip_id=:ip_id AND (b.cid=c.cid OR c.cid IS null)  
        LEFT JOIN individual_plan_budget br ON br.meth_id=w.kid AND br.ip_id=:ip_id AND br.cid IS null
        WHERE w.mode IN (4)
            AND (
                NULLIF(w.status, 0) IS null 
                OR COALESCE(b.plan_1, 0) + COALESCE(b.plan_2, 0) + COALESCE(b.fact_1, 0) + COALESCE(b.fact_2, 0) > 0
                );
        """
        stmt = text(sql)
        stmt = stmt.bindparams(ip_id=ip_id, mid=mid)
        query = conn.execute(stmt)
        result["4"] = [dict(i) for i in query]

        return result

    @staticmethod
    def get_educational_work_table(lname, ip_id, mid, xp_key):
        """ Вкладка 'III. Учебная работа' - таблица """
        conn = current_app.ms.db(lname).connect()

        sql = """
        SELECT 
            w.kid AS "id", 
            REPLACE(w.kind_name, E'\n', ' ') AS "kind_name",
            w.bycourses,
            c.cid,
            concat_ws(' ',c.title,c.course_index) AS "title",
            array_to_string (b.gids1,',') AS "gids1",
            array_to_string(b.gids2,',') AS "gids2",
            (
                SELECT string_agg(gh.name,',' order by year,name) 
                FROM group_history gh 
                WHERE gh.school_year=:xp_key AND gh.gid=any(b.gids1)) AS "groupnames1",
            (
                SELECT string_agg(gh.name,',' order by year,name) 
                FROM group_history gh WHERE gh.school_year=:xp_key AND gh.gid=any(b.gids2)) AS groupnames2,
            b.plan_1,
            b.fact_1,
            b.plan_2,
            b.fact_2,
            b.hiden,
            w.razdel
        FROM methodical_work_kinds w
        LEFT JOIN courses c ON w.bycourses 
            AND c.cid in (SELECT cid FROM teachers WHERE mid=:mid)
            AND c.cid in (SELECT cid FROM individual_plan_budget bc WHERE bc.ip_id=:ip_id AND bc.meth_id=w.kid)
        JOIN individual_plan_budget b ON b.meth_id=w.kid AND b.ip_id=:ip_id 
            AND (b.cid=c.cid OR c.cid IS null)
        LEFT JOIN individual_plan_budget br ON br.meth_id=w.kid 
            AND br.ip_id=:ip_id AND br.cid IS null
        WHERE w.mode=3
            AND (
                NULLIF(w.status, 0) IS null 
                OR COALESCE(b.plan_1,0) + COALESCE(b.plan_2, 0)+COALESCE(b.fact_1,0)+COALESCE(b.fact_2,0)>0
                )
        ORDER BY w.razdel, br.sorting, b.sorting, COALESCE(w.sorting, 1),
            w.kind_name, w.kid, c.title, c.course_index, c.cid
        """
        stmt = text(sql)
        stmt = stmt.bindparams(ip_id=ip_id, mid=mid, xp_key=xp_key)
        query = conn.execute(stmt)

        result = [dict(i) for i in query]

        return result

    @staticmethod
    def get_group(lname, ip_id):
        """ Уч группы для вкладки 'III. Методическая работа' """
        conn = current_app.ms.db(lname).connect()

        sql = """
        SELECT 
            gh.gid,
            gh.year AS "kurs",
            gh.name AS "groupname"
        FROM group_history gh
        JOIN individual_plans p ON p.school_year=gh.school_year
        WHERE p.id=:ip_id
        ORDER BY gh.year,gh.name
        """
        stmt = text(sql)
        stmt = stmt.bindparams(ip_id=ip_id)
        query = conn.execute(stmt)

        result = [dict(i) for i in query]

        return result

    @staticmethod
    def get_IV_V_VI_VII(lname, ip_id, mid, mode):
        """
            Вкладки:
                'IV. Методическая работа',
                'V. Научная (найчно-иследовательская) работа',
                'VI. Воспитательная работа',
                'VII. Профессионально-должностная подготовка и другие виды деятельности'
            mode: значение определяет вкладку
                методическая 0
                научная 1
                воспитательная 2
                проф-должн 4
        """
        conn = current_app.ms.db(lname).connect()

        sql = """
        SELECT 
            w.kid AS "id", 
            REPLACE(w.kind_name, E'\n', '') AS "kind_name", 
            w.bycourses,
            c.cid,
            concat_ws(' ', c.title, c.course_index) AS "title",
            plan_1,
            fact_1,
            plan_2,
            fact_2, 
            b.hiden, 
            w.razdel
        FROM methodical_work_kinds w
        LEFT JOIN courses c ON w.bycourses AND c.cid IN (SELECT cid FROM teachers WHERE mid=:mid)
        LEFT JOIN individual_plan_budget b ON b.meth_id=w.kid AND b.ip_id=:ip_id AND (b.cid=c.cid OR c.cid IS null)
        WHERE w.mode=:mode AND (c.cid IS NOT null OR NOT w.bycourses)
        ORDER BY w.razdel, COALESCE(w.sorting, 1), w.kind_name, w.kid, c.title, c.course_index, c.cid
        """
        stmt = text(sql)
        stmt = stmt.bindparams(ip_id=ip_id, mid=mid, mode=mode)
        query = conn.execute(stmt)

        result = [dict(i) for i in query]

        return result

    @staticmethod
    def update_III_IV_V_VI_VII(lname, kid, ip_id, updates):
        """
            Вкладки:
                'III. Учебная работа', mode = 3
                'IV. Методическая работа', mode = 0
                'V. Научная (найчно-иследовательская) работа', mode = 1
                'VI. Воспитательная работа', mode = 2
                'VII. Профессионально-должностная подготовка и другие виды деятельности' mode = 4
        """
        conn = current_app.ms.db(lname).connect()

        set_clause = ", ".join(["{0} = :{0}".format(key) for key in updates.keys()])
        updates['kid'] = kid
        updates['ip_id'] = ip_id

        sql = "UPDATE individual_plan_budget SET {set_clause} WHERE meth_id = :kid AND ip_id = :ip_id".format(
            set_clause=set_clause)
        stmt = text(sql).bindparams(**updates)

        conn.execute(stmt)

    @staticmethod
    def get_content_planned_works(lname, ip_id, mid, hiden):
        """ Вкладка 'VIII. Содержание планируемых работ' """
        conn = current_app.ms.db(lname).connect()

        sql = """
        SELECT 
            d.id,
            d.task, 
            d.plan, 
            d.fact, 
            d.termdate,
            CASE w.mode
                WHEN 0 THEN 'Методическая работа'
                WHEN 1 THEN 'Научная (научно-исследовательская) работа'
                WHEN 2 THEN 'Воспитательная работа'
                WHEN 3 THEN 'Учебная работа'
                WHEN 4 THEN 'Профессионально-должностная подготовка'
            END AS "mode_name", 
            c.cid, 
            c.title,
            b.id AS "budget_id", 
            b.termdate AS "budget_term", 
            b.hiden,
            COALESCE(b.plan_1, 0) + COALESCE(b.plan_2,0) AS "budget_plan", 
            COALESCE(b.fact_1, 0) + COALESCE(b.fact_2, 0) AS "budget_fact",
            w.kid, 
            w.kind_name,
            w.mode, 
            w.razdel
        FROM  methodical_work_kinds w
        JOIN individual_plan_budget b ON b.meth_id=w.kid
        LEFT JOIN courses c ON c.cid=b.cid AND w.bycourses AND c.cid IN (SELECT cid FROM teachers WHERE mid=:mid)
        LEFT JOIN individual_plan_details d ON b.id=d.budget_id
        WHERE b.ip_id=:ip_id AND COALESCE(b.plan_1, 0) + COALESCE(b.plan_2, 0) > 0
        AND (b.cid > 0 AND w.bycourses OR b.cid IS null AND NOT COALESCE(w.bycourses, false))
        """
        hide = " AND hiden = false " if not hiden else ''

        sql += hide + "ORDER BY w.mode, w.razdel, w.sorting, w.kind_name, w.kid, c.title, c.cid, d.task, d.id"

        stmt = text(sql)
        stmt = stmt.bindparams(ip_id=ip_id, mid=mid)
        query = conn.execute(stmt)

        result = [dict(i) for i in query]

        return result

    @staticmethod
    def create_content_planned_works(lname, data):
        """ Вкладка 'VIII. Содержание планируемых работ' """
        conn = current_app.ms.db(lname).connect()

        sql = """
            INSERT INTO individual_plan_details (budget_id, termdate, plan, fact, task)
            VALUES (:budget_id, :termdate, :plan, :fact, :task)
            RETURNING id
            """

        stmt = text(sql).bindparams(
            budget_id=data.get("budget_id"),
            termdate=data.get("termdate"),
            plan=data.get("plan"),
            fact=data.get("fact"),
            task=data.get("task"),
        )

        result = conn.execute(stmt)
        new_id = result.fetchone()[0]

        return new_id

    @staticmethod
    def update_content_planned_work(lname, work_id, updates):
        """ Вкладка 'VIII. Содержание планируемых работ' """
        conn = current_app.ms.db(lname).connect()

        set_clause = ", ".join(["{0} = :{0}".format(key) for key in updates.keys()])
        updates['work_id'] = work_id

        sql = "UPDATE individual_plan_details SET {set_clause} WHERE id = :work_id".format(set_clause=set_clause)
        stmt = text(sql).bindparams(**updates)

        conn.execute(stmt)

    @staticmethod
    def delete_content_planned_work(lname, work_id):
        """ Вкладка 'VIII. Содержание планируемых работ' """
        conn = current_app.ms.db(lname).connect()

        sql = "DELETE FROM individual_plan_details WHERE id = :work_id"
        stmt = text(sql).bindparams(work_id=work_id)

        conn.execute(stmt)
