import json
import os
import shutil
import tempfile
import zipfile
from collections import OrderedDict, defaultdict
from decimal import Decimal
from io import BytesIO
from typing import Tuple, Union, List, Dict, Any

from flask import current_app
from openpyxl.styles import Alignment
from openpyxl.utils import get_column_letter
from openpyxl.workbook import Workbook
from sqlalchemy import text

from LMSAPI.api.Models.File import File
from LMSAPI.api.Models.ScholarshipCandidatesFilesStorage import ScholarshipCandidatesFilesStorage


class ScholarshipListOptions:
    def __init__(self, cname):
        self.lname = cname

    @staticmethod
    def get_scholarship_list(
        lname: str,
        xp_key: int,
        education_level: int,
        prep_struc_category_id: int,
        prep_levels: Tuple[int] = (),
    ) -> Union[bool, List[Dict[str, Any]]]:
        # 1 - Высшее образование, 2 - Среднее образование
        # prep_struc_category_id: 1 - Курсант, 2 - Слушатель, 3 - Адъюнкт
        # prep_levels - Уровень подготовки из ОПОП
        where = ""
        if prep_levels:
            # Преобразование кортежа в строку для IN фильтра
            placeholders = ", ".join(
                [":prep_level_{i}".format(i=i) for i in range(len(prep_levels))]
            )
            where += " AND dc.preparation_level IN ({placeholders}) ".format(
                placeholders=placeholders
            )

        if prep_struc_category_id == 1:
            where += " AND ey1.number > 1 "
        elif prep_struc_category_id == 3:
            where += " AND dc.five_percentage = 100 "

        # Базовый SQL запрос
        sql = """
            SELECT DISTINCT
                dc.mid,
                dc.student_name,
                dc.gid,
                dc.group_name,
                dc.education_level, 
                dc.preparation_level,
                dc.name,
                dc.five_count,
                dc.five_percentage,
                dc.four_count,
                dc.four_percentage,
                dc.three_count,
                dc.three_percentage,
                dc.two_count,
                dc.two_percentage,
                dc.average_grade,
                dc.current_avg_grade,
                dc.record_count,
                dc.record_points,
                dc.record_count_year,
                dc.prep_struc_category_id,
                dc.prep_struc_category,
                dc.third_level_data,
                ey1.number yeducationyearid,
                ey1.name yeducationyear, 
                jsonb_agg(
                    jsonb_build_object(
                        'scholarship_candidates_id', sc.scholarship_candidates_id,
                        'scholarship_type', sc.scholarship_type
                    )
                ) FILTER (WHERE sc.scholarship_candidates_id IS NOT NULL) AS grants,
				mr.title militaryrank
            FROM degree_candidates12(:xp_key, :education_level, :prep_struc_category_id) dc
            LEFT JOIN scholarship_candidates sc 
                ON dc.mid = sc.mid AND sc.xp_key = :xp_key AND sc.education_level = :education_level
            LEFT JOIN groupname g 
                ON dc.gid = g.gid AND g.cid = -1
            LEFT JOIN group_history gh1 
                ON gh1.gid = g.gid AND gh1.school_year = (SELECT sy_sub.xp_key
				    FROM school_year sy_sub
				    WHERE sy_sub.begdate < (
				        SELECT sy.begdate
				        FROM school_year sy
				        WHERE sy.xp_key = :xp_key
				    )
				    ORDER BY sy_sub.begdate DESC
				    LIMIT 1
    			)
            LEFT JOIN educationyears ey1 
                ON ey1.number = gh1.year
            LEFT JOIN groupuser_transfer gt
                ON dc.mid = gt.mid
                AND gt.when = (
                    SELECT MAX(gut.when)
                    FROM groupuser_transfer gut
                    WHERE gut.mid = dc.mid
                )
            LEFT JOIN xp_status xs on xs.xp_key = gt.xp_status
            JOIN xp_personal_file xpf ON xpf.mid = dc.mid
			LEFT OUTER JOIN military_rank mr ON mr.id_mil_rank = xpf."MilitaryRank"
			LEFT JOIN militaryprofession mp on mp.mpid = g.f_militaryprofession
			LEFT JOIN edu_direction ed on ed.edu_direct_id = mp.edu_direct_id
            WHERE dc.education_level = :education_level
              AND dc.prep_struc_category_id = :prep_struc_category_id
              AND (xs."Type" != 'Отчисление' AND xs."Type" != 'Выпуск' AND xs."Type" != 'Архивная запись' OR xs."Type" IS NULL)
              AND dc.record_count >= 1
              AND (
                  ed.p_mastering_y IS NULL OR 
                  ey1.number <
                      CASE 
                          WHEN ed.p_mastering_m IS NULL OR ed.p_mastering_m = 0 
                          THEN ed.p_mastering_y
                          ELSE ed.p_mastering_y + 1
                      END
              )
              {where}
            GROUP BY 
                dc.mid, dc.student_name, dc.gid, dc.group_name, dc.education_level, 
                dc.preparation_level, dc.name, dc.five_count, dc.five_percentage, 
                dc.four_count, dc.four_percentage, dc.three_count, dc.three_percentage, 
                dc.two_count, dc.two_percentage, dc.average_grade, dc.current_avg_grade, 
                dc.record_count, dc.record_points, dc.record_count_year, 
                dc.prep_struc_category_id, dc.prep_struc_category, dc.third_level_data, 
                ey1.number, ey1.name, mr.title, ed.p_mastering_y, ed.p_mastering_m
            ORDER BY 
                dc.student_name;
        """.format(
            where=where
        )

        # Выполнение запроса
        conn = current_app.ms.db(lname).connect()
        stmt = text(sql)
        stmt = stmt.bindparams(
            xp_key=xp_key, education_level=education_level
        )

        if prep_struc_category_id != 0:
            stmt = stmt.bindparams(prep_struc_category_id=prep_struc_category_id)

        # Привязываем значения для preparation_level
        for i, level in enumerate(prep_levels):
            stmt = stmt.bindparams(**{"prep_level_{i}".format(i=i): level})

        query = conn.execute(stmt)
        result = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

        # Convert Decimal to float
        for row in result:
            for key, value in row.items():
                if isinstance(value, Decimal):
                    row[key] = float(value)

        return result

    @staticmethod
    def get_scholarship_list_options(
        lname: str, id, trmid, xp_key, education_level
    ) -> Union[bool, List[Dict[str, Any]]]:
        # Получаем параметры запроса
        filters = []
        params = {}

        # Создаем фильтры
        if id:
            filters.append("id = :id")
            params["id"] = id
        if trmid:
            filters.append("trmid = :trmid")
            params["trmid"] = trmid
        if xp_key:
            filters.append("xp_key = :xp_key")
            params["xp_key"] = xp_key
        if education_level:
            filters.append("education_level = :education_level")
            params["education_level"] = education_level

        # Базовый SQL запрос
        sql = """
            SELECT slo.*, t.title
            FROM scholarship_list_options slo
            JOIN Terms t ON t.trmid = slo.trmid 
            WHERE 1=1
            """

        # Если есть фильтры, добавляем их в запрос
        if filters:
            sql += " AND " + " AND ".join(filters)

        sql += " ORDER BY id"

        # Выполнение запроса
        conn = current_app.ms.db(lname).connect()
        query = conn.execute(text(sql), params)
        res = query.fetchall()

        if not res:
            return False
        return [dict(row) for row in res]

    @staticmethod
    def get_education_level_and_prep_struc_category_id_option(
        lname: str, xp_key
    ) -> Union[bool, List[Dict[str, Any]]]:
        # Базовый SQL запрос
        sql = """
                SELECT DISTINCT prep_struc_category_id, education_level
                FROM scholarship_list_options
                WHERE xp_key = :xp_key
                """

        # Выполнение запроса
        conn = current_app.ms.db(lname).connect()
        query = conn.execute(text(sql), {"xp_key": xp_key})
        res = query.fetchall()

        if not res:
            return False
        return [dict(row) for row in res]

    @staticmethod
    def insert_scholarship_list_option(lname: str, data: dict) -> Union[bool, int]:
        try:
            conn = current_app.ms.db(lname).connect()

            # Вставляем записи для каждого trmid с xp_key и education_level
            sql_trmid = """
            INSERT INTO scholarship_list_options (xp_key, trmid, date_from, date_to, education_level, prep_struc_category_id)
            VALUES (:xp_key, :trmid, :date_from, :date_to, :education_level, :prep_struc_category_id)
            RETURNING id
            """

            # Начинаем транзакцию
            with conn.begin():
                inserted_ids = []
                # Вставляем данные с trmid
                for trmid in data["trmids"]:
                    result = conn.execute(
                        text(sql_trmid),
                        {
                            "xp_key": data["xp_key"],
                            "trmid": trmid,
                            "education_level": data["education_level"],
                            "date_to": data["date_to"],
                            "date_from": data["date_from"],
                            "prep_struc_category_id": data["prep_struc_category_id"],
                        },
                    )
                    new_id = result.fetchone()[0]
                    inserted_ids.append(new_id)
                return inserted_ids if inserted_ids else False
        except Exception as e:
            print("Error inserting scholarship list option: {e}".format(e=e))
            return False

    @staticmethod
    def delete_scholarship_list_option(
        lname: str, xp_key: int, education_level: int, prep_struc_category_id: int
    ) -> bool:
        sql = "DELETE FROM scholarship_list_options WHERE xp_key = :xp_key AND education_level = :education_level AND prep_struc_category_id = :prep_struc_category_id"
        conn = current_app.ms.db(lname).connect()
        query = conn.execute(
            text(sql),
            {
                "xp_key": xp_key,
                "education_level": education_level,
                "prep_struc_category_id": prep_struc_category_id,
            },
        )

        return query.rowcount > 0  # Возвращает True, если удалено больше 0 строк

    @staticmethod
    def get_scholarship_candidates(
        lname: str,
        xp_key: int,
        mid: int,
        gid: int,
        education_level: int,
        scholarship_type: str,
    ) -> Union[bool, List[Dict[str, Any]]]:
        # Получаем параметры запроса
        filters = []
        params = {}

        if xp_key:
            filters.append("xp_key = :xp_key")
            params["xp_key"] = xp_key
        if mid:
            filters.append("mid = :mid")
            params["mid"] = mid
        if gid:
            filters.append("gid = :gid")
            params["gid"] = gid
        if education_level:
            filters.append("education_level = :education_level")
            params["education_level"] = education_level
        if scholarship_type:
            filters.append("scholarship_type = :scholarship_type")
            params["scholarship_type"] = scholarship_type

        sql = "SELECT * FROM scholarship_candidates WHERE 1=1"
        if filters:
            sql += " AND " + " AND ".join(filters)

        # Выполнение запроса
        conn = current_app.ms.db(lname).connect()
        query = conn.execute(text(sql), params)
        res = query.fetchall()

        if not res:
            return False
        return [dict(row) for row in res]

    @staticmethod
    def create_scholarship_candidate(lname: str, data: dict):
        sql = """
            INSERT INTO scholarship_candidates (xp_key, mid, gid, education_level, scholarship_type)
            VALUES (:xp_key, :mid, :gid, :education_level, :scholarship_type)
            RETURNING scholarship_candidates_id, xp_key, mid, gid, education_level, scholarship_type
            """

        conn = current_app.ms.db(lname).connect()
        query = conn.execute(text(sql), data)
        result = query.fetchone()
        if not result:
            return False
        return result

    @staticmethod
    def delete_scholarship_candidate(lname: str, xp_key: int, mid: int, gid: int, education_level: int):
        sql = "DELETE FROM scholarship_candidates WHERE xp_key = :xp_key AND mid = :mid AND gid = :gid AND education_level = :education_level"

        # Выполнение запроса
        conn = current_app.ms.db(lname).connect()
        result = conn.execute(
            text(sql), {"xp_key": xp_key, "mid": mid, "gid": gid, "education_level": education_level}
        )
        if result.rowcount == 0:
            return False
        return True

    @staticmethod
    def get_scholarship_candidates_list(
        lname: str,
        task_id,
        xp_key: int,
        education_level: int,
        scholarship_type: str,
        prep_struc_category_id: int,
        prep_levels: Tuple[int] = (),
    ) -> Union[bool, List[Dict[str, Any]]]:
        where = ""
        if scholarship_type:
            where += " AND sc.scholarship_type = :scholarship_type "

        if prep_levels:
            # Преобразование кортежа в строку для IN фильтра
            placeholders = ", ".join(
                [":prep_level_{i}".format(i=i) for i in range(len(prep_levels))]
            )
            where += " AND dc.preparation_level IN ({placeholders}) ".format(
                placeholders=placeholders
            )

        # Базовый SQL запрос
        sql = """
            WITH parsed_data AS (
                SELECT 
                    jsonb_array_elements(
                        CASE
                            WHEN :prep_struc_category_id = 1 AND :education_level = 1 THEN data->'scholarship_higher_education_cadet'
                            WHEN :prep_struc_category_id = 2 AND :education_level = 1 THEN data->'scholarship_higher_education_listener'
                            WHEN :prep_struc_category_id = 3 AND :education_level = 1 THEN data->'scholarship_higher_education_adjunct'
                            WHEN :prep_struc_category_id = 1 AND :education_level = 2 THEN data->'scholarship_secondary_education'
                            ELSE NULL
                        END
                    ) AS candidate_data
                FROM degree_candidates_task
                WHERE task_id = :task_id
            ),
            candidate_data AS (
                SELECT
                    CAST(candidate_data->>'gid' AS INT) AS gid,
                    CAST(candidate_data->>'mid' AS INT) AS mid,
                    candidate_data->>'name' AS name,
                    candidate_data->>'group_name' AS group_name,
                    candidate_data->>'student_name' AS student_name,
                    candidate_data->>'prep_struc_category' AS prep_struc_category,
                    CAST(candidate_data->>'prep_struc_category_id' AS INT) AS prep_struc_category_id,
                    CAST(candidate_data->>'average_grade' AS FLOAT) AS average_grade,
                    CAST(candidate_data->>'two_count' AS INT) AS two_count,
                    CAST(candidate_data->>'five_count' AS INT) AS five_count,
                    CAST(candidate_data->>'four_count' AS INT) AS four_count,
                    CAST(candidate_data->>'three_count' AS INT) AS three_count,
                    CAST(candidate_data->>'record_count' AS FLOAT) AS record_count,
                    CAST(candidate_data->>'record_points' AS FLOAT) AS record_points,
                    CAST(candidate_data->>'two_percentage' AS FLOAT) AS two_percentage,
                    candidate_data->>'yeducationyear' AS yeducationyear,
                    CAST(candidate_data->>'education_level' AS INT) AS education_level,
                    CAST(candidate_data->>'five_percentage' AS FLOAT) AS five_percentage,
                    CAST(candidate_data->>'four_percentage' AS FLOAT) AS four_percentage,
                    CAST(candidate_data->>'three_percentage' AS FLOAT) AS three_percentage,
                    CAST(candidate_data->>'yeducationyearid' AS INT) AS yeducationyearid,
                    CAST(candidate_data->>'current_avg_grade' AS FLOAT) AS current_avg_grade,
                    CAST(candidate_data->>'preparation_level' AS INT) AS preparation_level,
                    CAST(candidate_data->>'record_count_year' AS FLOAT) AS record_count_year,
                    candidate_data->'grants' AS grants,
                    candidate_data->'third_level_data' AS third_level_data
                FROM parsed_data
            )
            SELECT 
                sc.xp_key,
                sc.mid,
                sc.gid,
                sc.education_level,
                sc.scholarship_candidates_id,
				p.lastname || ' ' || p.firstname || ' ' || p.patronymic AS student_name, 
                ey.number yeducationyearid,
                ey.name yeducationyear, 
                sc.scholarship_type,
                xpf.number,
                mr.title militaryrank,
                xpf."Assignment" as assignment,
                dc.prep_struc_category,
                dc.prep_struc_category_id
            FROM candidate_data dc
            JOIN scholarship_candidates sc 
                ON dc.mid = sc.mid AND sc.xp_key = :xp_key AND sc.education_level = :education_level
			JOIN people p ON p.mid = sc.mid
            LEFT JOIN groupname g 
                ON sc.gid = g.gid AND g.cid = -1
            LEFT JOIN educationyears ey 
                ON ey.number = g.year
            LEFT JOIN groupuser_transfer gt
                ON sc.mid = gt.mid
                AND gt.when = (
                    SELECT MAX(gut.when)
                    FROM groupuser_transfer gut
                    WHERE gut.mid = sc.mid
                )
            left join xp_status xs on xs.xp_key = gt.xp_status
			JOIN xp_personal_file xpf ON xpf.mid = sc.mid
            JOIN prep_struc_category psc ON psc.cid = xpf.category
            LEFT OUTER JOIN military_rank mr ON mr.id_mil_rank = xpf."MilitaryRank"
            WHERE dc.average_grade > 0 
              AND dc.three_count = 0 
              AND dc.two_count = 0 
              AND dc.education_level = :education_level
              AND dc.prep_struc_category_id = :prep_struc_category_id
              AND (xs."Type" != 'Отчисление' AND xs."Type" != 'Выпуск' AND xs."Type" != 'Архивная запись' OR xs."Type" IS NULL)
              AND dc.five_percentage > 50.0
			  AND (sc.scholarship_type IS NULL OR sc.scholarship_type <> 'Нет')
			  {where}
			ORDER BY
              mr.title ASC NULLS LAST,
              p.lastname ASC,
              p.firstname ASC,
              p.patronymic ASC;
            """.format(
            where=where
        )

        # Выполнение запроса
        conn = current_app.ms.db(lname).connect()
        stmt = text(sql)
        stmt = stmt.bindparams(
            task_id=task_id,
            xp_key=xp_key,
            education_level=education_level,
            prep_struc_category_id=prep_struc_category_id,
        )

        for i, level in enumerate(prep_levels):
            stmt = stmt.bindparams(**{"prep_level_{i}".format(i=i): level})

        if scholarship_type:
            stmt = stmt.bindparams(scholarship_type=scholarship_type)

        query = conn.execute(stmt)
        result = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

        # Преобразование Decimal в float
        for row in result:
            for key, value in row.items():
                if isinstance(value, Decimal):
                    row[key] = float(value)
        return result

    @staticmethod
    def get_excel_data(lname, type, task_id, xp_key):
        if type == "ksmo":
            cadet_result, cadet_files, cadet_unique_mids_files = DegreeCandidatesTask(lname).export_candidates_from_task(
                lname, task_id, "Министерства обороны РФ", xp_key, 1, 1, (1, 2, 4, 5)
            )

            listener_result, listener_files, listener_unique_mids_files = DegreeCandidatesTask(
                lname).export_candidates_from_task(
                lname, task_id, "Министерства обороны РФ", xp_key, 1, 2, (1, 2, 4, 5)
            )

            result = sorted(cadet_result + listener_result, key=lambda x: x.get("ФИО", ""))
            files = cadet_files + listener_files
            unique_mids = list(
                set(cadet_unique_mids_files + listener_unique_mids_files))

            return result, files, unique_mids
        elif type == "amo":
            return DegreeCandidatesTask(
                lname).export_candidates_from_task(
                lname, task_id, "Министерства обороны РФ", xp_key, 1, 3, (1, 2, 4, 5)
            )
        elif type == "government":
            phc_result, phc_files, phc_unique_mids_files = DegreeCandidatesTask(
                lname).export_candidates_from_task(
                lname, task_id, "Президента РФ", xp_key, 1, 1, (1, 2, 4, 5)
            )

            phl_result, phl_files, phl_unique_mids_files = DegreeCandidatesTask(
                lname).export_candidates_from_task(
                lname, task_id, "Президента РФ", xp_key, 1, 2, (1, 2, 4, 5)
            )

            ghc_result, ghc_files, ghc_unique_mids_files = DegreeCandidatesTask(
                lname).export_candidates_from_task(
                lname, task_id, "Правительства РФ", xp_key, 1, 1, (1, 2, 4, 5)
            )

            ghl_result, ghl_files, ghl_unique_mids_files = DegreeCandidatesTask(
                lname).export_candidates_from_task(
                lname, task_id, "Правительства РФ", xp_key, 1, 2, (1, 2, 4, 5)
            )

            gsc_result, gsc_files, gsc_unique_mids_files = DegreeCandidatesTask(
                lname).export_candidates_from_task(
                lname, task_id, "Правительства РФ", xp_key, 2, 1, (3,)
            )

            result = sorted(phc_result + phl_result + ghc_result + ghl_result + gsc_result, key=lambda x: x.get("ФИО", ""))
            files = phc_files + phl_files + ghc_files + ghl_files + gsc_files
            unique_mids = list(
                set(phc_unique_mids_files + phl_unique_mids_files + ghc_unique_mids_files + ghl_unique_mids_files + gsc_unique_mids_files))

            return result, files, unique_mids
        else:
            return None

    @staticmethod
    def get_candidates_info(
        lname: str,
        mid: int,
        gid: int,
        xp_key: int,
        education_level: int,
        prep_struc_category_id: int,
    ):
        # Базовый SQL запрос
        sql = """
            SELECT 
                tg.mid, 
                p.lastname AS lastname, 
                p.firstname AS firstname, 
                p.patronymic AS patronymic, 
                gn.gid, 
                gn.name::TEXT AS groupname,
                COUNT(CASE WHEN NULLIF(tg.grade, '')::NUMERIC = 5 THEN 1 END)::INTEGER AS five_count,
                100.0 * COUNT(CASE WHEN NULLIF(tg.grade, '')::NUMERIC = 5 THEN 1 END) / NULLIF(COUNT(tg.grade), 0) AS five_percentage,
                COUNT(CASE WHEN NULLIF(tg.grade, '')::NUMERIC = 4 THEN 1 END)::INTEGER AS four_count,
                100.0 * COUNT(CASE WHEN NULLIF(tg.grade, '')::NUMERIC = 4 THEN 1 END) / NULLIF(COUNT(tg.grade), 0) AS four_percentage,
                ROUND(SUM(CAST(tg.grade AS INTEGER))::NUMERIC / NULLIF(COUNT(CAST(tg.grade AS INTEGER)), 0), 2) AS average_grade,
                ps.education_level,
                ps.preparation_level,
                ps.name::TEXT AS name,
                psc.cid AS prep_struc_category_id,
                psc.name AS prep_struc_category,
                gh.year AS kurs,
                mr.title AS mil_rank,
                jsonb_agg(DISTINCT jsonb_build_object(
					'scholarship_candidates_id', sc.scholarship_candidates_id,
					'scholarship_type', sc.scholarship_type
				)) FILTER (WHERE sc.scholarship_candidates_id IS NOT NULL) AS grants,
                xpf."Assignment" as assignment
            FROM term_grades tg
            JOIN courses c ON c.cid = tg.cid
            JOIN groupname gn ON gn.gid = :gid
            JOIN group_history gh ON gh.gid = gn.gid AND gh.school_year = :xp_key 
            JOIN militaryprofession mp ON gn.f_militaryprofession = mp.mpid
            JOIN edu_direction ed ON mp.edu_direct_id = ed.edu_direct_id
            JOIN preparation_structure ps ON ed.program_level = ps.ps_id
            JOIN people p ON p.mid = tg.mid
            LEFT JOIN xp_personal_file xpf ON xpf.mid = p.mid
            LEFT JOIN military_rank mr ON xpf."MilitaryRank" = mr.id_mil_rank
            LEFT JOIN prep_struc_category psc ON psc.cid = xpf.category
            LEFT JOIN scholarship_candidates sc ON sc.xp_key = :xp_key AND sc.mid = tg.mid AND sc.gid = gn.gid AND sc.education_level = :education_level 
            WHERE tg.mid = :mid
                AND tg.trmid IN (
                SELECT DISTINCT t.trmid
                FROM scholarship_list_options t
                WHERE t.xp_key = :xp_key  
                AND t.trmid IS NOT NULL
                AND t.education_level = :education_level
                AND t.prep_struc_category_id = :prep_struc_category_id
            )
            AND tg.jcid IN (
                SELECT DISTINCT jc.jcid
                FROM journalcertification jc
                JOIN nnz_schedule s ON s.sheid = jc.f_nnz_schedule
                JOIN eventtools e ON e.typeid = s.pair_type_id 
                    AND CASE WHEN 3 = 3 THEN e.processtypeid IN (3, 62) WHEN 3 = 8 THEN e.processtypeid = 8 END
                JOIN nnz_weeks w ON w.week_id = jc.f_nnz_weeks
                JOIN term_weeks tw ON tw.week_id = w.week_id
                JOIN terms t ON tw.trmid = t.trmid
                JOIN groupname g ON g.gid = s.gid
                JOIN group_history gh ON gh.gid = g.gid
                LEFT JOIN groupname g2 ON g2.gid = g.owner_gid
                LEFT JOIN group_history gh2 ON gh2.gid = g2.gid
                WHERE t.trmid IN (
                    SELECT DISTINCT slo.trmid
                    FROM scholarship_list_options slo
                    WHERE slo.xp_key = :xp_key 
                    AND slo.trmid IS NOT NULL
                    AND slo.education_level = :education_level
                    AND slo.prep_struc_category_id = :prep_struc_category_id
                )
                and coalesce(gh.school_year, gh2.school_year) = :xp_key
            )
            AND tg.grade IS NOT NULL  -- Убираем NULL значения
            AND tg.grade ~ '^[0-9]+(\.[0-9]+)?$'  -- Фильтруем только числовые значения
            GROUP BY tg.mid, gn.gid, gn.name, p.lastname, p.firstname, p.patronymic, ps.education_level, ps.preparation_level, 
                ps.name, psc.cid, psc.name, gh.year, mr.title, xpf."Assignment"
                """
        # Выполнение запроса
        conn = current_app.ms.db(lname).connect()
        stmt = text(sql)
        stmt = stmt.bindparams(
            mid=mid,
            gid=gid,
            xp_key=xp_key,
            education_level=education_level,
            prep_struc_category_id=prep_struc_category_id,
        )

        query = conn.execute(stmt)
        res = query.fetchone()
        if not res:
            return False
        else:
            return {
                "mid": res["mid"],
                "lastname": res["lastname"],
                "firstname": res["firstname"],
                "patronymic": res["patronymic"],
                "gid": res["gid"],
                "groupname": res["groupname"],
                "five_count": int(res["five_count"]),
                "five_percentage": float(res["five_percentage"]),
                "four_count": int(res["four_count"]),
                "four_percentage": float(res["four_percentage"]),
                "average_grade": float(res["average_grade"]),
                "education_level": int(res["education_level"]),
                "preparation_level": int(res["preparation_level"]),
                "name": res["name"],
                "prep_struc_category_id": int(res["prep_struc_category_id"]),
                "prep_struc_category": res["prep_struc_category"],
                "kurs": int(res["kurs"]),
                "mil_rank": res["mil_rank"],
                "grants": res["grants"],
                "assignment": res["assignment"],
            }

    @staticmethod
    def get_candidates_honours(
        lname: str,
        mid: int,
        xp_key: int,
        education_level: int,
        prep_struc_category_id: int,
    ):
        # Базовый SQL запрос
        sql = """
            WITH RECURSIVE indicator_tree AS (
                -- Шаг 1: Получаем записи первого уровня, где id_group = 0
                SELECT 
                    pi.degree_tree_id AS current_id,
                    pi.title AS current_title,
                    pi.degree_tree_id_group AS previous_id_group,
                    pi.degree_tree_id AS first_level_id,
                    pi.title AS first_level_title,
                    1 AS level
                FROM personnel_indicators pi
                JOIN rating_schema rs ON rs.id = pi.schema
                WHERE rs.stipend IS TRUE  
                  AND pi.degree_tree_id_group = 0
            
                UNION ALL
            
                -- Шаг 2: Рекурсивно получаем записи следующих уровней, где id_group = current_id предыдущего уровня
                SELECT 
                    pi.degree_tree_id AS current_id,
                    pi.title AS current_title,
                    pi.degree_tree_id_group AS previous_id_group,
                    it.first_level_id,
                    it.first_level_title,
                    it.level + 1 AS level
                FROM personnel_indicators pi
                JOIN rating_schema rs ON rs.id = pi.schema
                JOIN indicator_tree it ON pi.degree_tree_id_group = it.current_id
                WHERE rs.stipend IS TRUE
            ),
            final_table AS (
                SELECT distinct
                  it.first_level_id,
                  it.first_level_title,
                  it.current_id AS indicator_id,
                  pi2.name_for_report AS indicator_title,
                  pi2.weight AS indicator_weight,
                  it.previous_id_group AS previous_level_id,
                  pi.name_for_report AS previous_level_title,
                  it.level
                FROM indicator_tree it
                LEFT JOIN personnel_indicators pi ON it.previous_id_group = pi.degree_tree_id
                JOIN rating_schema rs ON rs.id = pi.schema and rs.stipend IS TRUE
                LEFT JOIN personnel_indicators pi2 ON it.current_id = pi2.degree_tree_id
                JOIN rating_schema rs1 ON rs1.id = pi2.schema and rs1.stipend IS TRUE
                WHERE pi2.forgrant IS TRUE
                ORDER BY it.first_level_id, it.level
            ),
            indicator_mapping  AS (
				-- 130
				SELECT 3670 AS indicator_id, 130 AS previous_level_id, 'Достижения за период освоения образовательной программы в научной (научно-исследовательской, научно-практической) деятельности - постоянное значение. количество публикаций, всего' AS previous_level_title, 0 as previous_level_group UNION ALL
				SELECT 3671, 130, 'Достижения за период освоения образовательной программы в научной (научно-исследовательской, научно-практической) деятельности - постоянное значение. количество публикаций, всего', 0 UNION ALL
				SELECT 3728, 130, 'Достижения за период освоения образовательной программы в научной (научно-исследовательской, научно-практической) деятельности - постоянное значение. количество публикаций, всего', 0 UNION ALL
				SELECT 3674, 130, 'Достижения за период освоения образовательной программы в научной (научно-исследовательской, научно-практической) деятельности - постоянное значение. количество публикаций, всего', 0 UNION ALL

				-- 140
				SELECT 3694, 140, 'Наличие гранта на выполнение научно-исследовательской работы', 0 UNION ALL

				-- 150 и/или 151
				SELECT 3637, 150, 'Достижения в учебе и науке, всего', 0 UNION ALL
				SELECT 3637, 151, 'участие в олимпиадах и иных конкурсных мероприятиях, всего', 150 UNION ALL
				SELECT 3638, 150, 'Достижения в учебе и науке, всего', 0 UNION ALL
				SELECT 3638, 151, 'участие в олимпиадах и иных конкурсных мероприятиях, всего', 150 UNION ALL
				SELECT 3639, 150, 'Достижения в учебе и науке, всего', 0 UNION ALL
				SELECT 3639, 151, 'участие в олимпиадах и иных конкурсных мероприятиях, всего', 150 UNION ALL
				SELECT 3640, 150, 'Достижения в учебе и науке, всего', 0 UNION ALL
				SELECT 3640, 151, 'участие в олимпиадах и иных конкурсных мероприятиях, всего', 150 UNION ALL
				SELECT 3695, 150, 'Достижения в учебе и науке, всего', 0 UNION ALL
				SELECT 3695, 151, 'участие в олимпиадах и иных конкурсных мероприятиях, всего', 150 UNION ALL

				-- 150 и/или 152
				SELECT 3653, 150, 'Достижения в учебе и науке, всего', 0 UNION ALL
				SELECT 3653, 152, 'участие в работе конференций, симпозиумов, семинаров с докладом о результатах научно-исследовательской работы в течение учебного года, предшествующего учебному году назначения стипендии, всего', 150 UNION ALL
				SELECT 3654, 150, 'Достижения в учебе и науке, всего', 0 UNION ALL
				SELECT 3654, 152, 'участие в работе конференций, симпозиумов, семинаров с докладом о результатах научно-исследовательской работы в течение учебного года, предшествующего учебному году назначения стипендии, всего', 150 UNION ALL
				SELECT 3655, 150, 'Достижения в учебе и науке, всего', 0 UNION ALL
				SELECT 3655, 152, 'участие в работе конференций, симпозиумов, семинаров с докладом о результатах научно-исследовательской работы в течение учебного года, предшествующего учебному году назначения стипендии, всего', 150 UNION ALL
				SELECT 3656, 150, 'Достижения в учебе и науке, всего', 0 UNION ALL
				SELECT 3656, 152, 'участие в работе конференций, симпозиумов, семинаров с докладом о результатах научно-исследовательской работы в течение учебного года, предшествующего учебному году назначения стипендии, всего', 150 UNION ALL

				-- 150 и/или 153
				SELECT 3677, 150, 'Достижения в учебе и науке, всего', 0 UNION ALL
				SELECT 3677, 153, 'наличие научных и учебно-методических работ, всего', 150 UNION ALL
				SELECT 3678, 150, 'Достижения в учебе и науке, всего', 0 UNION ALL
				SELECT 3678, 153, 'наличие научных и учебно-методических работ, всего', 150 UNION ALL
				SELECT 3729, 150, 'Достижения в учебе и науке, всего', 0 UNION ALL
				SELECT 3729, 153, 'наличие научных и учебно-методических работ, всего', 150 UNION ALL
				SELECT 3730, 150, 'Достижения в учебе и науке, всего', 0 UNION ALL
				SELECT 3730, 153, 'наличие научных и учебно-методических работ, всего', 150 UNION ALL
				SELECT 3679, 150, 'Достижения в учебе и науке, всего', 0 UNION ALL
				SELECT 3679, 153, 'наличие научных и учебно-методических работ, всего', 150 UNION ALL
				SELECT 3659, 150, 'Достижения в учебе и науке, всего', 0 UNION ALL
				SELECT 3659, 153, 'наличие научных и учебно-методических работ, всего', 150 UNION ALL
				SELECT 3660, 150, 'Достижения в учебе и науке, всего', 0 UNION ALL
				SELECT 3660, 153, 'наличие научных и учебно-методических работ, всего', 150 UNION ALL
				SELECT 3661, 150, 'Достижения в учебе и науке, всего', 0 UNION ALL
				SELECT 3661, 153, 'наличие научных и учебно-методических работ, всего', 150 UNION ALL
				SELECT 3666, 150, 'Достижения в учебе и науке, всего', 0 UNION ALL
				SELECT 3666, 153, 'наличие научных и учебно-методических работ, всего', 150 UNION ALL
				SELECT 3664, 150, 'Достижения в учебе и науке, всего', 0 UNION ALL
				SELECT 3664, 153, 'наличие научных и учебно-методических работ, всего', 150 UNION ALL

				-- 160
				SELECT 3576, 160, 'Служебная карточка, количество поощрений за период обучения, всего', 0 UNION ALL
				SELECT 3577, 160, 'Служебная карточка, количество поощрений за период обучения, всего', 0 UNION ALL
				SELECT 3731, 160, 'Служебная карточка, количество поощрений за период обучения, всего', 0 UNION ALL
				SELECT 3732, 160, 'Служебная карточка, количество поощрений за период обучения, всего', 0 UNION ALL
				SELECT 3578, 160, 'Служебная карточка, количество поощрений за период обучения, всего', 0 UNION ALL

				-- 170
				SELECT 3710, 170, 'Иные достижения в учебной и (или) научной деятельности, всего', 0 UNION ALL
				SELECT 3711, 170, 'Иные достижения в учебной и (или) научной деятельности, всего', 0 UNION ALL
				SELECT 3712, 170, 'Иные достижения в учебной и (или) научной деятельности, всего', 0
			),
            balls  AS (
                SELECT 
                    im.previous_level_id,
                    im.previous_level_title,
                    im.previous_level_group,
                    ft.indicator_id,
                    ft.indicator_title,
                    ft.indicator_weight,
                    COALESCE(prv.value, 0) AS value
                FROM final_table ft
                JOIN indicator_mapping im ON ft.indicator_id = im.indicator_id
                JOIN personnel_indicators pi ON pi.degree_tree_id = ft.indicator_id
                LEFT JOIN personnel_rating_values prv 
                    ON prv.indicator = pi.id
                    AND prv.mid = :mid
                    AND prv.value IS NOT NULL 
                    AND prv.value != 0
                    AND prv.periodend BETWEEN 
                        (SELECT DISTINCT slo.date_from 
                         FROM scholarship_list_options slo 
                         WHERE slo.xp_key = :xp_key
                           AND slo.date_from IS NOT NULL 
                           AND slo.education_level = :education_level
                           AND slo.prep_struc_category_id = :prep_struc_category_id) 
                        AND 
                        (SELECT DISTINCT slo.date_to 
                         FROM scholarship_list_options slo 
                         WHERE slo.xp_key = :xp_key
                           AND slo.date_from IS NOT NULL 
                           AND slo.education_level = :education_level
                           AND slo.prep_struc_category_id = :prep_struc_category_id)
                WHERE ft.level = (SELECT MAX(level) FROM final_table)
            ), 
            final_balls AS (
                SELECT 
                    b.previous_level_id,
                    b.previous_level_title,
                    b.previous_level_group,
                    JSON_AGG(indicator_obj ORDER BY indicator_weight DESC) AS indicators
                FROM (
                    SELECT 
                        previous_level_id,
                        previous_level_title,
                        previous_level_group,
                        indicator_id,
                        indicator_title,
                        indicator_weight,
                        SUM(value) AS total_value,
                        JSON_BUILD_OBJECT(
                            'indicator_id', indicator_id,
                            'indicator_title', indicator_title,
                            'value', SUM(value)
                        ) AS indicator_obj
                    FROM balls
                    GROUP BY previous_level_id, previous_level_title, previous_level_group, indicator_id, indicator_title, indicator_weight
                ) b
                GROUP BY b.previous_level_id, b.previous_level_title, b.previous_level_group
            ),
            summ_balls AS (
                SELECT 
                    previous_level_id, 
                    SUM(value) AS sum_value
                FROM balls
                GROUP BY previous_level_id
            )
            SELECT 
                b.*, 
                sb.sum_value
            FROM final_balls b
            JOIN summ_balls sb ON sb.previous_level_id = b.previous_level_id
            ORDER BY b.previous_level_id;
            """
        # Выполнение запроса
        conn = current_app.ms.db(lname).connect()
        stmt = text(sql)
        stmt = stmt.bindparams(
            mid=mid,
            xp_key=xp_key,
            education_level=education_level,
            prep_struc_category_id=prep_struc_category_id,
        )

        query = conn.execute(stmt)
        result = [OrderedDict(zip(tuple(query.keys()), i)) for i in query.cursor]

        # Преобразование Decimal в float
        for row in result:
            for key, value in row.items():
                if isinstance(value, Decimal):
                    row[key] = float(value)
        return result

    @staticmethod
    def build_flattened_honours_dict(honours: List[Dict[str, Any]]) -> OrderedDict:
        result = OrderedDict()
        seen_keys = defaultdict(int)

        # Мапа id -> title, для поиска родительского уровня
        id_to_title = {
            h["previous_level_id"]: h["previous_level_title"]
            for h in honours
        }

        # ID, которые используются как группы (то есть у них есть потомки)
        used_as_group = {
            h["previous_level_group"]
            for h in honours
            if h["previous_level_group"] != 0
        }

        for h in honours:
            current_id = h["previous_level_id"]

            # Пропускаем группы, они будут обработаны как родительские
            if current_id in used_as_group:
                continue

            parent_id = h["previous_level_group"]
            parent_title = id_to_title.get(parent_id) if parent_id != 0 else None
            current_title = h["previous_level_title"]

            for ind in h.get("indicators", []):
                indicator_title = ind["indicator_title"]
                value = ind["value"]

                # Формируем полный ключ
                if indicator_title == current_title:
                    full_key = indicator_title
                elif parent_title:
                    full_key = "{}__{}__{}".format(parent_title, current_title, indicator_title)
                else:
                    full_key = "{}__{}".format(current_title, indicator_title)

                # Обработка дубликатов
                seen_keys[full_key] += 1
                if seen_keys[full_key] == 1:
                    key = full_key
                else:
                    key = "{} .{}".format(full_key, seen_keys[full_key])

                result[key] = value

        return result

    @staticmethod
    def get_columns_honours(
        lname: str,
    ):
        # Базовый SQL запрос
        sql = """
             SELECT 130 AS previous_level_id, 'Достижения за период освоения образовательной программы в научной (научно-исследовательской, научно-практической) деятельности - постоянное значение. количество публикаций, всего' AS previous_level_title, 0 AS parent_id
			 UNION ALL
			 SELECT 140 AS previous_level_id, 'Наличие гранта на выполнение научно-исследовательской работы' AS previous_level_title, 0 AS parent_id
			 UNION ALL
			 SELECT 150 AS previous_level_id, 'Достижения в учебе и науке, всего' AS previous_level_title, 0 AS parent_id
			 UNION ALL
			 SELECT 151 AS previous_level_id, 'участие в олимпиадах и иных конкурсных мероприятиях, всего' AS previous_level_title, 150 AS parent_id
			 UNION ALL
			 SELECT 152 AS previous_level_id, 'участие в работе конференций, симпозиумов, семинаров с докладом о результатах научно-исследовательской работы в течение учебного года, предшествующего учебному году назначения стипендии, всего' AS previous_level_title, 150 AS parent_id
			 UNION ALL
			 SELECT 153 AS previous_level_id, 'наличие научных и учебно-методических работ, всего' AS previous_level_title, 150 AS parent_id
			 UNION ALL
			 SELECT 160 AS previous_level_id, 'Служебная карточка, количество поощрений за период обучения, всего' AS previous_level_title, 0 AS parent_id
			 UNION ALL
			 SELECT 170 AS previous_level_id, 'Иные достижения в учебной и (или) научной деятельности, всего' AS previous_level_title, 0 AS parent_id
			"""
        # Выполнение запроса
        conn = current_app.ms.db(lname).connect()
        stmt = text(sql)
        query = conn.execute(stmt)
        result = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

        # Преобразование Decimal в float
        for row in result:
            for key, value in row.items():
                if isinstance(value, Decimal):
                    row[key] = float(value)
        return result

    @staticmethod
    def check_levels_of_education(lname: str, xp_key: int):
        # Базовый SQL запрос
        sql = """
            -- Шаг 1: Создаем временную таблицу для хранения всех данных по группам
            CREATE TEMP TABLE all_group_data (
                mid INT,
                cid INT,
                gid INT
            ) ON COMMIT DROP;  -- Удалить таблицу при завершении транзакции
            -- Шаг 2: Создаем временную таблицу с валидными jcid
            DROP TABLE IF EXISTS valid_jsid;
            CREATE TEMP TABLE valid_jsid AS
            SELECT DISTINCT coalesce(g2.gid, g.gid) AS gid, coalesce(g.name, g2.name) AS groupname
            FROM journalcertification jc
            JOIN nnz_schedule s ON s.sheid = jc.f_nnz_schedule
            JOIN eventtools e ON e.typeid = s.pair_type_id 
                AND CASE WHEN 3 = 3 THEN e.processtypeid IN (3, 62) WHEN 3 = 8 THEN e.processtypeid = 8 END
            JOIN nnz_weeks w ON w.week_id = jc.f_nnz_weeks
            JOIN term_weeks tw ON tw.week_id = w.week_id
            JOIN terms t ON tw.trmid = t.trmid
            JOIN groupname g ON g.gid = s.gid
            JOIN group_history gh ON gh.gid = g.gid
            LEFT JOIN groupname g2 ON g2.gid = g.owner_gid
            LEFT JOIN group_history gh2 ON gh2.gid = g2.gid
            WHERE t.term_begin::date BETWEEN (
                            SELECT (sy.enddate - INTERVAL '2 year')::date 
                            FROM school_year sy
                            WHERE sy.xp_key = :xp_key
                        ) AND (
                            SELECT sy.enddate
                            FROM school_year sy
                            WHERE sy.xp_key = :xp_key
                        )
                        and coalesce(gh.school_year, gh2.school_year) = :xp_key;			
            SELECT DISTINCT
                psc.cid AS prep_struc_category_id,
                psc.name AS prep_struc_category,
                1 AS education_level
            FROM valid_jsid vj
            JOIN groupname g ON g.gid = vj.gid
            JOIN groupuser gu ON gu.gid = g.gid
            JOIN militaryprofession mp ON g.f_militaryprofession = mp.mpid
            JOIN edu_direction ed ON mp.edu_direct_id = ed.edu_direct_id
            JOIN preparation_structure ps ON ed.program_level = ps.ps_id
            JOIN xp_personal_file e ON e.mid = gu.mid 
            JOIN prep_struc_category psc ON psc.cid = e.category
            WHERE ps.education_level IN (1, 2) AND psc.cid IN (1, 2, 3)
            
            UNION ALL
            
            -- Добавляем запись с СПО, если есть education_level = 2
            SELECT DISTINCT
                1 AS prep_struc_category_id,
                'СПО' AS prep_struc_category,
                2 AS education_level
            FROM valid_jsid vj
            JOIN groupname g ON g.gid = vj.gid
            JOIN groupuser gu ON gu.gid = g.gid
            JOIN militaryprofession mp ON g.f_militaryprofession = mp.mpid
            JOIN edu_direction ed ON mp.edu_direct_id = ed.edu_direct_id
            JOIN preparation_structure ps ON ed.program_level = ps.ps_id
            WHERE ps.education_level = 2
            ORDER BY prep_struc_category_id desc;
            """
        # Выполнение запроса
        conn = current_app.ms.db(lname).connect()
        stmt = text(sql)
        stmt = stmt.bindparams(xp_key=xp_key)
        query = conn.execute(stmt)
        result = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

        # Преобразование Decimal в float
        for row in result:
            for key, value in row.items():
                if isinstance(value, Decimal):
                    row[key] = float(value)
        return result

    def get_org_short_name(self):
        conn = current_app.ms.db(self.lname).connect()
        sql = """
            SELECT convert_from(value, 'UTF-8') as value
            FROM options
            where name = 'GOU_Short'
            """
        stmt = text(sql)
        query = conn.execute(stmt)
        result = query.first()
        return result['value']



class DegreeCandidatesTask:
    def __init__(self, cname):
        self.lname = cname

    def create_degree_candidates_task(self, task_id, mid, status):
        try:
            conn = current_app.ms.db(self.lname).connect()
            sql = """
                insert into degree_candidates_task (task_id, mid, created_at, status) 
                values (:task_id, :mid, NOW(),:status)
                """
            stmt = text(sql)
            stmt = stmt.bindparams(task_id=task_id)
            stmt = stmt.bindparams(mid=mid)
            stmt = stmt.bindparams(status=status)
            query = conn.execute(stmt)
            return True
        except Exception as e:
            print(e)
            return False

    def update_status_degree_candidates_task(self, task_id, status):
        conn = current_app.ms.db(self.lname).connect()
        sql = """
            UPDATE degree_candidates_task
            SET status = :status
            WHERE task_id = :task_id
            """

        stmt = text(sql)
        stmt = stmt.bindparams(task_id=task_id, status=status)
        conn.execute(stmt)

    def update_data_degree_candidates_task(self, task_id, status, data):
        conn = current_app.ms.db(self.lname).connect()
        sql = """
            UPDATE degree_candidates_task
            SET status = :status, data = :data
            WHERE task_id = :task_id
            """

        stmt = text(sql)
        stmt = stmt.bindparams(
            task_id=task_id,
            status=status,
            data=json.dumps(data, ensure_ascii=False),
        )
        conn.execute(stmt)

    def get_degree_candidates_task(self, task_id):
        conn = current_app.ms.db(self.lname).connect()
        sql = """
            SELECT * FROM degree_candidates_task
            WHERE task_id = :task_id
            """
        stmt = text(sql)
        stmt = stmt.bindparams(task_id=task_id)
        query = conn.execute(stmt)
        result = query.first()
        return result

    def get_mids_from_task(self, task_id, xp_key, scholarship_type, education_level, prep_struc_category_id, prep_levels):
        conn = current_app.ms.db(self.lname).connect()

        where = ""
        bind_values = {
            "task_id": task_id,
            "xp_key": xp_key,
            "education_level": education_level,
            "prep_struc_category_id": prep_struc_category_id,
        }

        if scholarship_type:
            where += " AND sc.scholarship_type = :scholarship_type "
            bind_values["scholarship_type"] = scholarship_type

        if prep_levels:
            placeholders = []
            for i, level in enumerate(prep_levels):
                key = "prep_level_{i}".format(i=i)
                placeholders.append(":{key}".format(key=key))
                bind_values[key] = level
            where += " AND dc.preparation_level IN ({pch}) ".format(pch=', '.join(placeholders))

        sql = """
            WITH parsed_data AS (
                SELECT 
                    jsonb_array_elements(
                        CASE
                            WHEN :prep_struc_category_id = 1 AND :education_level = 1 THEN data->'scholarship_higher_education_cadet'
                            WHEN :prep_struc_category_id = 2 AND :education_level = 1 THEN data->'scholarship_higher_education_listener'
                            WHEN :prep_struc_category_id = 3 AND :education_level = 1 THEN data->'scholarship_higher_education_adjunct'
                            WHEN :prep_struc_category_id = 1 AND :education_level = 2 THEN data->'scholarship_secondary_education'
                            ELSE NULL
                        END
                    ) AS candidate_data
                FROM degree_candidates_task
                WHERE task_id = :task_id
            ),
            candidate_data AS (
                SELECT
                    CAST(candidate_data->>'gid' AS INT) AS gid,
                    CAST(candidate_data->>'mid' AS INT) AS mid,
                    candidate_data->>'name' AS name,
                    candidate_data->>'group_name' AS group_name,
                    candidate_data->>'student_name' AS student_name,
                    candidate_data->>'prep_struc_category' AS prep_struc_category,
                    CAST(candidate_data->>'prep_struc_category_id' AS INT) AS prep_struc_category_id,
                    CAST(candidate_data->>'average_grade' AS FLOAT) AS average_grade,
                    CAST(candidate_data->>'two_count' AS INT) AS two_count,
                    CAST(candidate_data->>'five_count' AS INT) AS five_count,
                    CAST(candidate_data->>'four_count' AS INT) AS four_count,
                    CAST(candidate_data->>'three_count' AS INT) AS three_count,
                    CAST(candidate_data->>'record_count' AS FLOAT) AS record_count,
                    CAST(candidate_data->>'record_points' AS FLOAT) AS record_points,
                    CAST(candidate_data->>'two_percentage' AS FLOAT) AS two_percentage,
                    candidate_data->>'yeducationyear' AS yeducationyear,
                    CAST(candidate_data->>'education_level' AS INT) AS education_level,
                    CAST(candidate_data->>'five_percentage' AS FLOAT) AS five_percentage,
                    CAST(candidate_data->>'four_percentage' AS FLOAT) AS four_percentage,
                    CAST(candidate_data->>'three_percentage' AS FLOAT) AS three_percentage,
                    CAST(candidate_data->>'yeducationyearid' AS INT) AS yeducationyearid,
                    CAST(candidate_data->>'current_avg_grade' AS FLOAT) AS current_avg_grade,
                    CAST(candidate_data->>'preparation_level' AS INT) AS preparation_level,
                    CAST(candidate_data->>'record_count_year' AS FLOAT) AS record_count_year,
                    candidate_data->'grants' AS grants,
                    candidate_data->'third_level_data' AS third_level_data
                FROM parsed_data
            ), file_links AS (
                SELECT
                    mid,
                    STRING_AGG(file_description, E'\n') AS link_description
                FROM scholarship_candidates_files_storage
                GROUP BY mid
            )
            SELECT 
                psc.name AS "Должность кандидата",
                ey.name AS "Курс обучения", 
                mr.title AS "Воинское звание",
                p.lastname || ' ' || p.firstname || ' ' || p.patronymic AS "ФИО", 
                CASE 
				    WHEN EXISTS (
				        SELECT 1 FROM scholarship_candidates_files_storage fs
				        WHERE fs.mid = p.mid AND fs.сhapter = 8
				    ) THEN 'В наличии' ELSE 'Отсутствует'
				END AS "Выписка из протокола заседания ученого (научного, научно-технического) совета (краткий вывод)",
				CASE 
				    WHEN EXISTS (
				        SELECT 1 FROM scholarship_candidates_files_storage fs
				        WHERE fs.mid = p.mid AND fs.сhapter = 9
				    ) THEN 'В наличии' ELSE 'Отсутствует'
				END AS "Характеристика-рекомендация (отзыв научного руководителя)",
				CASE 
				    WHEN EXISTS (
				        SELECT 1 FROM scholarship_candidates_files_storage fs
				        WHERE fs.mid = p.mid AND fs.сhapter = 10
				    ) THEN 'В наличии' ELSE 'Отсутствует'
				END AS "Служебная характеристика (краткий вывод)",
                dc.average_grade AS "Результаты промежуточной аттестации за учебный год, предшествующий учебному году назначения стипендии__средний бал",
                dc.five_count || '/' || dc.five_percentage AS "Результаты промежуточной аттестации за учебный год, предшествующий учебному году назначения стипендии__«отлично»",
                dc.four_count || '/' || dc.four_percentage AS "Результаты промежуточной аттестации за учебный год, предшествующий учебному году назначения стипендии__«хорошо»", 
                'Отсутствуют' AS "Академические задолженности",
                dc.record_points AS "Суммарный коэффициент достижений",
                dc.current_avg_grade AS "Текущая успеваемость",
                sc.scholarship_type AS "Вид стипендии",
                xpf.genitive_surname || ' ' || xpf.genitive_name || ' ' || xpf.genitive_patronymic AS "ФИО в Родительном падеже",
                p.mid AS "Ссылка",
                p.mid,
                g.gid
            FROM candidate_data dc
            JOIN scholarship_candidates sc 
                ON dc.mid = sc.mid AND sc.xp_key = :xp_key AND sc.education_level = :education_level
            JOIN people p ON p.mid = sc.mid
            LEFT JOIN groupname g 
                ON sc.gid = g.gid AND g.cid = -1
            LEFT JOIN group_history gh 
                ON gh.gid = g.gid AND gh.school_year = (SELECT sy_sub.xp_key
                    FROM school_year sy_sub
                    WHERE sy_sub.begdate < (
                        SELECT sy.begdate
                        FROM school_year sy
                        WHERE sy.xp_key = :xp_key
                    )
                    ORDER BY sy_sub.begdate DESC
                    LIMIT 1
                )
            LEFT JOIN educationyears ey 
                ON ey.number = gh.year
            LEFT JOIN groupuser_transfer gt
                ON sc.mid = gt.mid
                AND gt.when = (
                    SELECT MAX(gut.when)
                    FROM groupuser_transfer gut
                    WHERE gut.mid = sc.mid
                )
            LEFT JOIN xp_status xs ON xs.xp_key = gt.xp_status
            JOIN xp_personal_file xpf ON xpf.mid = sc.mid
            JOIN prep_struc_category psc ON psc.cid = xpf.category
            LEFT OUTER JOIN military_rank mr ON mr.id_mil_rank = xpf."MilitaryRank"
            LEFT JOIN file_links fl ON dc.mid = fl.mid
            WHERE dc.education_level = :education_level
              AND dc.prep_struc_category_id = :prep_struc_category_id
              AND (xs."Type" != 'Отчисление' AND xs."Type" != 'Выпуск' AND xs."Type" != 'Архивная запись' OR xs."Type" IS NULL)
              AND dc.five_percentage > 50.0
              AND (sc.scholarship_type IS NULL OR sc.scholarship_type <> 'Нет')
              {where}
        """.format(where=where)
        # Выполнение запроса
        stmt = text(sql).bindparams(**bind_values)

        query = conn.execute(stmt)
        columns = query.keys()
        result = [OrderedDict(zip(columns, row)) for row in query.cursor]

        # Преобразование Decimal в float
        for row in result:
            for key, value in row.items():
                if isinstance(value, Decimal):
                    row[key] = float(value)
        return result

    def export_candidates_from_task(
            self, lname, task_id, scholarship_type, xp_key, education_level, prep_struc_category_id, prep_levels
    ):
        result_info = DegreeCandidatesTask(lname).get_mids_from_task(
            task_id, xp_key, scholarship_type, education_level, prep_struc_category_id, prep_levels
        )

        for info in result_info:
            # Получаем honours
            honours = ScholarshipListOptions(lname).get_candidates_honours(
                lname, info.get("mid"), xp_key, education_level, prep_struc_category_id
            )
            ref_honours = ScholarshipListOptions(lname).build_flattened_honours_dict(honours)

            new_info = OrderedDict()
            inserted = False

            for key, value in info.items():
                new_info[key] = value

                if key == "Академические задолженности" and not inserted:
                    for k, v in ref_honours.items():
                        new_info[k] = v
                    inserted = True

            # Если "Академические задолженности" не найден, добавим honours в конец
            if not inserted:
                for k, v in ref_honours.items():
                    new_info[k] = v

            info.clear()
            info.update(new_info)

        files = ScholarshipCandidatesFilesStorage(lname).get_files(
            lname, task_id, xp_key, education_level, scholarship_type, prep_struc_category_id
        )
        unique_mids = list({item["mid"] for item in files if item.get("mid") is not None})
        return result_info, files, unique_mids

    @staticmethod
    def create_zip_files_v2(lname, type, result, files, unique_mids_files):
        import re
        org_name = ScholarshipListOptions(lname).get_org_short_name()

        with tempfile.TemporaryDirectory() as tmpdir:
            wb = Workbook()
            ws_main = wb.active
            ws_main.title = org_name

            if not result:
                return "Нет данных для экспорта", 204

            # Удаляем mid и gid, сохраняем порядок
            cleaned_result = []
            for row in result:
                cleaned = OrderedDict((k, v) for k, v in row.items() if k not in ("mid", "gid"))
                cleaned_result.append(cleaned)

            # Строим отображаемые и реальные ключи
            column_key_map = OrderedDict()
            for key in cleaned_result[0].keys():
                display_key = re.sub(r'\.\d+$', '', key)  # удаляем .1, .2 и т.д.
                if display_key not in column_key_map:
                    column_key_map[display_key] = key

            display_column_keys = list(column_key_map.keys())
            real_column_keys = list(column_key_map.values())

            # Построение дерева заголовков
            def build_header_tree(keys):
                tree = OrderedDict()
                for key in keys:
                    parts = key.split('__')
                    current = tree
                    for part in parts:
                        if part not in current:
                            current[part] = OrderedDict()
                        current = current[part]
                return tree

            def get_tree_depth(tree):
                if not tree:
                    return 1
                return 1 + max(get_tree_depth(sub) for sub in tree.values())

            def write_headers_from_tree(tree, ws, row=1, col=1, max_depth=None, col_key_list=[]):
                if max_depth is None:
                    max_depth = get_tree_depth(tree)

                def recursive_write(node, r, c):
                    for key, sub in node.items():
                        start_col = c
                        if sub:
                            c = recursive_write(sub, r + 1, c)
                            ws.merge_cells(start_row=r, start_column=start_col, end_row=r, end_column=c - 1)
                        else:
                            ws.merge_cells(start_row=r, start_column=c, end_row=max_depth, end_column=c)
                            col_key_list.append(c)
                            c += 1
                        ws.cell(row=r, column=start_col, value=key)
                    return c

                recursive_write(tree, row, col)
                return max_depth, col_key_list

            header_tree = build_header_tree(display_column_keys)
            max_depth, _ = write_headers_from_tree(header_tree, ws_main)

            # Центрирование
            for row in ws_main.iter_rows(min_row=1, max_row=max_depth, max_col=len(display_column_keys)):
                for cell in row:
                    cell.alignment = Alignment(horizontal='center', vertical='center', wrap_text=True)

            # Запись строк
            for i, row_data in enumerate(cleaned_result, start=max_depth + 1):
                for j, key in enumerate(real_column_keys, start=1):
                    display_key = display_column_keys[j - 1]
                    cell = ws_main.cell(row=i, column=j)

                    if display_key.lower() == "ссылка":
                        try:
                            mid = int(row_data.get(key, 0))
                            fio = next((f.get("fio") for f in files if f.get("mid") == mid), None)
                            base_dir = File(lname).get_scholarship_candidates_files_storage_directory(mid)
                            has_files = any(f.get("mid") == mid for f in files)
                            if base_dir and has_files and fio:
                                safe_fio = fio.replace("/", "_").replace("\\", "_")
                                relative_path = 'files/{}/'.format(safe_fio)
                                display_text = "Открыть файлы {}".format(safe_fio)
                                formula = '=HYPERLINK(LEFT(CELL("filename", A1),FIND("[",CELL("filename", A1))-1) & "{}", "{}")'.format(
                                    relative_path, display_text)
                                cell.value = formula
                                cell.style = "Hyperlink"
                            else:
                                cell.value = "Отсутствуют"
                        except Exception:
                            cell.value = "Отсутствуют"
                    else:
                        cell.value = row_data.get(key, "")

            # Автоширина
            for i, col_cells in enumerate(ws_main.columns, start=1):
                max_length = 0
                col_letter = get_column_letter(i)
                for cell in col_cells:
                    try:
                        if cell.value:
                            cell_text = str(cell.value)
                            if cell_text.startswith('=HYPERLINK('):
                                sep = ',' if ',' in cell_text else ';'
                                display_text = cell_text.split(sep)[-1].strip().strip('"')
                                length = len(display_text)
                            else:
                                length = len(cell_text)
                            max_length = max(max_length, length)
                    except:
                        pass
                ws_main.column_dimensions[col_letter].width = max_length + 2

            # Сохраняем Excel
            excel_path = os.path.join(tmpdir, "Список стипендиантов.xlsx")
            wb.save(excel_path)

            # Копирование файлов
            files_dir = os.path.join(tmpdir, "files")
            os.makedirs(files_dir, exist_ok=True)
            for mid in unique_mids_files:
                base_dir = File(lname).get_scholarship_candidates_files_storage_directory(mid)
                if not base_dir:
                    continue
                fio = next((f.get("fio") for f in files if f.get("mid") == mid), None)
                if not fio:
                    continue
                safe_fio = fio.replace("/", "_").replace("\\", "_")
                for file_info in files:
                    if file_info["mid"] != mid:
                        continue
                    chapter = str(file_info.get("сhapter"))
                    file_name = file_info["file_name"]
                    src = os.path.join(File(lname).getBaseFileDirecotry(), file_info["file_path"], file_name)
                    dst_dir = os.path.join(files_dir, safe_fio, "Глава {}".format(chapter))
                    os.makedirs(dst_dir, exist_ok=True)
                    dst = os.path.join(dst_dir, file_name)
                    shutil.copy2(src, dst)

            # Архивирование
            zip_path = os.path.join(tmpdir, "{}_package.zip".format(type))
            with zipfile.ZipFile(zip_path, 'w', zipfile.ZIP_DEFLATED) as zipf:
                zipf.write(excel_path, arcname="Список стипендиантов.xlsx")
                for root, _, filenames in os.walk(files_dir):
                    for filename in filenames:
                        abs_path = os.path.join(root, filename)
                        rel_path = os.path.relpath(abs_path, tmpdir)
                        zipf.write(abs_path, arcname=rel_path)

            with open(zip_path, "rb") as f:
                zip_bytes = BytesIO(f.read())

        zip_bytes.seek(0)
        return zip_bytes

    @staticmethod
    def create_zip_files_v3(lname, type, result, files, unique_mids_files, tmpdir):
        import re
        org_name = ScholarshipListOptions(lname).get_org_short_name()
        wb = Workbook()
        ws_main = wb.active
        ws_main.title = org_name

        if not result:
            return "Нет данных для экспорта", 204

        cleaned_result = []
        for row in result:
            cleaned = OrderedDict((k, v) for k, v in row.items() if k not in ("mid", "gid"))
            cleaned_result.append(cleaned)

        column_key_map = OrderedDict()
        for key in cleaned_result[0].keys():
            display_key = re.sub(r'\.\d+$', '', key)
            if display_key not in column_key_map:
                column_key_map[display_key] = key

        display_column_keys = list(column_key_map.keys())
        real_column_keys = list(column_key_map.values())

        def build_header_tree(keys):
            tree = OrderedDict()
            for key in keys:
                parts = key.split('__')
                current = tree
                for part in parts:
                    if part not in current:
                        current[part] = OrderedDict()
                    current = current[part]
            return tree

        def get_tree_depth(tree):
            if not tree:
                return 1
            return 1 + max(get_tree_depth(sub) for sub in tree.values())

        def write_headers_from_tree(tree, ws, row=1, col=1, max_depth=None, col_key_list=[]):
            if max_depth is None:
                max_depth = get_tree_depth(tree)

            def recursive_write(node, r, c):
                for key, sub in node.items():
                    start_col = c
                    if sub:
                        c = recursive_write(sub, r + 1, c)
                        ws.merge_cells(start_row=r, start_column=start_col, end_row=r, end_column=c - 1)
                    else:
                        ws.merge_cells(start_row=r, start_column=c, end_row=max_depth, end_column=c)
                        col_key_list.append(c)
                        c += 1
                    ws.cell(row=r, column=start_col, value=key)
                return c

            recursive_write(tree, row, col)
            return max_depth, col_key_list

        header_tree = build_header_tree(display_column_keys)
        max_depth, _ = write_headers_from_tree(header_tree, ws_main)

        for row in ws_main.iter_rows(min_row=1, max_row=max_depth, max_col=len(display_column_keys)):
            for cell in row:
                cell.alignment = Alignment(horizontal='center', vertical='center', wrap_text=True)

        for i, row_data in enumerate(cleaned_result, start=max_depth + 1):
            for j, key in enumerate(real_column_keys, start=1):
                display_key = display_column_keys[j - 1]
                cell = ws_main.cell(row=i, column=j)

                if display_key.lower() == "ссылка":
                    try:
                        mid = int(row_data.get(key, 0))
                        fio = next((f.get("fio") for f in files if f.get("mid") == mid), None)
                        has_files = any(f.get("mid") == mid for f in files)
                        if has_files and fio:
                            safe_fio = fio.replace("/", "_").replace("\\", "_").strip()
                            subfolder = "{}_{}_{}".format(org_name, safe_fio, mid)
                            relative_path = 'files/{}/'.format(subfolder)
                            display_text = "Открыть файлы {}".format(safe_fio)
                            formula = '=HYPERLINK(LEFT(CELL("filename", A1),FIND("[",CELL("filename", A1))-1) & "{}", "{}")'.format(
                                relative_path, display_text)
                            cell.value = formula
                            cell.style = "Hyperlink"
                        else:
                            cell.value = "Отсутствуют"
                    except Exception:
                        cell.value = "Отсутствуют"
                else:
                    cell.value = row_data.get(key, "")

        for i, col_cells in enumerate(ws_main.columns, start=1):
            max_length = 0
            col_letter = get_column_letter(i)
            for cell in col_cells:
                try:
                    if cell.value:
                        cell_text = str(cell.value)
                        if cell_text.startswith('=HYPERLINK('):
                            sep = ',' if ',' in cell_text else ';'
                            display_text = cell_text.split(sep)[-1].strip().strip('"')
                            length = len(display_text)
                        else:
                            length = len(cell_text)
                        max_length = max(max_length, length)
                except:
                    pass
            ws_main.column_dimensions[col_letter].width = 20

        excel_path = os.path.join(tmpdir, "Список кандидатов.xlsx")
        wb.save(excel_path)

        files_dir = os.path.join(tmpdir, "files")
        os.makedirs(files_dir, exist_ok=True)

        for mid in unique_mids_files:
            base_dir = File(lname).get_scholarship_candidates_files_storage_directory(mid)
            if not base_dir:
                continue
            fio = next((f.get("fio") for f in files if f.get("mid") == mid), None)
            if not fio:
                continue
            safe_fio = fio.replace("/", "_").replace("\\", "_").strip()
            subfolder = "{}_{}_{}".format(org_name, safe_fio, mid)
            for file_info in files:
                if file_info["mid"] != mid:
                    continue
                chapter = str(file_info.get("сhapter"))
                file_name = file_info["file_name"]
                src = os.path.join(File(lname).getBaseFileDirecotry(), file_info["file_path"], file_name)
                dst_dir = os.path.join(files_dir, subfolder, "Глава {}".format(chapter))
                os.makedirs(dst_dir, exist_ok=True)
                dst = os.path.join(dst_dir, file_name)
                shutil.copy2(src, dst)

        zip_path = os.path.join(tmpdir, "{}_package.zip".format(type))
        with zipfile.ZipFile(zip_path, 'w', zipfile.ZIP_DEFLATED) as zipf:
            zipf.write(excel_path, arcname="Список кандидатов.xlsx")
            for root, _, filenames in os.walk(files_dir):
                for filename in filenames:
                    abs_path = os.path.join(root, filename)
                    rel_path = os.path.relpath(abs_path, tmpdir)
                    zipf.write(abs_path, arcname=rel_path)

        with open(zip_path, "rb") as f:
            zip_bytes = BytesIO(f.read())

        zip_bytes.seek(0)
        return zip_bytes

    @staticmethod
    def create_zip_files_v4(lname, type, result, files, unique_mids_files):
        import re
        org_name = ScholarshipListOptions(lname).get_org_short_name()

        with tempfile.TemporaryDirectory() as tmpdir:
            wb = Workbook()
            ws_main = wb.active
            ws_main.title = org_name

            if not result:
                return "Нет данных для экспорта", 204

            cleaned_result = []
            for row in result:
                cleaned = OrderedDict((k, v) for k, v in row.items() if k not in ("mid", "gid"))
                cleaned_result.append(cleaned)

            column_key_map = OrderedDict()
            for key in cleaned_result[0].keys():
                display_key = re.sub(r'\.\d+$', '', key)
                if display_key not in column_key_map:
                    column_key_map[display_key] = key

            display_column_keys = list(column_key_map.keys())
            real_column_keys = list(column_key_map.values())

            def build_header_tree(keys):
                tree = OrderedDict()
                for key in keys:
                    parts = key.split('__')
                    current = tree
                    for part in parts:
                        if part not in current:
                            current[part] = OrderedDict()
                        current = current[part]
                return tree

            def get_tree_depth(tree):
                if not tree:
                    return 1
                return 1 + max(get_tree_depth(sub) for sub in tree.values())

            def write_headers_from_tree(tree, ws, row=1, col=1, max_depth=None, col_key_list=[]):
                if max_depth is None:
                    max_depth = get_tree_depth(tree)

                def recursive_write(node, r, c):
                    for key, sub in node.items():
                        start_col = c
                        if sub:
                            c = recursive_write(sub, r + 1, c)
                            ws.merge_cells(start_row=r, start_column=start_col, end_row=r, end_column=c - 1)
                        else:
                            ws.merge_cells(start_row=r, start_column=c, end_row=max_depth, end_column=c)
                            col_key_list.append(c)
                            c += 1
                        ws.cell(row=r, column=start_col, value=key)
                    return c

                recursive_write(tree, row, col)
                return max_depth, col_key_list

            header_tree = build_header_tree(display_column_keys)
            max_depth, _ = write_headers_from_tree(header_tree, ws_main)

            for row in ws_main.iter_rows(min_row=1, max_row=max_depth, max_col=len(display_column_keys)):
                for cell in row:
                    cell.alignment = Alignment(horizontal='center', vertical='center', wrap_text=True)

            for i, row_data in enumerate(cleaned_result, start=max_depth + 1):
                for j, key in enumerate(real_column_keys, start=1):
                    display_key = display_column_keys[j - 1]
                    cell = ws_main.cell(row=i, column=j)

                    if display_key.lower() == "ссылка":
                        try:
                            mid = int(row_data.get(key, 0))
                            fio = next((f.get("fio") for f in files if f.get("mid") == mid), None)
                            has_files = any(f.get("mid") == mid for f in files)
                            if has_files and fio:
                                safe_fio = fio.replace("/", "_").replace("\\", "_").strip()
                                subfolder = "{}_{}_{}".format(org_name, safe_fio, mid)
                                relative_path = 'files/{}/'.format(subfolder)
                                display_text = "Открыть файлы {}".format(safe_fio)
                                formula = '=HYPERLINK(LEFT(CELL("filename", A1),FIND("[",CELL("filename", A1))-1) & "{}", "{}")'.format(
                                    relative_path, display_text)
                                cell.value = formula
                                cell.style = "Hyperlink"
                            else:
                                cell.value = "Отсутствуют"
                        except Exception:
                            cell.value = "Отсутствуют"
                    else:
                        cell.value = row_data.get(key, "")

            # Установка фиксированной ширины всех столбцов на ~2 см (7.56 символов)
            for i in range(1, len(display_column_keys) + 1):
                col_letter = get_column_letter(i)
                ws_main.column_dimensions[col_letter].width = 20

            excel_path = os.path.join(tmpdir, "Список кандидатов.xlsx")
            wb.save(excel_path)

            files_dir = os.path.join(tmpdir, "files")
            os.makedirs(files_dir, exist_ok=True)

            for mid in unique_mids_files:
                base_dir = File(lname).get_scholarship_candidates_files_storage_directory(mid)
                if not base_dir:
                    continue
                fio = next((f.get("fio") for f in files if f.get("mid") == mid), None)
                if not fio:
                    continue
                safe_fio = fio.replace("/", "_").replace("\\", "_").strip()
                subfolder = "{}_{}_{}".format(org_name, safe_fio, mid)
                for file_info in files:
                    if file_info["mid"] != mid:
                        continue
                    chapter = str(file_info.get("сhapter"))
                    file_name = file_info["file_name"]
                    src = os.path.join(File(lname).getBaseFileDirecotry(), file_info["file_path"], file_name)
                    dst_dir = os.path.join(files_dir, subfolder, "Глава {}".format(chapter))
                    os.makedirs(dst_dir, exist_ok=True)
                    dst = os.path.join(dst_dir, file_name)
                    shutil.copy2(src, dst)

            zip_path = os.path.join(tmpdir, "{}_package.zip".format(type))
            with zipfile.ZipFile(zip_path, 'w', zipfile.ZIP_DEFLATED) as zipf:
                zipf.write(excel_path, arcname="Список кандидатов.xlsx")
                for root, _, filenames in os.walk(files_dir):
                    for filename in filenames:
                        abs_path = os.path.join(root, filename)
                        rel_path = os.path.relpath(abs_path, tmpdir)
                        zipf.write(abs_path, arcname=rel_path)

            with open(zip_path, "rb") as f:
                zip_bytes = BytesIO(f.read())

        zip_bytes.seek(0)
        return zip_bytes


