from decimal import Decimal
from typing import Tuple, Union, List, Dict, Any

from flask import current_app
from sqlalchemy import text
from sqlalchemy.orm import sessionmaker

from LMSAPI import db

class GrantIndicatorTree(db.Model):
    __tablename__ = 'grant_indicator_tree'

    id = db.Column(db.Integer, primary_key=True)
    title = db.Column(db.Text, nullable=False)
    parent_id = db.Column(db.Integer, db.ForeignKey('grant_indicator_tree.id', ondelete='CASCADE'))
    level = db.Column(db.Integer, nullable=False, default=1)
    ordering = db.Column(db.Integer, nullable=False, default=0)
    xp_key = db.Column(db.Integer)

    children = db.relationship('GrantIndicatorTree', cascade='all, delete-orphan')


class GrantLeafToPersonnelIndicator(db.Model):
    __tablename__ = 'grant_leaf_to_personnel_indicator'

    id = db.Column(db.Integer, primary_key=True)
    grant_leaf_id = db.Column(db.Integer, db.ForeignKey('grant_indicator_tree.id', ondelete='CASCADE'))
    personnel_indicator_id = db.Column(
        db.Integer, # db.ForeignKey('personnel_indicators.id', ondelete='CASCADE')
    )
    ordering = db.Column(db.Integer, nullable=False, default=0)
    weight = db.Column(db.Numeric, default=1.0)


class GrantIndicatorTreeService:
    def __init__(self, lname, session=None):
        if session:
            self.session = session
            self.lname = lname
        else:
            engine = current_app.ms.db(lname)
            Session = sessionmaker(bind=engine)
            self.session = Session()
            self.lname = lname

    def get_by_id(self, id_: int):
        return self.session.query(GrantIndicatorTree).get(id_)

    def get_by_parent_id(self, parent_id: int):
        return self.session.query(GrantIndicatorTree).filter_by(parent_id=parent_id).first()

    def get_all(self, parent_id=None):
        query = self.session.query(GrantIndicatorTree)
        if parent_id is not None:
            query = query.filter(GrantIndicatorTree.parent_id == parent_id)
        return query.order_by(GrantIndicatorTree.ordering).all()

    def get_tree(self, xp_key):
        query = text("""
                    WITH RECURSIVE tree_path AS (
                      SELECT
                        id,
                        parent_id,
                        title,
                        xp_key,
                        ordering,
                        ARRAY[id] AS path
                      FROM grant_indicator_tree
                      WHERE parent_id IS NULL

                      UNION ALL

                      SELECT
                        c.id,
                        c.parent_id,
                        c.title,
                        c.xp_key,
                        c.ordering,
                        p.path || c.id
                      FROM grant_indicator_tree c
                      JOIN tree_path p ON c.parent_id = p.id
                    ),

                    leaf_nodes AS (
                      SELECT tp.*
                      FROM tree_path tp
                      LEFT JOIN grant_indicator_tree c ON c.parent_id = tp.id
                      WHERE c.id IS NULL
                    ),

                    leaf_with_indicators AS (
                      SELECT
                        l.id AS leaf_id,
                        json_agg(
                          json_build_object(
                            'id', pi.id,
                            'title', pi.name_for_report,
                            'link_weight', pli.weight,
                            'link_ordering', pli.ordering
                          ) ORDER BY pli.ordering
                        ) AS indicators
                      FROM leaf_nodes l
                      JOIN grant_leaf_to_personnel_indicator pli ON pli.grant_leaf_id = l.id
                      JOIN personnel_indicators pi ON pi.id = pli.personnel_indicator_id
                      GROUP BY l.id
                    ),

                    tree_with_indicators AS (
                      SELECT
                        t.id,
                        t.parent_id,
                        t.title,
                        t.xp_key,
                        t.ordering,
                        COALESCE(lwi.indicators, '[]'::json) AS indicators
                      FROM tree_path t
                      LEFT JOIN leaf_with_indicators lwi ON t.id = lwi.leaf_id
                    )

                    SELECT json_agg(node) AS result
                    FROM (
                      SELECT
                        t.id,
                        t.parent_id,
                        t.title,
                        t.xp_key,
                        t.ordering,
                        COALESCE(
                          (
                            SELECT json_agg(
                              json_build_object(
                                'id', ci.id,
                                'parent_id', ci.parent_id,
                                'title', ci.title,
                                'xp_key', ci.xp_key,
                                'ordering', ci.ordering,
                                'child', ci.child
                              ) ORDER BY ci.ordering
                            )
                            FROM (
                              SELECT
                                ci.*,
                                CASE
                                  WHEN ci.indicators IS NOT NULL AND ci.indicators::text <> '[]' THEN ci.indicators
                                  ELSE NULL
                                END AS child
                              FROM tree_with_indicators ci
                              WHERE ci.parent_id = t.id
                            ) ci
                          ),
                          CASE
                            WHEN t.indicators IS NOT NULL AND t.indicators::text <> '[]' THEN t.indicators
                            ELSE NULL
                          END
                        ) AS child
                      FROM tree_with_indicators t
                      WHERE t.parent_id IS NULL AND t.xp_key = :xp_key
                    ) node;
                """)

        result = self.session.execute(query, {"xp_key": xp_key}).fetchone()
        self.session.close()
        return result[0] if result and result[0] is not None else []

    def get_tree_by_mid(self, mid: int, scholarship: int, xp_key: int, education_level: int, prep_struc_category_id: int):
        query = text("""
                    WITH RECURSIVE tree_path AS (
                      SELECT id, parent_id, title, xp_key, ordering, ARRAY[id] AS path
                      FROM grant_indicator_tree
                      WHERE parent_id IS NULL
                      UNION ALL
                      SELECT c.id, c.parent_id, c.title, c.xp_key, c.ordering, p.path || c.id
                      FROM grant_indicator_tree c
                      JOIN tree_path p ON c.parent_id = p.id
                    ),
                    leaf_nodes AS (
                      SELECT tp.* FROM tree_path tp
                      LEFT JOIN grant_indicator_tree c ON c.parent_id = tp.id
                      WHERE c.id IS NULL
                    ),
                    leaf_with_indicators AS (
                      SELECT
                        l.id AS leaf_id,
                        l.path,
                        json_agg(json_build_object(
                          'id', pi.id,
                          'title', pi.name_for_report,
                          'link_weight', pli.weight,
                          'link_ordering', pli.ordering,
                          'count', prv.value,
                          'point', ROUND(prv.value * pli.weight, 2)
                        ) ORDER BY pli.ordering) AS indicators,
                        SUM(prv.value) AS total_count,
                        SUM(ROUND(prv.value * pli.weight, 2)) AS total_points
                      FROM leaf_nodes l
                      JOIN grant_leaf_to_personnel_indicator pli ON pli.grant_leaf_id = l.id
                      JOIN personnel_indicators pi ON pi.id = pli.personnel_indicator_id
                      LEFT JOIN personnel_rating_values prv ON prv.indicator = pi.id
                      WHERE prv.mid = :mid
                        AND prv.period_type = 1
                        AND prv.value IS NOT NULL AND prv.value != 0
                        AND prv.periodend BETWEEN 
                            (SELECT DISTINCT slo.date_from FROM scholarship_settings slo 
                             WHERE slo.xp_key = :xp_key AND slo.scholarship_type_id = :scholarship AND slo.date_from IS NOT NULL 
                             AND slo.education_level = :edu_lvl AND slo.prep_struc_category_id = :prep_cat)
                        AND 
                            (SELECT DISTINCT slo.date_to FROM scholarship_settings slo 
                             WHERE slo.xp_key = :xp_key AND slo.scholarship_type_id = :scholarship AND slo.date_to IS NOT NULL 
                             AND slo.education_level = :edu_lvl AND slo.prep_struc_category_id = :prep_cat)
                      GROUP BY l.id, l.path
                    ),
                    node_aggregates AS (
                      SELECT 
                        t.id, t.parent_id, t.title, t.xp_key, t.ordering, t.path,
                        COALESCE(lwi.indicators, '[]'::json) AS indicators,
                        COALESCE(lwi.total_count, 0) AS leaf_count,
                        COALESCE(lwi.total_points, 0) AS leaf_points,
                        NOT EXISTS (SELECT 1 FROM grant_indicator_tree WHERE parent_id = t.id) AS is_leaf
                      FROM tree_path t
                      LEFT JOIN leaf_with_indicators lwi ON t.id = lwi.leaf_id
                    ),
                    leaf_aggregates AS (
                      SELECT unnest(path) AS node_id, total_count, total_points
                      FROM leaf_with_indicators
                    ),
                    node_totals AS (
                      SELECT node_id, SUM(total_count) AS total_count, SUM(total_points) AS total_points
                      FROM leaf_aggregates GROUP BY node_id
                    ),
                    final_nodes AS (
                      SELECT
                        n.id, n.parent_id, n.title, n.xp_key, n.ordering, n.path,
                        n.indicators, n.is_leaf,
                        COALESCE(nt.total_count, 0) AS total_count,
                        COALESCE(nt.total_points, 0) AS total_points
                      FROM node_aggregates n
                      LEFT JOIN node_totals nt ON n.id = nt.node_id
                    ),
                    tree_result AS (
                      SELECT json_agg(node) AS tree
                      FROM (
                        SELECT
                          t.id, t.title, t.xp_key, t.ordering,
                          t.total_count AS count,
                          t.total_points AS points,
                          COALESCE((
                            SELECT json_agg(json_build_object(
                              'id', ci.id,
                              'title', ci.title,
                              'xp_key', ci.xp_key,
                              'ordering', ci.ordering,
                              'count', ci.total_count,
                              'points', ci.total_points,
                              'child', ci.child
                            ) ORDER BY ci.ordering)
                            FROM (
                              SELECT ci.*,
                                     CASE
                                       WHEN ci.indicators IS NOT NULL AND json_array_length(ci.indicators) > 0 AND ci.is_leaf THEN ci.indicators
                                       ELSE NULL
                                     END AS child
                              FROM final_nodes ci
                              WHERE ci.parent_id = t.id
                            ) ci
                          ),
                          CASE
                            WHEN t.indicators IS NOT NULL AND json_array_length(t.indicators) > 0 AND t.is_leaf THEN t.indicators
                            ELSE NULL
                          END) AS child
                        FROM final_nodes t
                        WHERE t.parent_id IS NULL
                      ) node
                    ),
                    total_sums AS (
                      SELECT SUM(total_count) AS total_count, SUM(total_points) AS total_point
                      FROM final_nodes
                      WHERE parent_id IS NULL
                    ),
                    count_year AS (
                    SELECT 
                        COALESCE(SUM(CASE 
                            WHEN prv.periodend BETWEEN 
                                (SELECT DISTINCT (slo.date_to - INTERVAL '1 year')::date 
                                 FROM scholarship_settings slo 
                                 WHERE slo.xp_key = :xp_key 
                                     AND slo.scholarship_type_id = :scholarship
                                     AND slo.date_to IS NOT NULL 
                                     AND slo.education_level = :edu_lvl
                                     AND slo.prep_struc_category_id = :prep_cat)
                                AND 
                                (SELECT DISTINCT slo.date_to 
                                 FROM scholarship_settings slo 
                                 WHERE slo.xp_key = :xp_key 
                                   AND slo.scholarship_type_id = :scholarship
                                   AND slo.date_to IS NOT NULL 
                                   AND slo.education_level = :edu_lvl
                                   AND slo.prep_struc_category_id = :prep_cat) 
                            THEN prv.value
                            ELSE NULL 
                        END), 0) AS annual_number
                    FROM grant_indicator_tree git
                    JOIN grant_leaf_to_personnel_indicator gpi ON git.id = gpi.grant_leaf_id
                    JOIN personnel_indicators pi ON pi.id = gpi.personnel_indicator_id
                    JOIN personnel_rating_values prv ON prv.indicator = pi.id
                    WHERE 
                            prv.mid = :mid
                            AND pi.forgrant IS TRUE
                            AND prv.period_type = 1
                            AND prv.value IS NOT NULL 
                            AND prv.value != 0
                        GROUP BY 
                            prv.mid
                    ),
					min_achievements AS (
					  SELECT slo.min_achievements
					  FROM scholarship_settings slo
					  WHERE slo.xp_key = :xp_key
						AND slo.scholarship_type_id = :scholarship
						AND slo.date_from IS NOT NULL 
						AND slo.education_level = :edu_lvl
						AND slo.prep_struc_category_id = :prep_cat
					  LIMIT 1
					)
					SELECT 
					  CASE 
						WHEN (SELECT total_count FROM total_sums) < (SELECT min_achievements FROM min_achievements)
						THEN NULL
						ELSE json_build_object(
						  'record_count', (SELECT total_count FROM total_sums),
						  'record_points', (SELECT total_point FROM total_sums),
						  'record_count_year', (SELECT annual_number FROM count_year),
						  'tree', (SELECT tree FROM tree_result)
						)
					  END AS result;
                """)

        result = self.session.execute(query, {
            "mid": mid,
            "xp_key": xp_key,
            "scholarship": scholarship,
            "edu_lvl": education_level,
            "prep_cat": prep_struc_category_id
        }).fetchone()

        self.session.close()
        return result[0]

    def get_grants_for_mids(self, xp_key, education_level, prep_struc_category_id, mids: list):
        if not mids:
            return []

        sql = """
            WITH RECURSIVE 
            -- Сначала получаем достижения пользователей
            user_achievements AS (
                SELECT 
                    prv.mid AS mid,
                    git.id AS id,
                    git.title AS title,
                    git.parent_id AS parent_id,
                    git.level AS level,
                    git.ordering AS ordering,
                    SUM(prv.value) AS count,
                    SUM(ROUND(prv.value * glpi.weight, 2)) AS points,
                    json_agg(
                      json_build_object(
                        'id', pi.id,
                        'title', pi.name_for_report,
                        'link_ordering', glpi.ordering,
                        'count', prv.value,
                        'point', ROUND(prv.value * glpi.weight, 2)
                      ) ORDER BY glpi.ordering
                    ) AS indicators
                FROM personnel_indicators pi
                JOIN grant_leaf_to_personnel_indicator glpi ON glpi.personnel_indicator_id = pi.id
                JOIN grant_indicator_tree git ON git.id = glpi.grant_leaf_id
                JOIN personnel_rating_values prv ON prv.indicator = pi.id
                WHERE prv.mid = ANY(:mids)
                    AND prv.period_type = 1
                    AND prv.value IS NOT NULL 
                    AND prv.value != 0
                    AND prv.periodend BETWEEN 
                        (SELECT DISTINCT slo.date_from 
                         FROM scholarship_list_options slo 
                         WHERE slo.xp_key = :xp_key 
                         AND slo.date_from IS NOT NULL 
                         AND slo.education_level = :education_level
                         AND slo.prep_struc_category_id = :prep_struc_category_id) 
                        AND 
                        (SELECT DISTINCT slo.date_to 
                         FROM scholarship_list_options slo 
                         WHERE slo.xp_key = :xp_key 
                         AND slo.date_to IS NOT NULL 
                         AND slo.education_level = :education_level
                         AND slo.prep_struc_category_id = :prep_struc_category_id)
                GROUP BY prv.mid, git.id
                ORDER BY prv.mid, git.ordering
            ),
            tree AS (
                SELECT
                    ua.*
                FROM user_achievements ua
                WHERE ua.parent_id is NULL
                
                UNION ALL
                
                -- Рекурсивный случай: дочерние элементы
                SELECT 
                    ua.mid AS mid,
                    git.id AS id,
                    git.title AS title,
                    git.parent_id AS parent_id,
                    git.level AS level,
                    git.ordering AS ordering,
                    SUM(ua.count) AS count,
                    SUM(ua.points) AS points,
                    json_agg(
                      json_build_object(
                        'id', ua.id,
                        'title', ua.title,
                        'ordering', ua.ordering,
                        'count', ua.count,
                        'points', ua.points,
                        'indicators', ua.indicators
                      ) ORDER BY ua.ordering
                    ) AS indicators
                FROM user_achievements ua
                JOIN grant_indicator_tree git ON git.id = ua.parent_id
                WHERE ua.parent_id is not NULL
                GROUP BY ua.mid, git.id
            ),
            final_tree AS (
            SELECT 
                tree.mid, 
                SUM(tree.count) AS total_count,
                SUM(tree.points) AS total_point,
                json_agg(
                  json_build_object(
                    'id', tree.id,
                    'title', tree.title,
                    'ordering', tree.ordering,
                    'count', tree.count,
                    'points', tree.points,
                    'indicators', tree.indicators
                  ) ORDER BY tree.ordering
                ) AS tree
            FROM tree
            GROUP BY tree.mid
            ),
            count_year AS (
            SELECT 
                COALESCE(SUM(CASE 
                    WHEN prv.periodend BETWEEN 
                        (SELECT DISTINCT (slo.date_to - INTERVAL '1 year')::date 
                         FROM scholarship_list_options slo 
                         WHERE slo.xp_key = :xp_key 
                             AND slo.date_to IS NOT NULL 
                             AND slo.education_level = :education_level
                             AND slo.prep_struc_category_id = :prep_struc_category_id)
                        AND 
                        (SELECT DISTINCT slo.date_to 
                         FROM scholarship_list_options slo 
                         WHERE slo.xp_key = :xp_key 
                           AND slo.date_to IS NOT NULL 
                           AND slo.education_level = :education_level
                           AND slo.prep_struc_category_id = :prep_struc_category_id) 
                    THEN prv.value
                    ELSE NULL 
                END), 0) AS annual_number,
                prv.mid
            FROM grant_indicator_tree git
            JOIN grant_leaf_to_personnel_indicator gpi ON git.id = gpi.grant_leaf_id
            JOIN personnel_indicators pi ON pi.id = gpi.personnel_indicator_id
            JOIN personnel_rating_values prv ON prv.indicator = pi.id
            WHERE 
                    prv.mid = ANY(:mids)
                    AND pi.forgrant IS TRUE
                    AND prv.period_type = 1
                    AND prv.value IS NOT NULL 
                    AND prv.value != 0
                GROUP BY 
                    prv.mid
            )
            SELECT ft.*, cy.annual_number AS record_count_year 
            FROM final_tree ft
            JOIN count_year cy ON ft.mid = cy.mid
            WHERE ft.total_count > 1::numeric
        """

        query = self.session.execute(text(sql), {"mids": mids, "xp_key": xp_key, "education_level": education_level, "prep_struc_category_id": prep_struc_category_id})
        res = [dict(zip(tuple(query.keys()), i)) for i in query.cursor] # JSON array
        return res if res else None

    def get_list_stipend_grants(self):
        sql = """
            WITH RECURSIVE indicator_tree AS (
              -- Шаг 1: Получаем записи первого уровня, где id_group = 0
              SELECT 
                pi.degree_tree_id AS current_id,
                pi.title AS current_title,
                pi.degree_tree_id_group AS previous_id_group,
                pi.degree_tree_id AS first_level_id,
                pi.title AS first_level_title,
                1 AS level,
                pi.weight AS cumulative_weight -- Начинаем с weight первого уровня
              FROM personnel_indicators pi
              JOIN rating_schema rs ON rs.id = pi.schema
              WHERE rs.stipend IS TRUE  
                AND pi.degree_tree_id_group = 0
              UNION ALL
              -- Шаг 2: Рекурсивно добавляем потомков, умножая вес родительского уровня на текущий
              SELECT 
                pi.degree_tree_id AS current_id,
                pi.title AS current_title,
                pi.degree_tree_id_group AS previous_id_group,
                it.first_level_id,
                it.first_level_title,
                it.level + 1 AS level,
                it.cumulative_weight * COALESCE(pi.weight, 1) AS cumulative_weight -- Накопление веса на текущем уровне
              FROM personnel_indicators pi
              JOIN rating_schema rs ON rs.id = pi.schema
              JOIN indicator_tree it ON pi.degree_tree_id_group = it.current_id
              WHERE rs.stipend IS TRUE
             )
              SELECT distinct
                it.current_id AS indicator_id,
                pi2.name_for_report AS indicator_title
              FROM indicator_tree it
              LEFT JOIN personnel_indicators pi ON it.previous_id_group = pi.degree_tree_id
              JOIN rating_schema rs ON rs.id = pi.schema and rs.stipend IS TRUE
              LEFT JOIN personnel_indicators pi2 ON it.current_id = pi2.degree_tree_id
              JOIN rating_schema rs1 ON rs1.id = pi2.schema and rs1.stipend IS TRUE
              WHERE pi2.forgrant IS TRUE
        """

        query = self.session.execute(text(sql))
        res = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]  # JSON array
        return res if res else None

    def get_descendants(self, id: int):
        query = text("""
            WITH RECURSIVE descendants AS (
                SELECT id, parent_id
                FROM grant_indicator_tree
                WHERE id = :id
                UNION ALL
                SELECT g.id, g.parent_id
                FROM grant_indicator_tree g
                JOIN descendants d ON g.parent_id = d.id
            )
            SELECT id FROM descendants WHERE id != :id;
        """)
        result = self.session.execute(query, {"id": id})
        return [{"id": row[0]} for row in result]

    def create(self, title, parent_id=None, level=1, ordering=0, xp_key=None):
        item = GrantIndicatorTree(
            title=title,
            parent_id=parent_id,
            level=level,
            ordering=ordering,
            xp_key=xp_key
        )
        self.session.add(item)
        self.session.flush()
        return item

    def update(self, id_, **kwargs):
        item = self.get_by_id(id_)
        if not item:
            raise ValueError("Элемент не найден")
        for key, value in kwargs.items():
            if hasattr(item, key):
                setattr(item, key, value)
        self.session.flush()
        return item

    def delete(self, id_):
        item = self.get_by_id(id_)
        if not item:
            raise ValueError("Элемент не найден")
        self.session.delete(item)
        self.session.commit()

    def delete_many(self, ids: list):
        self.session.execute(
            text("DELETE FROM grant_indicator_tree WHERE id = ANY(:ids)"),
            {"ids": ids}
        )

    def create_grant_indicator_tree(self, title, parent_id=None, level=1, ordering=0, xp_key=None, personnel_indicator_ids=[]):
        try:
            # Создаем новый узел в grant_indicator_tree
            new_node = self.create(
                title=title,
                parent_id=parent_id if parent_id != 0 else None,
                level=level,
                ordering=ordering,
                xp_key=xp_key
            )

            links = []
            # Если personnel_indicator_ids переданы и parent_id не имеет связей
            if personnel_indicator_ids:
                # Для каждого personnel_indicator_id вычисляем вес и вставляем в grant_leaf_to_personnel_indicator
                for pi in personnel_indicator_ids:
                    personnel_id = pi.get("personnel_indicator_id")
                    ordering_pi = pi.get("ordering", 0)

                    # Выполняем рекурсивный запрос для вычисления веса
                    res = GrantLeafToPersonnelIndicatorService(self.lname).get_weight_by_personnel_indicator_id(personnel_id)
                    weight = Decimal(res.cumulative_weight) if res and res.cumulative_weight else Decimal(1.0)

                    link = GrantLeafToPersonnelIndicatorService(self.lname, session=self.session).create(
                        grant_leaf_id=new_node.id,
                        personnel_indicator_id=personnel_id,
                        ordering=ordering_pi,
                        weight=weight
                    )
                    links.append({
                        "id": link.id,
                        "grant_leaf_id": link.grant_leaf_id,
                        "personnel_indicator_id": link.personnel_indicator_id,
                        "ordering": link.ordering,
                        "weight": float(link.weight) if link.weight else None,
                    })
            return {
                "id": new_node.id,
                "title": new_node.title,
                "parent_id": new_node.parent_id,
                "level": new_node.level,
                "ordering": new_node.ordering,
                "personnel_indicator_ids": links
            }

        except Exception as e:
            self.session.rollback()
            print(e)
            raise Exception("При создании ветки дерева произошла ошибка")

        finally:
            self.session.commit()
            self.session.close()

    def update_grant_indicator_tree(self, id_, title, parent_id=None, level=1, ordering=0, xp_key=None, personnel_indicator_ids=[]):
        try:
            # Обновление узла
            updated_node = self.update(
                id_,
                title=title,
                parent_id=parent_id if parent_id != 0 else None,
                level=level,
                ordering=ordering,
                xp_key=xp_key
            )

            new_links = []
            # Очистка старых связей и добавление новых (если переданы)
            if personnel_indicator_ids is not None:
                # Удаляем старые связи
                existing_links = GrantLeafToPersonnelIndicatorService(self.lname).get_all(grant_leaf_id=id_)
                for link in existing_links:
                    GrantLeafToPersonnelIndicatorService(self.lname, session=self.session).delete(link.id)

                # Добавляем новые связи
                for pi in personnel_indicator_ids:
                    personnel_id = pi.get("personnel_indicator_id")
                    ordering_pi = pi.get("ordering", 0)
                    res = GrantLeafToPersonnelIndicatorService(self.lname).get_weight_by_personnel_indicator_id(personnel_id)
                    weight = Decimal(res.cumulative_weight) if res and res.cumulative_weight else Decimal(1.0)
                    new_link = GrantLeafToPersonnelIndicatorService(self.lname, session=self.session).create(
                        grant_leaf_id=id_,
                        personnel_indicator_id=personnel_id,
                        ordering=ordering_pi,
                        weight=weight
                    )
                    new_links.append({
                        "id": new_link.id,
                        "grant_leaf_id": new_link.grant_leaf_id,
                        "personnel_indicator_id": new_link.personnel_indicator_id,
                        "ordering": new_link.ordering,
                        "weight": float(new_link.weight) if new_link.weight else None,
                    })

            return {
                "id": updated_node.id,
                "title": updated_node.title,
                "parent_id": updated_node.parent_id,
                "level": updated_node.level,
                "ordering": updated_node.ordering,
                "personnel_indicator_ids": new_links
            }

        except Exception as e:
            self.session.rollback()
            print(e)
            raise Exception("При обновлении ветки дерева произошла ошибка")

        finally:
            self.session.commit()
            self.session.close()

    def delete_grant_indicator_tree(self, all_ids_to_delete: list):
        try:
            # Удаление связей в grant_leaf_to_personnel_indicator
            GrantLeafToPersonnelIndicatorService(self.lname, session=self.session).delete_leaf_links(all_ids_to_delete)
            # Удаление узлов дерева
            self.delete_many(all_ids_to_delete)
        except Exception as e:
            self.session.rollback()
            print(e)
            raise Exception("При удалении ветки дерева произошла ошибка")

        finally:
            self.session.commit()
            self.session.close()

class GrantLeafToPersonnelIndicatorService:
    def __init__(self, lname, session=None):
        if session:
            self.session = session
            self.lname = lname
        else:
            engine = current_app.ms.db(lname)
            Session = sessionmaker(bind=engine)
            self.session = Session()
            self.lname = lname

    def get_by_id(self, id_: int):
        return self.session.query(GrantLeafToPersonnelIndicator).get(id_)

    def get_all(self, grant_leaf_id=None):
        query = self.session.query(GrantLeafToPersonnelIndicator)
        if grant_leaf_id is not None:
            query = query.filter(GrantLeafToPersonnelIndicator.grant_leaf_id == grant_leaf_id)
        return query.order_by(GrantLeafToPersonnelIndicator.ordering).all()

    def get_links_by_grant_leaf_id(self, grant_leaf_id: int):
        return self.session.query(GrantLeafToPersonnelIndicator) \
            .filter(GrantLeafToPersonnelIndicator.grant_leaf_id == grant_leaf_id) \
            .first()

    def get_weight_by_personnel_indicator_id(self, personnel_indicator_id: int):
        weight_query = text("""
            WITH RECURSIVE weight_chain AS (
              SELECT
                pi.id,
                pi.id_group,
                COALESCE(pi.weight, 2) AS weight
              FROM personnel_indicators pi
              WHERE pi.id = :personnel_id

              UNION ALL

              SELECT
                parent.id,
                parent.id_group,
                COALESCE(parent.weight, 1)
              FROM personnel_indicators parent
              JOIN weight_chain wc ON wc.id_group = parent.id
            )
            SELECT
              ROUND(EXP(SUM(LN(weight))), 4) AS cumulative_weight
            FROM weight_chain;
        """)

        return self.session.execute(weight_query, {"personnel_id": personnel_indicator_id}).fetchone()

    def create(self, grant_leaf_id, personnel_indicator_id, ordering=0, weight=1.0):
        item = GrantLeafToPersonnelIndicator(
            grant_leaf_id=grant_leaf_id,
            personnel_indicator_id=personnel_indicator_id,
            ordering=ordering,
            weight=weight
        )
        self.session.add(item)
        self.session.flush()
        return item

    def update(self, id_, **kwargs):
        item = self.get_by_id(id_)
        if not item:
            raise ValueError("Элемент не найден")
        for key, value in kwargs.items():
            if hasattr(item, key):
                setattr(item, key, value)
        self.session.commit()
        return item

    def delete(self, id_):
        item = self.get_by_id(id_)
        if not item:
            raise ValueError("Элемент не найден")
        self.session.delete(item)

    def delete_leaf_links(self, ids: list):
        self.session.execute(
            text("DELETE FROM grant_leaf_to_personnel_indicator WHERE grant_leaf_id = ANY(:ids)"),
            {"ids": ids}
        )
