from flask import Blueprint, request, jsonify
from flask_cors import CORS

from LMSAPI.api.Models.CriteriaTeacherEvaluation import CriteriaTeacherEvaluation, CriteriaTeacherEvaluationService
from LMSAPI.api.Views.TokenAPI import auth
from LMSAPI.api.utils.access_utils import user_permission_modes

criteria_api = Blueprint('criteria_api', __name__)

CORS(criteria_api)


@criteria_api.route("/lms/api/v1.0/<lname>/criteria", methods=["GET"])
@auth.login_required
def list_criteria(lname):
    """Получить список критериев"""
    include_archived = request.args.get("include_archived", "false").lower() == "true"
    return jsonify(CriteriaTeacherEvaluationService(lname).get_all(include_archived))


@criteria_api.route("/lms/api/v1.0/<lname>/criteria", methods=["POST"])
@auth.login_required
@user_permission_modes("Справочники", "Критерии оценки качеств преподавателей для опроса обучающихся", ["Нет", "Чтение"])
def create_criteria(lname):
    """Добавить новый критерий"""
    data = request.get_json()
    name = data.get("name", "").strip()

    if not name:
        return jsonify({"error": "Название критерия обязательно"}), 400

    try:
        new_criteria = CriteriaTeacherEvaluationService(lname).create_criteria(
            name=name,
            is_archive=data.get("is_archive", False)
        )
        return jsonify(new_criteria.to_dict()), 201
    except ValueError as e:
        return jsonify({"error": str(e)}), 400


@criteria_api.route("/lms/api/v1.0/<lname>/criteria/<int:criteria_id>", methods=["PUT"])
@auth.login_required
@user_permission_modes("Справочники", "Критерии оценки качеств преподавателей для опроса обучающихся", ["Нет", "Чтение"])
def update_criteria_api(lname, criteria_id):
    data = request.get_json()
    new_name = data.get("name", None)
    new_is_archive = data.get("is_archive", None)

    try:
        new_name = new_name.strip()
        if not new_name:
            raise ValueError("Название критерия не может быть пустым")

        if not isinstance(new_is_archive, bool):
            raise ValueError("Поле is_archive должно быть логическим (true/false)")

        updated, error = CriteriaTeacherEvaluationService(lname).update_criteria(
            criteria_id=criteria_id,
            new_name=new_name,
            new_is_archive=new_is_archive
        )
        if error:
            result = updated.to_dict()
            result["error"] = error
            return jsonify(result)
        return jsonify(updated.to_dict())
    except ValueError as e:
        return jsonify({"error": str(e)}), 400
