# coding=UTF-8

from flask import jsonify, request, g
from flask import Blueprint

from LMSAPI.api.Models.IndividualPlans import IndividualPlans
from LMSAPI.api.Views.TokenAPI import auth
from flask_cors import CORS

from LMSAPI.api.utils.access_utils import user_permission_modes

individual_plans_api = Blueprint("individual_plans_api", __name__)

CORS(individual_plans_api)


@individual_plans_api.route("/lms/api/v1.0/<lname>/individual_plans", methods=["GET"])
@auth.login_required
def get_individual_plans_new(lname):
    xp_key = request.args.get("xp_key")
    did = request.args.get("did", type=int, default=0)

    if not xp_key:
        return jsonify({"error": "Missing required parameters"}), 400

    full_access = 0
    if g.user.access_modes["Справочники"]["Индивидуальные планы работы преподавателя. Администратор"] == "Полный":
        full_access = 2
    elif g.user.access_modes["Справочники"]["Индивидуальные планы работы преподавателя. Администратор"] == "Чтение":
        full_access = 1

    full_podr = 0
    if g.user.access_modes["Справочники"]["Индивидуальные планы работы преподавателя. Подразделение"] == "Полный":
        full_podr = 2
    elif g.user.access_modes["Справочники"]["Индивидуальные планы работы преподавателя. Подразделение"] == "Чтение":
        full_podr = 1

    full_main = 0
    if g.user.access_modes["Справочники"]["Индивидуальные планы работы преподавателя"] == "Полный":
        full_main = 2
    elif g.user.access_modes["Справочники"]["Индивидуальные планы работы преподавателя"] == "Чтение":
        full_main = 1

    edit_access = 0
    if g.user.access_modes["Справочники"]["Индивидуальные планы работы преподавателя. Проверка и утверждение"] == "Полный":
        edit_access = 1

    result = IndividualPlans().get_individual_plans(lname, did, full_access, full_podr, full_main, edit_access, g.user.mid, xp_key)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@individual_plans_api.route("/lms/api/v1.0/<lname>/individual_plans/filters", methods=["GET"])
@auth.login_required
def get_individual_plans_filters(lname):
    result = None
    if g.user.access_modes["Справочники"]["Индивидуальные планы работы преподавателя. Администратор"] != "Нет":
        result = IndividualPlans().get_individual_plans_filters_admin(lname, g.user.mid)
    elif g.user.access_modes["Справочники"]["Индивидуальные планы работы преподавателя"] != "Нет":
        result = IndividualPlans().get_individual_plans_filters(lname, g.user.mid)

    if result is None:
        return jsonify("У пользователя нет роли"), 404

    return jsonify(result)


@individual_plans_api.route("/lms/api/v1.0/<lname>/individual_plans", methods=["POST"])
@auth.login_required
def insert_individual_plans(lname):
    data = request.get_json()
    if data is None:
        return (
            jsonify(
                {
                    "error": "The request body must be populated with the xp_key and cid fields"
                }
            ),
            400,
        )
    xp_key = data.get("xp_key")
    cid = data.get("cid")

    if not data.get("xp_key") or not data.get("cid"):
        return jsonify({"error": "Missing required parameters"}), 400

    result = IndividualPlans().insert_individual_plans(lname, xp_key, cid)

    if result is None:
        return jsonify(success=False, message="Insertion failed"), 500

    return jsonify(success=True, message="Records inserted successfully"), 201


@individual_plans_api.route(
    "/lms/api/v1.0/<lname>/individual_plans/<int:plan_id>", methods=["PUT"]
)
@auth.login_required
def update_individual_plan(lname, plan_id):
    data = request.get_json()

    result = IndividualPlans().update_individual_plan(lname, plan_id, data)

    if result is None:
        return jsonify(success=False, message="Individual plan not found"), 404

    return jsonify(success=True, plan=result)


@individual_plans_api.route("/lms/api/v1.0/<lname>/individual_plans/<int:ip_id>/main", methods=["GET"])
@auth.login_required
@user_permission_modes("Справочники", "Индивидуальные планы работы преподавателя", ["Нет"])
def get_main(lname, ip_id):
    """ Вкладка 'Главная' """

    result = IndividualPlans().get_main(lname, ip_id)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@individual_plans_api.route("/lms/api/v1.0/<lname>/individual_plans/<int:ip_id>/I", methods=["GET"])
@auth.login_required
@user_permission_modes("Справочники", "Индивидуальные планы работы преподавателя", ["Нет"])
def get_tasks_for_the_academic_year(lname, ip_id):
    """ Вкладка 'I. Задачи на учебный год' """

    result = IndividualPlans().get_tasks_for_the_academic_year(lname, ip_id)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@individual_plans_api.route("/lms/api/v1.0/<lname>/individual_plans/<int:ip_id>/I", methods=["POST"])
@auth.login_required
@user_permission_modes("Справочники", "Индивидуальные планы работы преподавателя", ["Нет"])
def create_task_for_the_academic_year(lname, ip_id):
    """ Вкладка 'I. Задачи на учебный год' """

    data = request.get_json()
    if not data:
        return jsonify({"error": "Missing required task data"}), 400

    new_id = IndividualPlans().create_task_for_the_academic_year(lname, ip_id, data)

    return jsonify({"success": True, "new_id": new_id}), 201


@individual_plans_api.route("/lms/api/v1.0/<lname>/individual_plans/<int:ip_id>/I/<int:task_id>", methods=["PUT"])
@auth.login_required
@user_permission_modes("Справочники", "Индивидуальные планы работы преподавателя", ["Нет"])
def update_tasks_for_the_academic_year(lname, ip_id, task_id):
    """ Вкладка 'I. Задачи на учебный год' """

    updates = request.get_json()
    if not updates:
        return jsonify({"error": "Missing JSON payload"}), 400

    try:
        IndividualPlans().update_tasks_for_the_academic_year(lname, task_id, updates)
        return jsonify({"success": True, "message": "Task updated successfully"}), 200
    except Exception as e:
        return jsonify({"error": str(e)}), 500


@individual_plans_api.route("/lms/api/v1.0/<lname>/individual_plans/<int:ip_id>/I/<int:task_id>", methods=["DELETE"])
@auth.login_required
@user_permission_modes("Справочники", "Индивидуальные планы работы преподавателя", ["Нет"])
def delete_tasks_for_the_academic_year(lname, ip_id, task_id):
    """ Вкладка 'I. Задачи на учебный год' """

    try:
        IndividualPlans().delete_tasks_for_the_academic_year(lname, task_id)
        return jsonify({"success": True, "message": "Task deleted successfully"}), 200
    except Exception as e:
        return jsonify({"error": str(e)}), 500


@individual_plans_api.route("/lms/api/v1.0/<lname>/individual_plans/<int:ip_id>/II", methods=["GET"])
@auth.login_required
@user_permission_modes("Справочники", "Индивидуальные планы работы преподавателя", ["Нет"])
def get_annual_budget_of_office_time(lname, ip_id):
    """ Вкладка 'II. Годовой бюджет служебного (рабочего) времени' """

    result = IndividualPlans().get_annual_budget_of_office_time(lname, ip_id)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@individual_plans_api.route("/lms/api/v1.0/<lname>/individual_plans/<int:ip_id>/<int:mid>/<int:xp_key>/III/times",
                            methods=["GET"])
@auth.login_required
@user_permission_modes("Справочники", "Индивидуальные планы работы преподавателя", ["Нет"])
def get_educational_work_times(lname, ip_id, mid, xp_key):
    """ Вкладка 'III. Учебная работа' """

    result = IndividualPlans().get_educational_work_times(lname, ip_id=ip_id, mid=mid, xp_key=xp_key)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@individual_plans_api.route("/lms/api/v1.0/<lname>/individual_plans/<int:ip_id>/<int:mid>/<int:xp_key>/III",
                            methods=["GET"])
@auth.login_required
@user_permission_modes("Справочники", "Индивидуальные планы работы преподавателя", ["Нет"])
def get_educational_work_table(lname, ip_id, mid, xp_key):
    """ Вкладка 'III. Учебная работа' """

    result = IndividualPlans().get_educational_work_table(lname, ip_id=ip_id, mid=mid, xp_key=xp_key)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@individual_plans_api.route("/lms/api/v1.0/<lname>/individual_plans/<int:ip_id>/group", methods=["GET"])
@auth.login_required
@user_permission_modes("Справочники", "Индивидуальные планы работы преподавателя", ["Нет"])
def get_group(lname, ip_id):
    """ Уч группы для вкладки 'III. Методическая работа' """

    result = IndividualPlans().get_group(lname, ip_id=ip_id)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@individual_plans_api.route("/lms/api/v1.0/<lname>/individual_plans/<int:ip_id>/<int:mid>/<int:mode>", methods=["GET"])
@auth.login_required
@user_permission_modes("Справочники", "Индивидуальные планы работы преподавателя", ["Нет"])
def get_IV_V_VI_VII(lname, ip_id, mid, mode):
    """
        Вкладка:
            'IV. Методическая работа'
            'V. Научная (найчно-иследовательская) работа'
            'VI. Воспитательная работа'
            'VII. Профессионально-должностная подготовка и другие виды деятельности'
    """

    result = IndividualPlans().get_IV_V_VI_VII(lname, ip_id=ip_id, mid=mid, mode=mode)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@individual_plans_api.route("/lms/api/v1.0/<lname>/individual_plans/III_IV_V_VI_VII/<int:kid>/<int:ip_id>",
                            methods=["PUT"])
@auth.login_required
@user_permission_modes("Справочники", "Индивидуальные планы работы преподавателя", ["Нет"])
def update_III_IV_V_VI_VII(lname, kid, ip_id):
    """ Редактирование записи для вкладок III, IV, V, VI, VII """

    updates = request.get_json()
    if not updates:
        return jsonify({"error": "Missing JSON payload"}), 400

    try:
        IndividualPlans.update_III_IV_V_VI_VII(lname, kid, ip_id, updates)
        return jsonify({"success": True, "message": "Record updated successfully"}), 200
    except Exception as e:
        return jsonify({"error": str(e)}), 500


@individual_plans_api.route("/lms/api/v1.0/<lname>/individual_plans/<int:ip_id>/<int:mid>/VIII", methods=["GET"])
@auth.login_required
@user_permission_modes("Справочники", "Индивидуальные планы работы преподавателя", ["Нет"])
def get_content_planned_works(lname, ip_id, mid):
    """ Вкладка 'VIII. Содержание планируемых работ' """
    hiden = request.args.get("hiden", type=bool)

    result = IndividualPlans().get_content_planned_works(lname, ip_id=ip_id, mid=mid, hiden=hiden)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@individual_plans_api.route("/lms/api/v1.0/<lname>/individual_plans/<int:ip_id>/VIII", methods=["POST"])
@auth.login_required
@user_permission_modes("Справочники", "Индивидуальные планы работы преподавателя", ["Нет"])
def create_content_planned_works(lname, ip_id):
    """ Создание записи для вкладки 'VIII. Содержание планируемых работ'. """

    data = request.get_json()
    if not data:
        return jsonify({"error": "Missing JSON payload"}), 400

    try:
        new_id = IndividualPlans.create_content_planned_works(lname, data)
        return jsonify({"success": True, "id": new_id}), 201
    except Exception as e:
        return jsonify({"error": str(e)}), 500


@individual_plans_api.route("/lms/api/v1.0/<lname>/individual_plans/<int:ip_id>/<int:mid>/VIII/<int:work_id>",
                            methods=["PUT"])
@auth.login_required
@user_permission_modes("Справочники", "Индивидуальные планы работы преподавателя", ["Нет"])
def update_content_planned_work(lname, ip_id, mid, work_id):
    """ Редактирование записи для 'VIII. Содержание планируемых работ' """

    updates = request.get_json()
    if not updates:
        return jsonify({"error": "Missing JSON payload"}), 400

    try:
        IndividualPlans.update_content_planned_work(lname, work_id, updates)
        return jsonify({"success": True, "message": "Record updated successfully"}), 200
    except Exception as e:
        return jsonify({"error": str(e)}), 500


@individual_plans_api.route("/lms/api/v1.0/<lname>/individual_plans/<int:ip_id>/<int:mid>/VIII/<int:work_id>",
                            methods=["DELETE"])
@auth.login_required
@user_permission_modes("Справочники", "Индивидуальные планы работы преподавателя", ["Нет"])
def delete_content_planned_work(lname, ip_id, mid, work_id):
    """ Удаление записи для 'VIII. Содержание планируемых работ' """
    try:
        IndividualPlans.delete_content_planned_work(lname, work_id)
        return jsonify({"success": True, "message": "Record deleted successfully"}), 200
    except Exception as e:
        return jsonify({"error": str(e)}), 500
