# coding=UTF-8
import sys
from decimal import Decimal

from flask import jsonify, request, json, g
from flask import Blueprint

from LMSAPI.api.Models.EnrollAnswers import EnrollAnswers
from LMSAPI.api.Models.EnrollQuestions import EnrollQuestions
from LMSAPI.api.Models.QuestionAnswers import QuestionAnswers
from LMSAPI.api.Views.TokenAPI import auth
from flask_cors import CORS

question_answers_api = Blueprint("question_answers_api", __name__)

CORS(question_answers_api)


@question_answers_api.route(
    "/lms/api/v1.0/<lname>/question_answers",
    methods=["GET"],
)
@auth.login_required
def get_question_answers_all(lname):
    hwid = request.args.get("hwid", default=None, type=int)
    mid = request.args.get("mid", default=None, type=int)
    questionset = request.args.get("questionset", default=None, type=int)
    question = request.args.get("question", default=None, type=int)

    result = QuestionAnswers().get_question_answers_all(
        lname, hwid, mid, questionset, question
    )

    if result is None:
        return jsonify(success=False), 404

    # Convert Decimal to float
    for row in result:
        for key, value in row.items():
            if isinstance(value, Decimal):
                row[key] = float(value)

    return jsonify(result)


@question_answers_api.route(
    "/lms/api/v1.0/<lname>/question_answers/<int:id>",
    methods=["GET"],
)
@auth.login_required
def get_question_answers_by_id(lname, id: int):
    result = QuestionAnswers().get_question_answers_by_id(lname, id)

    if result is None:
        return jsonify(success=False), 404

    for key, value in result.items():
        if isinstance(value, Decimal):
            result[key] = float(value)

    return jsonify(result)


@question_answers_api.route(
    "/lms/api/v1.0/<lname>/question_answers",
    methods=["POST"],
)
@auth.login_required
def create_question_answers(lname):
    data = request.data
    if not data:
        return jsonify({"error": "No data provided"}), 400

    if sys.version_info[0] < 3:
        data = json.loads(data)
    else:
        data = json.loads(data.decode("utf-8"))

    result = QuestionAnswers().create_question_answers(lname, data)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(success=True), 201


@question_answers_api.route(
    "/lms/api/v1.0/<lname>/question_answers/list",
    methods=["POST"],
)
@auth.login_required
def create_question_answers_list(lname):
    data = request.data
    if not data:
        return jsonify({"error": "No data provided"}), 400

    if sys.version_info[0] < 3:
        data = json.loads(data)
    else:
        data = json.loads(data.decode("utf-8"))

    if len(data) == 0:
        return jsonify({"error": "Empty data list"}), 400

    result = []
    for row in data:
        row["mid"] = g.user.mid
        answer = QuestionAnswers().create_question_answers(lname, row)
        result.append(answer)

    if not result or len(result) == 0:
        return jsonify({"error": "No answers created", "success": False}), 404

    return jsonify(success=True), 201


@question_answers_api.route(
    "/lms/api/v1.0/<lname>/question_answers/<int:id>",
    methods=["PUT"],
)
@auth.login_required
def update_question_answers(lname, id: int):
    data = request.data
    if not data:
        return jsonify({"error": "No data provided"}), 400

    if sys.version_info[0] < 3:
        data = json.loads(data)
    else:
        data = json.loads(data.decode("utf-8"))

    result = QuestionAnswers().update_question_answers(lname, id, data)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(success=True), 201


@question_answers_api.route(
    "/lms/api/v1.0/<lname>/question_answers/<int:id>",
    methods=["DELETE"],
)
@auth.login_required
def delete_question_answers(lname, id: int):
    result = QuestionAnswers().delete_question_answers(lname, id)

    if result is None:
        return jsonify(success=False), 404
    return jsonify(success=True), 200
