import os

from flask import Flask, jsonify, request, current_app, g
from flask_restful import Resource, Api
from sqlalchemy import create_engine
from sqlalchemy.sql import text
from flask import Blueprint

from LMSAPI.api.Models.File import File
from LMSAPI.api.Views.TokenAPI import auth
from LMSAPI.api.Models.Rating import Rating
from flask_cors import CORS
import json, sys
from decimal import Decimal

from LMSAPI.api.utils.access_utils import user_permission_modes

rating_api = Blueprint("rating_api", __name__)

CORS(rating_api)


@rating_api.route(
    "/lms/api/v1.0/<lname>/rating_schema/<int:school_year>", methods=["GET"]
)
def getRatingSchema(lname, school_year):
    rating = Rating(lname).getRatingSchema(school_year)

    return jsonify(rating)


@rating_api.route(
    "/lms/api/v1.0/<lname>/rating/<int:schema>/<int:groupid>/<date>/<int:type>",
    methods=["GET"],
)
@auth.login_required
@user_permission_modes("Отчётность", "Рейтинг-распределение", ["Нет"])
def getRating(lname, schema, groupid, date, type):
    rating = Rating(lname).getRating(schema, groupid, date, type)

    json_data = json.dumps(rating, ensure_ascii=False, default=str)

    return json_data


@rating_api.route("/lms/api/v1.0/<lname>/rating", methods=["POST"])
@auth.login_required
@user_permission_modes("Режимы", "Ввод и корректировка оценок", ["Нет"])
def setRatingValue(lname):
    data = request.data
    if sys.version_info[0] < 3:
        dataDict = json.loads(data)
    else:
        dataDict = json.loads(data.decode("utf-8"))

    result = Rating(lname).setRating(dataDict)

    return jsonify(result)


@rating_api.route(
    "/lms/api/v1.0/<lname>/rating_calc/<int:schema>/<int:groupid>/<date>/<int:type>",
    methods=["GET"],
)
@auth.login_required
def getRatingCalc(lname, schema, groupid, date, type):
    rating = Rating(lname).getRatingCalc(schema, groupid, date, type)

    json_data = json.dumps(rating, ensure_ascii=False, default=str)

    return json_data


@rating_api.route("/lms/api/v1.0/<lname>/rating_calc_updated", methods=["GET"])
@auth.login_required
def getRatingCalcUpdated(lname):
    schema = request.args.get("schema") or None
    groupid = request.args.get("groupid") or None
    date = request.args.get("date") or None
    period_type = request.args.get("period_type") or None
    rating = Rating(lname).getRatingCalcUpdated(schema, groupid, date, period_type)

    json_data = json.dumps(rating, ensure_ascii=False, default=str)

    return json_data


@rating_api.route(
    "/lms/api/v1.0/<lname>/rating_calc_details/<int:schema>/<int:gid>/<date>/<int:type>",
    methods=["GET"],
)
# @auth.login_required
def get_rating_calc_details(lname, schema, gid, date, type):
    diagrams = request.args.get("diagrems", default=0, type=int)
    if diagrams == 1:
        rating = Rating(lname).get_rating_calc_details_diagrems(schema, gid, date, type)
    else:
        rating = Rating(lname).get_rating_calc_details(schema, gid, date, type)

    json_data = json.dumps(rating, ensure_ascii=False, default=str)

    return json_data


@rating_api.route(
    "/lms/api/v1.0/<lname>/digital_portrait/<int:schema>/<int:mid>/<int:yid>",
    methods=["GET"],
)
@auth.login_required
@user_permission_modes("Режимы", "Ввод и корректировка оценок", ["Нет"])
def getDigitalPortrait(lname, schema, mid, yid):
    rating = Rating(lname).getDigitalPortrait(schema, mid, yid)

    json_data = json.dumps(rating, ensure_ascii=False, default=str)

    return json_data


@rating_api.route(
    "/lms/api/v1.0/<lname>/rating_edge/<int:schema>/<int:type>/<date>/<int:kind>/<int:id>",
    methods=["GET"],
)
# @auth.login_required
def getRatingEdge(lname, schema, type, date, kind, id):
    rating = Rating(lname).get_rating_edge(schema, type, date, kind, id)

    json_data = json.dumps(rating, ensure_ascii=False, default=str)

    return json_data


@rating_api.route(
    "/lms/api/v1.0/<lname>/rating_reacalc/<int:schema>/<int:type>/<date>",
    methods=["GET"],
)
@auth.login_required
def getRatingReacal(lname, schema, type, date):
    result = Rating(lname).getRatingReacal(schema, type, date)

    return jsonify(result)


@rating_api.route("/lms/api/v1.0/<lname>/rating/<int:id>", methods=["GET"])
@auth.login_required
@user_permission_modes("Режимы", "Ввод и корректировка оценок", ["Нет"])
def get_rating_info(lname, id):
    result = Rating(lname).get_rating_info(id)
    return jsonify(result)


@rating_api.route("/lms/api/v1.0/<lname>/rating/<int:id>", methods=["PUT"])
@auth.login_required
@user_permission_modes("Режимы", "Ввод и корректировка оценок", ["Нет"])
def update_rating_info(lname, id):
    data = request.data
    if sys.version_info[0] < 3:
        dataDict = json.loads(data)
    else:
        dataDict = json.loads(data.decode("utf-8"))

    result = Rating(lname).update_rating_info(id, dataDict)

    return jsonify(result)


@rating_api.route("/lms/api/v1.0/<lname>/rating/file/<int:id>", methods=["POST"])
@auth.login_required
def save_rating_file(lname, id):
    ip_address = (
        request.headers.get("X-Forwarded-For")
        or request.headers.get("X-Real-IP")
        or request.environ.get("REMOTE_ADDR")
    )
    url = request.form.get("link")
    if url:
        Rating(lname).update_rating_link(id, 2, url)
        return jsonify(success=True), 201

    submitted_file = request.files["file"]
    old_rating_info = Rating(lname).get_rating_info(id)
    if old_rating_info["link_type"] == 1:
        os.remove(
            os.path.join(
                File(lname).get_personnel_rating_values_directory(id),
                old_rating_info["link"],
            )
        )
    if submitted_file:
        File(lname).create_personnel_rating_values_directory(id)
        file_path = File(lname).get_personnel_rating_values_directory(id)
        Rating(lname).update_rating_link(id, 1, submitted_file.filename)
        return File.save_file_to_path(
            lname=lname,
            submitted_file=submitted_file,
            file_path=file_path,
            action_source=request.environ.get("HTTP_USER_AGENT"),
            login=g.user.id,
            user_mid=g.user.mid,
            ip_address=ip_address,
        )

    return jsonify({"success": False, "error": "upload file with key file"})


@rating_api.route("/lms/api/v1.0/<lname>/rating/file/<int:id>", methods=["DELETE"])
@auth.login_required
def delete_rating_file(lname, id):
    try:
        result = Rating(lname).get_rating_info(id)
        if result["link_type"] == 1:
            os.remove(
                os.path.join(
                    File(lname).get_personnel_rating_values_directory(id),
                    result["link"],
                )
            )
        Rating(lname).update_rating_link(id, None, None)
        return jsonify(success=True)

    except:
        return jsonify(success=False), 404


@rating_api.route(
    "/lms/api/v1.0/<lname>/rating/file/<int:id>/<filename>", methods=["GET"]
)
@auth.login_required
def get_rating_file(lname, id, filename):
    path = File(lname).get_personnel_rating_values_directory(id)
    ip_address = (
        request.headers.get("X-Forwarded-For")
        or request.headers.get("X-Real-IP")
        or request.environ.get("REMOTE_ADDR")
    )
    return File.send_from_directory(
        lname=lname,
        file_path=path,
        file_name=filename,
        action_source=request.environ.get("HTTP_USER_AGENT"),
        login=g.user.id,
        user_mid=g.user.mid,
        ip_address=ip_address,
    )
