import base64
import os
from decimal import Decimal

from flask import Flask, jsonify, request, current_app, send_from_directory, g
from flask_restful import Resource, Api
from sqlalchemy import create_engine
from sqlalchemy.sql import text
from flask import Blueprint
from LMSAPI.api.Models.File import File
from LMSAPI.api.Models.UserActionLog import UserActionLog
from LMSAPI.api.Views.TokenAPI import auth
from LMSAPI.api.Models.User import User
from LMSAPI.api.Models.UMKCourse import UMKCourse
from flask_cors import CORS

from LMSAPI.api.utils.access_utils import user_permission_modes

# https://stackoverflow.com/questions/39769666/flask-how-to-use-app-context-inside-blueprints
# Use app context from blueprint

umk_course_api = Blueprint("umk_course_api", __name__)

CORS(umk_course_api)


@umk_course_api.route(
    "/lms/api/v1.0/<lname>/umk/course/lesson/<int:lesson_id>/<file_name>",
    methods=["GET"],
)
@auth.login_required
def getUMKCourseLessonFile(lname, lesson_id, file_name):
    f = UMKCourse(lname)

    ip_address = (
        request.headers.get("X-Forwarded-For")
        or request.headers.get("X-Real-IP")
        or request.environ.get("REMOTE_ADDR")
    )
    return File.send_from_directory(
        lname=lname,
        file_path=f.getLessonFileLocation(lesson_id),
        file_name=file_name,
        action_source=request.environ.get("HTTP_USER_AGENT"),
        login=g.user.id,
        user_mid=g.user.mid,
        ip_address=ip_address,
    )


@umk_course_api.route(
    "/lms/api/v1.0/<lname>/umk/course/lesson/<int:lesson_id>",
    methods=["GET"],
)
@auth.login_required
def getListUMKCourseLessonFile(lname, lesson_id):
    if g.user.isTeacher == False:
        return jsonify("Error: Not teacher owning project id"), 401

    f = UMKCourse(lname)
    return jsonify(f.getLessonFiles(lesson_id))


@umk_course_api.route(
    "/lms/api/v1.0/<lname>/umk/course/lesson/student/<int:lesson_id>/<file_name>",
    methods=["GET"],
)
@auth.login_required
def get_students_umk_course_lesson_file(lname, lesson_id, file_name):
    f = UMKCourse(lname)

    ip_address = (
        request.headers.get("X-Forwarded-For")
        or request.headers.get("X-Real-IP")
        or request.environ.get("REMOTE_ADDR")
    )
    return File.send_from_directory(
        lname=lname,
        file_path=f.get_lesson_student_file_location(lesson_id),
        file_name=file_name,
        action_source=request.environ.get("HTTP_USER_AGENT"),
        login=g.user.id,
        user_mid=g.user.mid,
        ip_address=ip_address,
    )


@umk_course_api.route(
    "/lms/api/v1.0/<lname>/umk/course/lesson/student/<int:lesson_id>",
    methods=["GET"],
)
@auth.login_required
def getstudents_list_umk_course_lesson_file(lname, lesson_id):
    f = UMKCourse(lname)
    return jsonify(f.get_lesson_student_files(lesson_id))


@umk_course_api.route("/lms/api/v1.0/<lname>/interim_certification", methods=["GET"])
@auth.login_required
def get_interim_certification(lname):
    """GET with course id to get list of course"""
    umk = UMKCourse(lname)
    column_titles = umk.get_xp_mid_grades_table_view(student_id=g.user.mid)
    course_data = umk.get_xp_mid_grades_table(student_id=g.user.mid)

    # Заменяем названия столбцов на соответствующие титулы и удаляем поля, оканчивающиеся на _info
    renamed_course_data = [
        {
            column_titles.get(k, k): v
            for k, v in record.items()
            if not k.endswith("_info")
        }
        for record in course_data
    ]

    # Преобразуем Decimal в float для JSON
    for record in renamed_course_data:
        for k, v in record.items():
            if isinstance(v, Decimal):
                record[k] = float(v)

    return jsonify(renamed_course_data)


# Get all cathedras for a faculty
@umk_course_api.route("/lms/api/v1.0/<lname>/umk/course/<int:cid>", methods=["GET"])
@auth.login_required
def getUMKCourse(lname, cid):
    """GET with course id to get list of course"""
    umk = UMKCourse(lname)
    result = umk.getUMKCourses(cid)
    if result is None:
        return jsonify(success=False), 404
    return jsonify(result)
    # return jsonify([fa.serialize() for fa in result])


@umk_course_api.route(
    "/lms/api/v1.0/<lname>/umk/<int:cid>/<int:idcurriculum>", methods=["GET"]
)
@auth.login_required
@user_permission_modes("Режимы", "ЭУМК", ["Нет"])  # ЭУМК. Администратор
def getUMKtheme(lname, cid, idcurriculum):
    umk = UMKCourse(lname)
    result = umk.getUMKtheme(
        cid,
        idcurriculum,
        g.user.mid,
        g.user.access_modes["Режимы"]["ЭУМК. Администратор"],
    )
    # result = umk.get_theme_plan(cid, idcurriculum)
    if result is None:
        return jsonify(success=False), 404
    return jsonify(result)


@umk_course_api.route(
    "/lms/api/v1.0/<lname>/umk/<int:cid>/<int:idcurriculum>/<int:meta_course>/<int:idcathedra>/<int:xp_key>",
    methods=["GET"],
)
@auth.login_required
@user_permission_modes(
    "Справочники", "Комплексный тем.план (КТП) по дисциплинам", ["Нет"]
)
def getUMK(
    lname, cid: int, idcurriculum: int, meta_course: int, idcathedra: int, xp_key: int
):
    umk = UMKCourse(lname)

    result = umk.getUMK(cid, idcurriculum, meta_course, idcathedra, xp_key)
    if result is None:
        return jsonify(), 200
    return jsonify(result)


@umk_course_api.route("/lms/api/v1.0/<lname>/umk/<int:id>", methods=["GET"])
@auth.login_required
@user_permission_modes("Режимы", "ЭУМК", ["Нет"])
def getUMKDetails(lname, id):
    umk = UMKCourse(lname)
    result = umk.getUMKDetails(id)
    if result is None:
        return jsonify(success=False), 404
    return jsonify(result)


@umk_course_api.route(
    "/lms/api/v1.0/<lname>/umk/course/<int:cid>/<int:idcurriculum>/<int:syid>/<int:eyid>",
    methods=["GET"],
)
@auth.login_required
def getUMKCourseDetails(lname, cid, syid, idcurriculum, eyid):
    """GET with course id to get list of course"""
    umk = UMKCourse(lname)
    result = umk.getUMKCourse(cid, syid, idcurriculum, eyid)
    if result is None:
        return jsonify(success=False), 404
    return jsonify(result)


@umk_course_api.route(
    "/lms/api/v1.0/<lname>/umk/course/files/<int:id_curriculum_course>/<int:eyid>/<int:tlid>/<oid>/<filename>",
    methods=["GET"],
)
@auth.login_required
def getUMKCourseFile(lname, id_curriculum_course, eyid, tlid, oid, filename):
    f = UMKCourse(lname)
    # current_app.logger.debug("Sending from " + f.getMethodFileLocation(mrid))

    path = f.getUMKCourseFileLocation(id_curriculum_course, eyid, tlid, oid)
    ip_address = (
        request.headers.get("X-Forwarded-For")
        or request.headers.get("X-Real-IP")
        or request.environ.get("REMOTE_ADDR")
    )
    return File.send_from_directory(
        lname=lname,
        file_path=path,
        file_name=filename,
        action_source=request.environ.get("HTTP_USER_AGENT"),
        login=g.user.id,
        user_mid=g.user.mid,
        ip_address=ip_address,
    )


@umk_course_api.route(
    "/lms/api/v1.0/<lname>/umk/<int:id>/links/<int:type>", methods=["GET"]
)
@auth.login_required
def getUMKLinks(lname, id, type):
    umk = UMKCourse(lname)
    result = umk.getUMKLinks(id, type)
    if result is None:
        return jsonify(success=False), 404
    return jsonify(result)


@umk_course_api.route("/lms/api/v1.0/<lname>/umk/<int:id>/edu_plan", methods=["GET"])
@auth.login_required
def getUMKEduPlan(lname, id):
    umk = UMKCourse(lname)
    result = umk.getUMKEduPlan(id)
    if result is None:
        return jsonify(success=False), 404
    return jsonify(result)


@umk_course_api.route(
    "/lms/api/v1.0/<lname>/umk/<int:id>/work_program", methods=["GET"]
)
@auth.login_required
def getUMKWorkProgram(lname, id):
    umk = UMKCourse(lname)
    result = umk.getUMKWorkProgram(id)
    if result is None:
        return jsonify(success=False), 404
    return jsonify(result)


# @umk_course_api.route("/lms/api/v1.0/<lname>/umk/<int:id>/tema_plan/<int:id_curriculum_course>", methods=["GET"])
# @auth.login_required
# # @user_permission_modes("Справочники", "Факультеты", ["Нет"])
# def getUMKTemaPlan(lname, id, id_curriculum_course):
#     umk = UMKCourse(lname)
#     result = umk.getUMKTemaPlan(id, id_curriculum_course)
#     # result = umk.newUmkThemePlan(id, id_curriculum_course)
#     if result is None:
#         return jsonify(success=False), 404
#     return jsonify(result)


@umk_course_api.route(
    "/lms/api/v1.0/<lname>/umk/<int:id>/tema_plan/<int:id_curriculum_course>",
    methods=["GET"],
)
@auth.login_required
# @user_permission_modes("Справочники", "Факультеты", ["Нет"])
def getUMKTemaPlanNew(lname, id, id_curriculum_course):
    umk = UMKCourse(lname)
    result = umk.newUmkThemePlan(id, id_curriculum_course)
    if result is None:
        return jsonify(success=False), 404
    return jsonify(result)


@umk_course_api.route("/lms/api/v1.0/<lname>/event_tools", methods=["GET"])
# @auth.login_required
def get_all_event_tools(lname):
    res = UMKCourse(lname).get_all_event_tools(lname)
    if not res:
        return jsonify(success=False), 400
    # del this attribute cause jsonify cant serialize it
    for item in res:
        del item["bmp"]
    return jsonify(res)


@umk_course_api.route(
    "/lms/api/v1.0/<lname>/umk_links/<int:idcurriculum>/<int:cid>",
    methods=["GET"],
)
@auth.login_required
def get_umk_links_url_files_text(lname, idcurriculum, cid):
    result = {
        "main": {
            "url": UMKCourse(lname).get_umk_links_by_idcurriculum_cid(
                idcurriculum, cid, 1, 1, 0
            ),
            "files": UMKCourse(lname).get_umk_links_by_idcurriculum_cid(
                idcurriculum, cid, 1, 1, 1
            ),
            "methodical_files": UMKCourse(lname).get_umk_links_by_idcurriculum_cid(
                idcurriculum, cid, 1, 1, 2
            ),
            "text": UMKCourse(lname).get_umk_links_by_idcurriculum_cid(
                idcurriculum, cid, 1, 1, 3
            ),
        },
        "additional": {
            "url": UMKCourse(lname).get_umk_links_by_idcurriculum_cid(
                idcurriculum, cid, 2, 2, 0
            ),
            "files": UMKCourse(lname).get_umk_links_by_idcurriculum_cid(
                idcurriculum, cid, 2, 2, 1
            ),
            "methodical_files": UMKCourse(lname).get_umk_links_by_idcurriculum_cid(
                idcurriculum, cid, 2, 2, 2
            ),
            "text": UMKCourse(lname).get_umk_links_by_idcurriculum_cid(
                idcurriculum, cid, 2, 2, 3
            ),
        },
        "methodical": {
            "url": UMKCourse(lname).get_umk_links_by_idcurriculum_cid(
                idcurriculum, cid, 3, 4, 0
            ),
            "files": UMKCourse(lname).get_umk_links_by_idcurriculum_cid(
                idcurriculum, cid, 3, 4, 1
            ),
            "methodical_files": UMKCourse(lname).get_umk_links_by_idcurriculum_cid(
                idcurriculum, cid, 3, 4, 2
            ),
            "text": UMKCourse(lname).get_umk_links_by_idcurriculum_cid(
                idcurriculum, cid, 3, 4, 3
            ),
        },
    }
    if result is None:
        return jsonify(success=False), 404
    return jsonify(result)


@umk_course_api.route("/lms/api/v1.0/<lname>/umk/file/<int:id>", methods=["GET"])
@auth.login_required
def get_umk_links_file(lname, id):
    if request.args.get("is_methodical_file"):
        umk_links = {"link": UMKCourse(lname).get_filename_by_mrid(id)["mr_name"]}
        path = File(lname).get_umk_links_directory(
            id, None,2
        )
    else:
        umk_links = UMKCourse(lname).get_umk_by_id(id)

        if umk_links["link_type"] != 1 and umk_links["link_type"] != 2:
            return (
                jsonify(success=False, error="Переданный id не содержит в себе файл"),
                404,
            )
        path = File(lname).get_umk_links_directory(
            umk_links["umk"], umk_links["umk_subtype"], umk_links["link_type"]
        )
    ip_address = (
        request.headers.get("X-Forwarded-For")
        or request.headers.get("X-Real-IP")
        or request.environ.get("REMOTE_ADDR")
    )
    return File.send_from_directory(
        lname=lname,
        file_path=path,
        file_name=umk_links["link"],
        action_source=request.environ.get("HTTP_USER_AGENT"),
        login=g.user.id,
        user_mid=g.user.mid,
        ip_address=ip_address,
    )
