import re

from flask import Flask, jsonify, request, current_app, session, Response, g
from flask_restful import Resource, Api
from sqlalchemy import create_engine
from flask import Blueprint
from flask_httpauth import HTTPBasicAuth

from LMSAPI.api.Models.Group import Group
from LMSAPI.api.Views.TokenAPI import auth
from LMSAPI.api.Models.User import User
from sqlalchemy.sql import text
from flask_cors import CORS
from sqlalchemy.sql import text
from LMSAPI.api.Models.Rate import Rate
import json, sys

from LMSAPI.api.utils.access_utils import user_permission_modes
from LMSAPI.api.utils.swagger_comments import SwaggerComments
from LMSAPI.api.utils.swagger_utils import swagger_doc

user_api = Blueprint("user_api", __name__)

CORS(user_api)


@user_api.route("/lms/api/v1.0/<lname>/users/password", methods=["POST"])
@auth.login_required
def changeUserPassword(lname):
    data = request.data
    if sys.version_info[0] < 3:
        dataDict = json.loads(data)
    else:
        dataDict = json.loads(data.decode("utf-8"))

    if "oldpassword" in dataDict:
        oldpassword = dataDict["oldpassword"]

    if oldpassword == None:
        return jsonify("missing old password parameter"), 502

    if "newpassword" in dataDict:
        newpassword = dataDict["newpassword"]

    if newpassword == None:
        return jsonify("missing old password parameter"), 502

    if newpassword == oldpassword:
        return jsonify("Новый и старый пароли совпадают"), 400

    filters = User.get_password_filters(User, lname)

    if filters:
        if len(newpassword) < int(filters[1]):
            return (
                jsonify(
                    str("Длина пароля должна быть не меньше {} символов").format(
                        filters[1]
                    )
                ),
                400,
            )

        if str(filters[0]) == "1":
            if not re.search("[a-zA-Z]", newpassword):
                return jsonify("Пароль не содержит букв"), 400
            if not re.search("[0-9]", newpassword):
                return jsonify("Пароль не содержит цифр"), 400
            if not re.search("[a-z]", newpassword):
                return jsonify("Пароль не содержит строчных букв"), 400
            if not re.search("[A-Z]", newpassword):
                return jsonify("Пароль не содержит прописных  букв"), 400

    # if int(filters[2]) > 0 and int(filters[3]) > 0:
    #     if int(User(lname).check_user_try(lname, g.user.id, filters[3])) > int(filters[3]):
    #         User(lname).block_user_try(lname, g.user.id, [request.remote_addr, g.user.mid])

    u = User(lname).get_user_from_sql_id(g.user.mid)

    if u == None:
        return jsonify("ERROR"), 404

    if u.verify_password(oldpassword) == False:
        return jsonify("Неверно указан текущий пароль"), 403

    u.change_password(newpassword)

    return jsonify("OK")


@user_api.route("/lms/api/v1.0/<lname>/users/students", methods=["GET"])
@auth.login_required
@user_permission_modes("Справочники", "Переменный состав. Прочие сведения", ["Нет"])
@swagger_doc(SwaggerComments.user_api_getStudentList)
def getStudentList(lname):
    """
    Получить список студентов (режим "Переменный состав")
    """
    args = request.args

    # Флаг администратора
    admin_flag = int(
        g.user.access_modes["Справочники"]["Переменный состав. Администратор"] in ["Полный"]
    )

    # Удобная функция для получения параметров с алиасами
    def get_arg(*names, default=None):
        for n in names:
            if n in args and args[n]:
                return args[n]
        return default

    # Определяем список групп
    group = get_arg("group", "gid")
    if group:
        group_list = (group,)
    else:
        groups = Group(lname).get_groups_by_faculty_cathedra_year_eyid(
            lname=lname,
            xp_key=get_arg("year", "xp_key", default=0),
            idfaculty=get_arg("faculty", "idfaculty", default=0),
            idcathedra=get_arg("cathedra", "idcathedra", default=0),
            eyid=get_arg("kurs", "eyid", default=-1),
            mid=g.user.mid,
            admin_flag=admin_flag
        )
        group_list = tuple(g["gid"] for g in groups) or (0,)

    # Получаем студентов
    result = User(lname).search_student_list(
        args=args,
        mid=g.user.mid,
        group_list=group_list,
        is_admin=admin_flag
    )

    return jsonify(result)


@user_api.route("/lms/api/v1.0/<lname>/users/students/<name>", methods=["GET"])
@auth.login_required
def getStudentListName(lname, name):
    """GET students with special name"""
    args = request.args
    admin_flag = int(g.user.access_modes["Справочники"]["Переменный состав. Администратор"] in ["Полный"])

    if "group" in args.keys() and args["group"]:
        group_list = tuple([args["group"]])
    else:
        group_list = Group(lname).get_groups_by_faculty_cathedra_year_eyid(
            lname=lname,
            xp_key=args.get("year", 0),
            idfaculty=args.get("faculty", 0),
            idcathedra=args.get("cathedra", 0),
            eyid=args.get("kurs", -1),
            mid=g.user.mid,
            admin_flag=admin_flag
        )
        group_list = tuple(gid["gid"] for gid in group_list)
    if not group_list:
        group_list = tuple([0])

    result = User(lname).search_student_list(
        args=args, mid=g.user.mid, name=name, group_list=group_list, is_admin=admin_flag)
    return jsonify(result)


@user_api.route("/lms/api/v1.0/<lname>/users/students", methods=["POST"])
@auth.login_required
def addStudent(lname):
    data = request.data
    dataDict = json.loads(data)
    studentid = User(lname).addStudent(dataDict)

    if studentid is None:
        return jsonify(success=False), 404

    return jsonify({"mid": studentid})


# isteacher -1 does not matter 0 is not 1 is
# name will %fio%
@user_api.route("/lms/api/v1.0/<lname>/users/teachers", methods=["GET"])
@auth.login_required
@user_permission_modes("Отчётность", "Постоянный состав", ["Нет"])
@swagger_doc(SwaggerComments.user_api_getUserList)
def getUserList(lname):
    """
    Получить список преподавателей (режим "Постоянный состав")
    """
    us = User(lname)
    admin_flag = g.user.access_modes["Справочники"]["Постоянный состав. Администратор"] in ["Полный"]
    result = us.search_teacher_list_v2(None, request.args, admin_flag, g.user.mid)
    if bool(str(request.args.get("count", "false")).lower() in ("true", "1", "yes")):
        return jsonify(result)
    return jsonify([u.serialize() for u in result])


@user_api.route("/lms/api/v1.0/<lname>/users/teachers", methods=["POST"])
@auth.login_required
def addTeacher(lname):
    data = request.data
    dataDict = json.loads(data)
    studentid = User(lname).addTeacher(dataDict)

    if studentid is None:
        return jsonify(success=False), 404

    return jsonify({"mid": studentid})


@user_api.route("/lms/api/v1.0/<lname>/users/teachers/<name>", methods=["GET"])
@auth.login_required
@user_permission_modes("Отчётность", "Постоянный состав", ["Нет"])
def getUserListName(lname, name):
    """GET user with special name"""
    us = User(lname)
    admin_flag = g.user.access_modes["Справочники"]["Постоянный состав. Администратор"] in ["Полный"]
    result = us.search_teacher_list_v2(name, request.args, admin_flag, g.user.mid)
    if request.args.get("count", False):
        return jsonify(result)
    return jsonify([u.serialize() for u in result])


@user_api.route("/lms/api/v1.0/<lname>/users/searchlogin/<int:name>", methods=["GET"])
@auth.login_required
def getUserName(lname, name):
    """GET user with special name"""
    conn = current_app.ms.db(lname).connect()

    stmt = text("select * from people where login=:name")
    stmt = stmt.bindparams(name=name)
    query = conn.execute(stmt)

    result = {"data": [dict(zip(tuple(query.keys()), i)) for i in query.cursor]}

    return jsonify(result)


@user_api.route("/lms/api/v1.0/<lname>/users", methods=["GET"])
@auth.login_required
def getUsers(lname):
    """GET user information from ID"""

    result = User(lname).get_users()
    return jsonify(result)


@user_api.route("/lms/api/v1.0/<lname>/users/<int:id>", methods=["GET"])
@auth.login_required
def getUserDetailsId(lname, id):
    """GET user information from ID"""

    us = User(lname)

    result = us.get_user_details_id(id)
    is_student, is_teacher = us.check_students_or_teachers_mid(id)
    if is_student:
        completeness = us.get_student_stats(id)
    else:
        completeness = us.get_teachers_stats(id)
    result["userDetails"][0]["completeness"] = completeness


    if result is None:
        return jsonify(success=False), 404

    teacherdetails = us.get_teacher_details_id(id)

    result["teacherDetails"] = teacherdetails

    return jsonify(result)


@user_api.route("/lms/api/v1.0/<lname>/users/<int:id>/rate", methods=["GET"])
@auth.login_required
def getUserRating(lname, id):
    """GET user rate from ID"""
    ra = Rate()
    result = ra.getRate(lname, id)
    if result is None:
        return jsonify(success=False), 404
    for objs in result:
        for key, value in objs.items():
            for i in value:
                if i["mid"] != id and i["xp_f_get_mid_fio"]:
                    i["xp_f_get_mid_fio"] = i["xp_f_get_mid_fio"][0] + "*****"
                    del i["mid"]
    return jsonify(result)


@user_api.route(
    "/lms/api/v1.0/<lname>/users/<int:id>/rate_exam/<int:year>/<int:term>",
    methods=["GET"],
)
@auth.login_required
def getUserRatingExam(lname, id, year, term):
    """GET user rate from ID"""
    ra = Rate()
    result = ra.getRateExam(lname, id, year, term)
    if result is None:
        return jsonify(success=False), 404
    for objs in result:
        for key, value in objs.items():
            for i in value:
                if i["mid"] != id and i["xp_f_get_mid_fio"]:
                    i["xp_f_get_mid_fio"] = i["xp_f_get_mid_fio"][0] + "*****"
                    del i["mid"]
    return jsonify(result)


@user_api.route("/lms/api/v1.0/<lname>/user/<int:id>/photo", methods=["GET"])
# @auth.login_required - removed 31.08.2024 - kiosk does not work
def get_user_details_photo(lname, id):
    """GET user information from your ID"""
    us = User(lname)
    # if g.user.mid != id: - removed 29.10.2024 - cause @auth.login_required is not supported in route
    #     return jsonify({"error": "Нет доступа"})
    result = us.get_user_details_photo(id)

    if result is None:
        return jsonify(success=False), 404

    response = Response(str(id))
    response.content_type = "image/jpeg; charset=utf-8"
    response.headers["Content-disposition"] = (
        "attachment; filename = " + str(id) + ".jpg"
    )
    response.headers["Cache-control"] = "max-age=86400"
    response.content_length = len(result)
    response.data = result

    return response


@user_api.route("/lms/api/v1.0/<lname>/users/<int:mid>/photo", methods=["GET"])
@swagger_doc(SwaggerComments.user_api_getUserDetailsPhotoId)
# @auth.login_required - removed 31.08.2024 - kiosk does not work
# @user_permission_modes("Отчётность", "Портфолио", ["Нет"])
def getUserDetailsPhotoId(lname, mid):
    """
    Получить фотографию пользователя по его ID
    """

    us = User(lname)

    result = us.get_user_details_photo(mid)

    if result is None:
        return jsonify(success=False), 404

    response = Response(str(mid))
    response.content_type = "image/jpeg; charset=utf-8"
    response.headers["Content-disposition"] = (
        "attachment; filename = " + str(mid) + ".jpg"
    )
    response.headers["Cache-control"] = "max-age=86400"
    response.content_length = len(result)
    response.data = result

    return response


@user_api.route("/lms/api/v1.0/<lname>/users/<int:id>/biography", methods=["GET"])
def getUserBiography(lname, id):
    """GET user biography from ID"""

    result = User(lname).getUserBiography(lname, id)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@user_api.route("/lms/api/v1.0/<lname>/students_data", methods=["POST"])
@auth.login_required
def createStudentUser(lname: str):
    """Create student row"""
    data = request.json
    result = User(lname).createStudentUser(lname, data)

    if not result:
        return jsonify(success=False, message="У вас недостаточно прав"), 403

    return jsonify(result)


@user_api.route("/lms/api/v1.0/<lname>/students_data/<int:mid>", methods=["PUT"])
@auth.login_required
def updateStudentUser(lname: str, mid: int):
    """Update student row by mid"""
    data = request.json
    result = User(lname).updateStudentUser(lname, mid, data)

    if not result:
        return jsonify(success=False, message="У вас недостаточно прав"), 403

    return jsonify(result)


@user_api.route("/lms/api/v1.0/<lname>/teachers_data", methods=["POST"])
@auth.login_required
def createTeacherUser(lname: str):
    """Update student row by mid"""
    data = request.json
    result = User(lname).createTeacherUser(lname, data)

    if not result:
        return jsonify(success=False, message="У вас недостаточно прав"), 403

    return jsonify(result)


@user_api.route("/lms/api/v1.0/<lname>/teachers_data/<int:mid>", methods=["PUT"])
@auth.login_required
def updateTeacherUser(lname: str, mid: int):
    """Update student row by mid"""
    data = request.json
    result = User(lname).updateTeacherUser(lname, mid, data)

    if not result:
        return jsonify(success=False, message="У вас недостаточно прав"), 403

    return jsonify(result)


@user_api.route("/lms/api/v1.0/<lname>/users_photo/<int:mid>", methods=["PUT"])
@auth.login_required
def uploadUsersPhoto(lname: str, mid: int):
    """Update student row by mid"""
    photo = request.files.get("photo").stream.read()
    result = User(lname).uploadUsersPhoto(lname, mid, photo)

    if not result:
        return jsonify(success=False, message="У вас недостаточно прав"), 403

    return jsonify(result)


@user_api.route('/lms/api/v1.0/<lname>/teacher_reviews/users', methods=['GET'])
@auth.login_required
def get_teacher_by_cathedra_meta_course(lname: str):
    idcathedra = request.args.get("idcathedra", type=int, default=None)
    meta_course = request.args.get("meta_course", type=int, default=None)
    limit = request.args.get("limit", type=int, default=None)
    offset = request.args.get("offset", type=int, default=None)
    search = request.args.get("search", type=str, default=None)

    res = User(lname).get_teacher_by_cathedra_meta_course(
        lname, idcathedra, meta_course, search, limit, offset
    )
    if not res:
        return jsonify(success=False), 404
    return jsonify(res)
