import textwrap

from LMSAPI.api.utils.swagger_params import Param
from LMSAPI.api.utils.swagger_utils import SwaggerUtils


class SwaggerComments:
    orders_api_getSRW_comment = {
        "tags": ["Научно-исследовательские работы"],
        "summary": "Получить список Научно-исследовательских работ (НИР, SRW)",
        "description": textwrap.dedent("""
            **Работает для пользователей с правами:**
            **- Режимы → Научные работы → Чтение**
            **- Режимы → Научные работы → Полный**

            Данный запрос возвращает список **научно-исследовательских работ**,
            фильтруя результаты по дате, факультету, типу работы, статусу, номеру приказа и
            ряду дополнительных условий.

            🔹 Логика работы:
            - По умолчанию выбираются приказы с типом документа `S` (НИР).
            - Фильтрация:
              - `idorderdate` (год приказа). Если указан, возвращаются приказы только за этот год.
              - `idfaculty` — фильтр по факультету (0 = все факультеты).
              - `order_type_id` — фильтр по виду работы (0 = все типы).
              - `order_state`:
                - `0` — все работы,
                - `1` — только действующие,
                - `2` — только закрытые.
              - `ordernum` — поиск по номеру приказа (`all` = все).
              - `by_user=true` — ограничение по пользователю: выбираются только те приказы, где пользователь:
                - инициатор (`ord_init_mid`),
                - научный руководитель (`nir_instructor_id`),
                - заместитель руководителя (`nir_subinstructor_id`),
                - ответственный исполнитель (`nir_main_executor`),
                - включён в таблицу `order_people`.
              - `from_date` и `to_date` — фильтрация по диапазону дат приказа.
              - `paginateFrom`, `maxResults` — поддержка пагинации.
        """),
        "security": [SwaggerUtils.security],
        "parameters": [
            Param.lname(in_="path", required=True),
            Param.from_date(in_="query", required=False),
            Param.to_date(in_="query", required=False),
            Param.idfaculty(in_="path", required=True),
            Param.xp_key(in_="query", required=False, example=42),
            {
                "name": "idorderdate",
                "in": "path",
                "type": "integer",
                "required": True,
                "description": "Год приказа (0 = все годы)",
                "example": 2024,
            },
            {
                "name": "order_type_id",
                "in": "path",
                "type": "integer",
                "required": True,
                "description": "ID вида работы (0 = все типы)",
                "example": 2,
            },
            {
                "name": "order_state",
                "in": "path",
                "type": "integer",
                "required": True,
                "description": "0=все, 1=действующие, 2=закрытые",
                "example": 1,
            },
            {
                "name": "ordernum",
                "in": "path",
                "type": "string",
                "required": True,
                "description": "Номер приказа (all = все)",
                "example": "all",
            },
            {
                "name": "by_user",
                "in": "query",
                "type": "boolean",
                "required": False,
                "description": "true = только приказы пользователя",
                "example": True,
            },
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — список НИР",
                "schema": {
                    "type": "object",
                    "properties": {
                        "count": {
                            "type": "integer",
                            "description": "Общее количество найденных записей",
                            "example": 25
                        },
                        "scientific_research_work_table": {
                            "type": "array",
                            "items": {
                                "type": "object",
                                "properties": {
                                    "idorder": {
                                        "type": "integer",
                                        "description": "ID приказа",
                                        "example": 123
                                    },
                                    "order_type": {
                                        "type": "string",
                                        "description": "Вид работы",
                                        "example": "НИР (Научная работа)"
                                    },
                                    "ordernum": {
                                        "type": "string",
                                        "description": "Номер приказа",
                                        "example": "45/Н"
                                    },
                                    "orderdate": {
                                        "type": "string",
                                        "description": "Дата приказа",
                                        "example": "2024-03-15"
                                    },
                                    "order_state": {
                                        "type": "string",
                                        "description": "Статус приказа",
                                        "example": "Действующий"
                                    },
                                    "ord_close_date": {
                                        "type": "string",
                                        "description": "Дата закрытия приказа (если есть)",
                                        "example": "2024-07-01"
                                    },
                                    "executor": {
                                        "type": "string",
                                        "description": "Исполнители (ФИО)",
                                        "example": "Иванов И.И., Петров П.П."
                                    },
                                    "nir_instructor_id": {
                                        "type": "string",
                                        "description": "Научный руководитель (ФИО)",
                                        "example": "Сидоров С.С."
                                    },
                                    "nir_subinstructor_id": {
                                        "type": "string",
                                        "description": "Заместитель научного руководителя (ФИО)",
                                        "example": "Кузнецов А.А."
                                    },
                                    "nir_main_executor": {
                                        "type": "string",
                                        "description": "Ответственный исполнитель (ФИО)",
                                        "example": "Васильев В.В."
                                    },
                                    "ord_content": {
                                        "type": "string",
                                        "description": "Содержание приказа",
                                        "example": "Исследование по направлению искусственного интеллекта."
                                    },
                                    "coalesce": {
                                        "type": "string",
                                        "description": "Отдел",
                                        "example": "11 отдел"
                                    },
                                }
                            }
                        }
                    }
                }
            },
            401: SwaggerUtils.defoult_401,
            404: SwaggerUtils.defoult_404,
        }
    }

    division_api_get_divisions_permanent_staff = {
        "tags": ["Постоянный состав", "Научно-исследовательские работы"],
        "summary": "Получить список подразделений (режим 'Постоянный состав')",
        "description": textwrap.dedent("""
            Если пользователь имеет **права администратора (Справочники → Постоянный состав → Администратор → Полный)**:
            - возвращается список всех подразделений

            Если прав **нет**:
            - возвращаются только подразделения, к которым он принадлежит
        """),
        "security": [SwaggerUtils.security],
        "parameters": [
            Param.lname(in_="path", required=True),
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — список подразделений.",
                "schema": {
                    "type": "array",
                    "items": {
                        "type": "object",
                        "properties": {
                            "did": {
                                "type": "integer",
                                "description": "ID подразделения",
                                "example": 101,
                            },
                            "owner_dep": {
                                "type": "integer",
                                "nullable": True,
                                "description": "ID родительского подразделения",
                                "example": 10,
                            },
                            "name": {
                                "type": "string",
                                "description": "Название подразделения",
                                "example": "1 Курс",
                            },
                        },
                    },
                },
            },
            401: SwaggerUtils.defoult_401,
            404: SwaggerUtils.defoult_404,
        },
    }

    orders_api_getOrderTypes = {
        "tags": ["Научно-исследовательские работы"],
        "summary": "Получить список видов приказов (работ)",
        "description": textwrap.dedent("""
                **Запрос возвращает ответ даже если у пользователя нет никаких прав.**

                Данный запрос возвращает список возможных **видов приказов/работ**, 
                которые могут оформляться в системе (например: НИР, дипломные работы, практики и т.д.).

                🔹 Логика работы:
                - Выполняется запрос к таблице `order_types`.
                - Отбираются записи только с `doc_type = 'S'`.
                - Список сортируется по названию (`order_type`).
                - Результат возвращается в виде JSON-объекта с ключом `"order_type"`.
                - Если список пуст → возвращается `404` и объект `{"success": false}`.
            """),
        "security": [SwaggerUtils.security],
        "parameters": [
            Param.lname(in_="path", required=True),
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — список видов приказов",
                "schema": {
                    "type": "object",
                    "properties": {
                        "order_type": {
                            "type": "array",
                            "items": {
                                "type": "object",
                                "properties": {
                                    "order_type_id": {
                                        "type": "integer",
                                        "description": "ID вида работ",
                                        "example": 5
                                    },
                                    "order_type": {
                                        "type": "string",
                                        "description": "Наименование вида работ",
                                        "example": "НИР"
                                    }
                                }
                            }
                        }
                    }
                }
            },
            401: SwaggerUtils.defoult_401,
            404: SwaggerUtils.defoult_404,
        }
    }

    year_api_getSchoolYearList = {
        "tags": ["Переменный состав", "Расписание занятий", "Работы/проекты обучающегося", "Электронные учебно-методические комплексы дисциплин"],
        "summary": "Получить список учебных годов",
        "description": textwrap.dedent("""
                **Запрос возвращает ответ даже если у него нет никаких прав.**

                Возвращает список учебных лет с датами начала/окончания и доп. атрибутами.
            """),
        "parameters": [
            Param.lname(in_="path", required=True),
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — список учебных лет",
                "schema": {
                    "type": "object",
                    "properties": {
                        "schoolYear": {
                            "type": "array",
                            "items": {
                                "type": "object",
                                "properties": {
                                    "xp_key": {
                                        "type": "integer",
                                        "description": "ID учебного года",
                                        "example": 37,
                                    },
                                    "name": {
                                        "type": "string",
                                        "description": "Название учебного года",
                                        "example": "2024/2025 учебный год",
                                    },
                                    "begdate": {
                                        "type": "string",
                                        "description": "Дата начала",
                                        "example": "01-09-2024",
                                    },
                                    "enddate": {
                                        "type": "string",
                                        "description": "Дата окончания",
                                        "example": "31-05-2025",
                                    },
                                    "year_grade_types": {
                                        "type": "string",
                                        "description": "Не известно",
                                        "example": None,
                                    },
                                },
                            },
                        }
                    }
                },
            },
            404: SwaggerUtils.defoult_404,
        },
    }

    year_api_getEducationYear = {
        "tags": ["Переменный состав", "Расписание занятий"],
        "summary": "Получить список возможных курсов обучения",
        "description": textwrap.dedent("""
            **Запрос возвращает ответ даже если у пользователя нет никаких прав.**
        """),
        "parameters": [
            Param.lname(in_="path", required=True),
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — список курсов обучения",
                "schema": {
                    "type": "object",
                    "properties": {
                        "educationYear": {
                            "type": "array",
                            "items": {
                                "type": "object",
                                "properties": {
                                    "eyid": {
                                        "type": "integer",
                                        "description": "ID курса обучения",
                                        "example": 1
                                    },
                                    "name": {
                                        "type": "string",
                                        "description": "Название курса обучения",
                                        "example": "1 курс"
                                    }
                                }
                            }
                        }
                    }
                }
            },
            404: SwaggerUtils.defoult_404,
        }
    }

    schedule_api_getScheduleFilter = {
        "tags": ["Расписание занятий"],
        "summary": "Получить расписание занятий по фильтрам",
        "description": textwrap.dedent("""
                **Запрос возвращает ответ даже если у пользователя нет прав.**

                Данный запрос формирует и возвращает расписание занятий для заданного факультета,
                курса и учебного года.

                🔹 Основные шаги:
                1. На основе параметров (`schoolyear`, `month`) вычисляется **начальная и конечная дата календарного периода**.
                2. Через функцию `nnz_fill_period_schedule` в БД подготавливаются временные таблицы с расписанием.
                3. Если указан `gid` (ID группы) в query-параметрах — расписание фильтруется только для этой группы.
                4. В итоговом JSON-ответе расписание группируется по **дате** и **номеру пары**.

                🔹 Особенности:
                - Если задан `gid=0` (по умолчанию), то возвращаются все группы факультета.
                - Для каждой записи дополнительно формируется "очищенное" поле `cell`, где из текста убраны HTML-теги.
            """),
        "security": [SwaggerUtils.security],
        "parameters": [
            Param.lname(in_="path", required=True),
            Param.idfaculty(in_="path", required=True),
            Param.educationyear(in_="path", required=True),
            Param.xp_key(in_="path", required=True),
            Param.gid(in_="query", required=False),
            {
                "name": "month",
                "in": "path",
                "type": "integer",
                "required": True,
                "description": "Номер месяца (1–12)",
                "example": 9,
            }
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — расписание занятий",
                "schema": {
                    "type": "object",
                    "additionalProperties": {
                        "type": "object",
                        "description": "Ключ = дата занятия",
                        "additionalProperties": {
                            "type": "array",
                            "description": "Ключ = период (номер пары), значение = список занятий",
                            "items": {
                                "type": "object",
                                "properties": {
                                    "cathedra": {"type": "string", "description": "Название кафедры",
                                                 "example": "12 кафедра"},
                                    "cell": {"type": "string", "description": "Не известно", "example": "Не известно"},
                                    "cellraw": {"type": "string", "description": "Не известно",
                                                "example": "Не известно"},
                                    "course_alias": {"type": "string", "description": "Сокращенное название курса",
                                                     "example": "ВТС1"},
                                    "course_index": {"type": "string", "description": "Индекс курса", "example": "1"},
                                    "course_title": {"type": "string", "description": "Название курса",
                                                     "example": "Введение в теорию систем"},
                                    "educationyear": {"type": "string", "description": "Курс обучения",
                                                      "example": "2 курс"},
                                    "faculty": {"type": "string", "description": "Название факультета",
                                                "example": "Факультет информатики"},
                                    "group_id": {"type": "integer", "description": "ID группы", "example": 1},
                                    "groupname": {"type": "string", "description": "Название группы",
                                                  "example": "ИСТ-21"},
                                    "idcathedra": {"type": "integer", "description": "ID кафедры", "example": 12},
                                    "idfaculty": {"type": "integer", "description": "ID факультета", "example": 1},
                                    "kurs": {"type": "integer", "description": "Курс обучения", "example": 1},
                                    "lessondate": {"type": "string", "description": "Дата занятия",
                                                   "example": "Thu, 04 Sep 2025 00:00:00 GMT"},
                                    "lessondate2": {"type": "string", "description": "Не известно",
                                                    "example": "Thu, 04 Sep 2025 00:00:00 GMT"},
                                    "lessontype_alias": {"type": "string",
                                                         "description": "Сокращенное название типа занятия",
                                                         "example": "Л"},
                                    "lessontype_name": {"type": "string", "description": "Тип занятия",
                                                        "example": "Лекция"},
                                    "meta_course_name": {"type": "string", "description": "Не известно",
                                                         "example": "Не известно"},
                                    "period_id": {"type": "integer", "description": "ID пары занятий", "example": 1},
                                    "period_name": {"type": "string", "description": "Пара/период занятия",
                                                    "example": "2-3 пара"},
                                    "period_time": {"type": "string", "description": "Время занятия",
                                                    "example": "10:00-11:30"},
                                    "rooms": {"type": "string", "description": "Аудитории", "example": "Ауд. 305"},
                                    "schedule_id": {"type": "integer", "description": "ID расписания", "example": 1},
                                    "subject_index": {"type": "string", "description": "Не известно",
                                                      "example": "Не известно"},
                                    "subject_title": {"type": "string", "description": "Не известно",
                                                      "example": "Не известно"},
                                    "teacher_mids": {"type": "string", "description": "ID преподавателей",
                                                     "example": "1,2,3"},
                                    "teachers": {"type": "string", "description": "Преподаватели",
                                                 "example": "Иванов И.И., Петров П.П., Сидоров С.С."},
                                    "tema_num": {"type": "integer", "description": "Не известно",
                                                 "example": "Не известно"},
                                    "wday": {"type": "integer", "description": "ID День недели", "example": 6},
                                    "wday_name": {"type": "string", "description": "День недели", "example": "Сб"},
                                    "week_id": {"type": "integer", "description": "ID недели", "example": 1},
                                }
                            }
                        }
                    }
                }
            },
            401: SwaggerUtils.defoult_401,
            404: SwaggerUtils.defoult_404,
        }
    }

    faculty_api_getFacultyList = {
        "tags": ["Переменный состав", "Расписание занятий"],
        "summary": "Получить список факультетов (только действующих)",
        "description": textwrap.dedent("""
            **Запрос возвращает ответ даже если у пользователя нет никаких прав.**
        """),
        "parameters": [
            Param.lname(in_="path", required=True),
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — список факультетов",
                "schema": {
                    "type": "array",
                    "items": {
                        "type": "object",
                        "properties": {
                            "idfaculty": {
                                "type": "integer",
                                "description": "ID факультета",
                                "example": 1
                            },
                            "faculty": {
                                "type": "string",
                                "description": "Название факультета",
                                "example": "Факультет информатики"
                            }
                        }
                    }
                }
            },
            404: SwaggerUtils.defoult_404,
        }
    }

    group_api_get_groups_to_schedule_classes = {
        "tags": ["Расписание занятий"],
        "summary": "Получить список групп для составления расписания занятий",
        "description": textwrap.dedent("""
            **Запрос возвращает ответ даже если у него нет никаких прав.**

            Данный запрос используется для получения списка учебных групп, 
            к которым можно привязать расписание.  

            Результат зависит от:
            - выбранного учебного года (`xp_key`)
            - факультета (`idfaculty`)
            - курса обучения (`eyid`).

            🔹 Логика работы:
            - Всегда выбираются группы, связанные с указанным учебным годом (`xp_key` или его "главный").
            - Если указан `idfaculty != 0`, то фильтрация ограничивается только группами указанного факультета.
            - Если указан `eyid != 0`, то дополнительно применяется фильтрация по курсу обучения.
            - Если параметры `idfaculty=0` и `eyid=0`, то возвращаются все группы за указанный учебный год.
        """),
        "security": [SwaggerUtils.security],
        "parameters": [
            Param.lname(in_="path", required=True),
            Param.xp_key(in_="query", required=False),
            Param.idfaculty(in_="query", required=False),
            Param.eyid(in_="query", required=False),
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — список групп для расписания",
                "schema": {
                    "type": "array",
                    "items": {
                        "type": "object",
                        "properties": {
                            "gid": {
                                "type": "integer",
                                "description": "ID группы",
                                "example": 1012
                            },
                            "name": {
                                "type": "string",
                                "description": "Название группы",
                                "example": "ИСТ-21"
                            }
                        }
                    }
                }
            },
            401: SwaggerUtils.defoult_401,
            404: SwaggerUtils.defoult_404,
        }
    }

    group_api_get_groups_by_faculty_cathedra_year_eyid = {
        "tags": ["Переменный состав"],
        "summary": "Получить список учебных групп (режим «Переменный состав»)",
        "description": textwrap.dedent("""
            🔹 Логика работы:
            - Если пользователь имеет право **Справочники → Переменный состав → Администратор → Полный**,
              то ему доступны все группы.
            - Если прав администратора нет:
              - то применяются дополнительные фильтры:
                - ограничение по факультетам, кафедрам и подразделениям, доступным пользователю через `permission_groups`;
                - учёт того, что пользователь может быть **классным руководителем** (`class_teacher` / `class_teacher2`).

            🔹 Параметры фильтрации:
            - `idfaculty=0` → фильтр по факультету отключён.
            - `idcathedra=0` → фильтр по кафедре отключён.
            - `eyid=-1` → фильтр по курсу обучения отключён.
            - Если `xp_key > 0`, то отображается только название группы.
            - Если `xp_key = 0`, то к названию группы добавляется учебный год.
        """),
        "security": [SwaggerUtils.security],
        "parameters": [
            Param.lname(in_="path", required=True),
            Param.xp_key(in_="path", required=True),
            Param.idfaculty(in_="query", required=False),
            Param.eyid(in_="query", required=False),
            Param.idcathedra(in_="query", required=False),
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — список групп",
                "schema": {
                    "type": "array",
                    "items": {
                        "type": "object",
                        "properties": {
                            "gid": {
                                "type": "integer",
                                "description": "ID группы",
                                "example": 1012
                            },
                            "group_name": {
                                "type": "string",
                                "description": "Название группы (может содержать год обучения, если xp_key=0)",
                                "example": "ИСТ-21 (2024/2025)"
                            }
                        }
                    }
                }
            },
            401: SwaggerUtils.defoult_401,
            404: SwaggerUtils.defoult_404,
        }
    }

    user_api_getUserDetailsPhotoId = {
        "tags": ["Постоянный состав", "Переменный состав"],
        "summary": "Получить фотографию пользователя по его ID",
        "description": textwrap.dedent("""
            **Запрос возвращает фотографию пользователя даже если у него нет никаких прав.**
        """),
        "produces": ["image/jpeg"],
        "parameters": [
            Param.lname(in_="path", required=True),
            Param.mid(in_="path", required=True),
        ],
        "responses": {
            200: {
                "description": "✅ Фотография пользователя (image/jpeg)",
                "schema": {
                    "type": "string",
                    "format": "binary"
                },
                "headers": {
                    "Content-Disposition": {
                        "type": "string",
                        "description": "Имя файла в формате \"<id>.jpg\""
                    },
                    "Cache-control": {
                        "type": "string",
                        "description": "Заголовок кэширования (max-age=86400)"
                    }
                }
            },
            404: SwaggerUtils.defoult_404
        }
    }

    user_api_getUserList = {
        "tags": ["Постоянный состав"],
        "summary": "Получить список преподавателей (режим 'Постоянный состав')",
        "description": textwrap.dedent("""
                Если пользователь имеет **права администратора (Справочники → Постоянный состав → Администратор → Полный)**:
                  - возвращается полный список преподавателей.

                Если права **(Отчётность → Постоянный состав → Чтение) или (Отчётность → Постоянный состав → Полный)**:
                  - возвращается список только доступных преподавателей (фильтрация по подразделениям, к которым принадлежит запрашиваемый).

                Поддерживает:
                  - фильтрацию по статусу (`status`: works, dismissed, relocated, external, archieved, all)
                  - фильтрацию по подразделению (`division`: id)
                  - пагинацию (`paginateFrom`, `maxResults`)
                  - возврат только количества (`count=true`)
            """),
        "security": [SwaggerUtils.security],
        "parameters": [
            Param.lname(in_="path", required=True),
            Param.division(in_="query", required=False),
            Param.limit(in_="query", required=False),
            Param.offset(in_="query", required=False),
            Param.count(in_="query", required=False),
            {
                "name": "status",
                "in": "query",
                "type": "string",
                "required": False,
                "enum": ["works", "dismissed", "relocated", "external", "archieved", "all"],
                "description": "Фильтр по статусу сотрудника"
            },
        ],
        "responses": {
            200: {
                "description": "**Вариант 1 — список преподавателей:** Возвращается массив объектов Teacher",
                "schema": {
                    "type": "array",
                    "items": {"$ref": "#/definitions/Teacher"}
                }
            },
            201: {
                "description": "**Вариант 2 (возвращается 200) — количество записей:** Возвращается массив с одним объектом Count (если count=true)",
                "schema": {
                    "type": "array",
                    "items": {"$ref": "#/definitions/Count"}
                }
            },
            401: SwaggerUtils.defoult_401,
            404: SwaggerUtils.defoult_404,
        },
        "definitions": {
            "Teacher": {
                "type": "object",
                "properties": {
                    "id": {"type": "integer", "description": "ID преподавателя", "example": 602},
                    "gid": {"type": "integer", "description": "ID группы", "example": 123},
                    "lastname": {"type": "string", "description": "Фамилия", "example": "Иванов"},
                    "firstname": {"type": "string", "description": "Имя", "example": "Иван"},
                    "patronymic": {"type": "string", "description": "Отчество", "example": "Иванович"},
                    "birthday": {"type": "string", "description": "Дата рождения", "example": "12.05.1980"},
                    "email": {"type": "string", "description": "Электронная почта", "example": "ivanov@gmail.com"},
                    "photosize": {"type": "integer", "description": "Размер фото", "example": 12345},
                    "contact": {"type": "string", "description": "Контактные данные", "example": "+79999999999"},
                    "employername": {
                        "type": "string",
                        "description": "Подразделение в структуре",
                        "example": "Факультет информатики / Кафедра математики"
                    },
                    "position": {"type": "string", "description": "Должность",
                                 "example": "Заместитель начальника отдела"},
                    "type": {"type": "string", "description": "Тип персонала", "example": "Военнослужащие"},
                    "militaryrank": {"type": "string", "description": "Воинское звание",
                                     "example": "Старший лейтенант"},
                    "division": {"type": "string", "description": "Подразделение", "example": "Кафедра математики"},
                }
            },
            "Count": {
                "type": "object",
                "properties": {
                    "count": {"type": "integer", "description": "Количество найденных записей", "example": 152}
                }
            }
        }
    }

    cathedra_api_getCathedras = {
        "tags": ["Переменный состав"],
        "summary": "Получить список кафедр по ID факультета",
        "description": textwrap.dedent("""
            **Запрос возвращает ответ даже если у пользователя нет никаких прав.**

            Если `id=0` → возвращаются все действующие кафедры.
        """),
        "parameters": [
            Param.lname(in_="path", required=True),
            Param.idfaculty(in_="path", required=True),
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — список кафедр",
                "schema": {
                    "type": "array",
                    "items": {
                        "type": "object",
                        "properties": {
                            "idcathedra": {
                                "type": "integer",
                                "description": "ID кафедры",
                                "example": 12
                            },
                            "cathedra": {
                                "type": "string",
                                "description": "Название кафедры",
                                "example": "Кафедра информатики"
                            },
                            "faculty": {
                                "type": "integer",
                                "description": "ID факультета, к которому относится кафедра",
                                "example": 1
                            }
                        }
                    }
                }
            },
            404: SwaggerUtils.defoult_404
        }
    }

    user_api_getStudentList = {
        "summary": "Получить список студентов (режим 'Переменный состав')",
        "description": textwrap.dedent("""
            Администратор имеет права: 
            **Справочники → Переменный состав. Администратор → Полный**
        
            Данный запрос возвращает список студентов с учётом:
                - выбранного учебного года
                - факультета, кафедры, курса обучения
                - принадлежности к конкретной группе
                - статуса студента (текущий, выпускник, отчисленный и т. д.)
                - прав доступа пользователя (администратор или нет)
            
            🔹 Логика работы:
                - Если передан параметр `group`, то поиск выполняется только по указанной группе.
                - Если `group` не указан:
                  - выполняется выборка групп через метод `get_groups_by_faculty_cathedra_year_eyid`
                    с фильтрацией по факультету (`faculty`), кафедре (`cathedra`), курсу (`kurs`) и году (`year`).
                  - если список групп пуст, используется фиктивное значение `gid=0`, чтобы запрос не упал.
                - В зависимости от параметра `status`:
                  - `current` → текущие студенты (без отчисленных/выпускников/архивных).
                  - `deducted` → отчисленные в рамках указанного учебного года.
                  - `graduates` → выпускники (при наличии `year` — фильтрация по дате выпуска).
                  - `archieved` → архивные записи.
                  - `new` → вновь поступившие (без группы или только поступление).
                - Если параметр `count=true` → возвращается только количество записей.
                - **Если пользователь не администратор**:
                  - **То есть (Справочники -> Переменный состав. Прочие сведения -> Чтение) или (Справочники -> Переменный состав. Прочие сведения -> Полный)**
                  - применяется проверка прав доступа на основании `permission_groups` и автора записи (`whocreate_mid`).
            
            🔹 Поддерживает:
                - Поиск по ФИО (`search`)
                - Фильтр по категории (`catname`)
                - Пагинацию (`paginateFrom`, `maxResults`)
        """),
        "tags": ["Переменный состав"],
        "security": [SwaggerUtils.security],
        "parameters": [
            Param.lname(in_="path", required=True),
            Param.xp_key(in_="query", required=True, example=37),
            Param.idfaculty(in_="query", required=False, example=0),
            Param.idcathedra(in_="query", required=False, example=0),
            Param.eyid(in_="query", required=False, example=-1),  # курс
            Param.gid(in_="query", required=False),  # группа
            Param.search(in_="query", required=False, description="Поиск по ФИО (частичное совпадение)"),
            Param.count(in_="query", required=False, example=False),
            Param.limit(in_="query", required=False, example=50),
            Param.offset(in_="query", required=False, example=0),
            {
                "name": "status",
                "in": "query",
                "type": "string",
                "required": False,
                "enum": ["current", "deducted", "graduates", "archieved", "new"],
                "description": "Статус студента"
            },
            {
                "name": "catname",
                "in": "query",
                "type": "string",
                "required": False,
                "description": "Фильтр по категории (prep_struc_category.name)"
            }
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — список студентов или количество записей",
                "schema": {
                    "type": "array",
                    "items": {
                        "type": "object",
                        "properties": {
                            "mid": {"type": "integer", "description": "ID студента", "example": 10234},
                            "Фамилия Имя Отчество": {"type": "string", "example": "Иванов Иван Иванович"},
                            "Пол": {"type": "string", "example": "Мужской"},
                            "birthday": {"type": "string", "example": "2003-05-12"},
                            "groupname": {"type": "string", "example": "ИСТ-21"},
                            "Логин": {"type": "string", "example": "ivanov21"},
                            "Статус": {"type": "string", "example": "Включен в состав группы"},
                            "Категория": {"type": "string", "example": "Курсант"},
                            "Фото": {"type": "boolean", "example": True},
                            "Фамилия": {"type": "string", "example": "Иванов"},
                            "Имя": {"type": "string", "example": "Иван"},
                            "Субъект РФ": {"type": "string", "example": "Москва"},
                            "Отчество": {"type": "string"},
                            "Страна": {"type": "string", "example": "Россия"},
                            "Род войск (отец)": {"type": "string", "example": "Род войск"},
                            "Род войск (мать)": {"type": "string", "example": "Род войск"},
                            "Наставник": {"type": "string", "example": "Иванов Иван Иванович"},
                            "Льготы": {"type": "string", "example": "Льготы"},
                            "Куратор": {"type": "string", "example": "Иванов Иван Иванович"},
                            "Военный округ прибытия": {"type": "string", "example": ""},
                            "xp_pf_id": {"type": "integer", "example": 12345},
                            "patronymic": {"type": "string", "example": "Иванович"},
                            "lastname": {"type": "string", "example": "Иванов"},
                            "firstname": {"type": "string", "example": "Иван"},
                        }
                    }
                }
            },
            400: SwaggerUtils.defoult_400,
            401: SwaggerUtils.defoult_401,
            404: SwaggerUtils.defoult_404,
        }
    }

    year_api_get_education_year_by_school_year = {
        "tags": ["Работы/проекты обучающегося"],
        "summary": "Получить список курсов обучения за выбранный учебный год",
        "description": textwrap.dedent("""
            **Возвращает список записей из таблицы `educationyears`, связанных с указанным учебным годом (`school_year`).**
    
            Данный запрос используется для получения курсов (1, 2, 3 и т.д.), которые существуют в рамках выбранного учебного года.  
            В основе лежит вызов функции `xp_fill_groupuser_period_all`, которая подготавливает временные данные о распределении студентов по группам за указанный период.
    
            🔹 **Логика работы:**
            - Находит даты начала и конца указанного учебного года (`school_year`).
            - Заполняет временные данные в таблице `groupuser_period`.
            - Извлекает уникальные курсы (`educationyears`), которые присутствуют в группах (`group_history`) указанного учебного года.
            - Результат сортируется по возрастанию `eyid`.
            
            ⚙️ **Заявки:**
            6543
        """),
        "security": [SwaggerUtils.security],
        "parameters": [
            Param.lname(in_="path", required=True),
            Param.xp_key(in_="path", required=True, example=42)
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — список курсов обучения за указанный учебный год",
                "schema": {
                    "type": "array",
                    "items": {
                        "type": "object",
                        "properties": {
                            "eyid": {
                                "type": "integer",
                                "description": "Идентификатор записи курса обучения",
                                "example": 3
                            },
                            "number": {
                                "type": "integer",
                                "description": "Номер курса (1, 2, 3 и т.д.)",
                                "example": 2
                            },
                            "name": {
                                "type": "string",
                                "description": "Название курса обучения",
                                "example": "2 курс"
                            },
                            "archive": {
                                "type": "boolean",
                                "description": "Флаг архивного курса",
                                "example": False
                            }
                        }
                    }
                }
            },
            401: SwaggerUtils.defoult_401,
            404: SwaggerUtils.defoult_404,
        }
    }

    group_api_get_all_groups = {
        "tags": ["Работы/проекты обучающегося"],
        "summary": "Получить список всех учебных групп",
        "description": textwrap.dedent("""
            **Запрос возвращает список всех учебных групп** с возможностью фильтрации
            по учебному году (`xp_key`) и курсу обучения (`eyid`).

            🔹 **Логика работы:**
            - Если `xp_key` указан — возвращаются группы только за этот учебный год.
            - Если `eyid` указан — возвращаются группы только определённого курса.
            - Если оба параметра указаны — фильтрация применяется одновременно.
            - Если оба параметра равны 0 или не переданы — возвращаются все группы.
            
            ⚙️ **Заявки:**
            6543
        """),
        "security": [SwaggerUtils.security],
        "parameters": [
            Param.lname(in_="path", required=True),
            Param.xp_key(in_="query", required=True, example=42),
            Param.eyid(in_="query", required=False, example=3),
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — список групп",
                "schema": {
                    "type": "array",
                    "items": {
                        "type": "object",
                        "properties": {
                            "gid": {
                                "type": "integer",
                                "description": "ID группы",
                                "example": 1256
                            },
                            "name": {
                                "type": "string",
                                "description": "Название учебной группы",
                                "example": "ЭКО-23"
                            }
                        }
                    }
                }
            },
            401: SwaggerUtils.defoult_401,
            404: SwaggerUtils.defoult_404,
        }
    }

    course_api_get_courses_by_students_projects = {
        "tags": ["Работы/проекты обучающегося"],
        "summary": "Получить список учебных дисциплин для студенческих проектов",
        "description": textwrap.dedent("""
            **Запрос возвращает список учебных дисциплин**, связанных со студенческими проектами,
            с возможностью фильтрации по учебному году (`xp_key`), триместру (`trmid`)
            и курсу обучения (`eyid`).

            🔹 **Логика работы:**
            - Возвращаются только дисциплины, относящиеся к активным блокам дисциплин (тип блока 0, 1 или 2).
            - Если `xp_key` указан — фильтруются дисциплины по учебному году.
            - Если `trmid` указан — фильтрация выполняется по триместру.
            - Если `eyid` указан — фильтрация выполняется по курсу обучения (educationyears.eyid).
            - При отсутствии параметров возвращаются все доступные дисциплины для всех активных блоков.

            ⚙️ **Заявки:**
            6543
        """),
        "security": [SwaggerUtils.security],
        "parameters": [
            Param.lname(in_="path", required=True),
            Param.xp_key(in_="query", required=False, example=42),
            Param.trmid(in_="query", required=False, example=2),
            Param.eyid(in_="query", required=False, example=3)
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — список курсов, связанных со студенческими проектами",
                "schema": {
                    "type": "array",
                    "items": {
                        "type": "object",
                        "properties": {
                            "cid": {
                                "type": "integer",
                                "description": "Идентификатор дисциплины (`courses.cid`)",
                                "example": 4521
                            },
                            "title": {
                                "type": "string",
                                "description": "Название дисциплины",
                                "example": "Проектирование информационных систем"
                            }
                        }
                    }
                }
            },
            401: SwaggerUtils.defoult_401,
            404: SwaggerUtils.defoult_404,
        }
    }

    curriculum_api_get_curriculum_by_umk_teacher = {
        "tags": ["Электронные учебно-методические комплексы дисциплин"],
        "summary": "Получить список учебных планов (curriculum)",
        "description": textwrap.dedent("""
            **Запрос возвращает список учебных планов (curriculum)**, с возможностью фильтрации по учебному году, кафедре и дисциплине.

            🔹 **Логика работы:**
            - Если указан `xp_key` — возвращаются учебные планы, привязанные к выбранному учебному году.
            - Если указан `idcathedra` — фильтрация выполняется по кафедре (используется связь через `l_mcourse_cathedra`).
            - Если указан `idmetacourse` — фильтрация выполняется по конкретной дисциплине.
            - Если параметры равны `0` или не переданы, возвращаются все планы без ограничений.

            ⚙️ **Заявки:**
            6536
        """),
        "security": [SwaggerUtils.security],
        "parameters": [
            Param.lname(in_="path", required=True),
            Param.xp_key(in_="query", required=False, example=41),
            Param.idcathedra(in_="query", required=False, example=12),
            Param.idmetacourse(in_="query", required=False, example=58)
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — список учебных планов, соответствующих фильтрам",
                "schema": {
                    "type": "array",
                    "items": {
                        "type": "object",
                        "properties": {
                            "idcurriculum": {
                                "type": "integer",
                                "description": "Идентификатор учебного плана (`curriculum.idcurriculum`)",
                                "example": 301
                            },
                            "cname": {
                                "type": "string",
                                "description": "Название учебного плана",
                                "example": "Информатика и вычислительная техника (бакалавриат)"
                            }
                        }
                    }
                }
            },
            401: SwaggerUtils.defoult_401,
            404: SwaggerUtils.defoult_404,
        }
    }

    meta_course_api_get_meta_course_by_cathedra = {
        "tags": ["Электронные учебно-методические комплексы дисциплин"],
        "summary": "Получить список дисциплин (meta_course) по кафедре и учебному плану",
        "description": textwrap.dedent("""
            **Запрос возвращает список дисциплин (`meta_course`)**, связанных с указанной кафедрой
            и/или учебным планом (`curriculum`).

            🔹 **Логика работы:**
            - Если указан `idcurriculum`, то выбираются только дисциплины, связанные с курсами этого учебного плана.
            - Если указан `idcathedra`, то результат ограничивается дисциплинами, привязанными к данной кафедре.
            - Если оба параметра указаны — применяется комбинированная фильтрация.
            - Если оба параметра равны `0` — возвращаются все доступные дисциплины.

            ⚙️ **Заявки:**
            6536
        """),
        "security": [SwaggerUtils.security],
        "parameters": [
            Param.lname(in_="path", required=True, example="vimto"),
            Param.idcathedra(in_="query", required=False, example=153),
            Param.idcurriculum(in_="query", required=False, example=491)
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — список метакурсов, соответствующих фильтрам",
                "schema": {
                    "type": "array",
                    "items": {
                        "type": "object",
                        "properties": {
                            "id": {
                                "type": "integer",
                                "description": "Идентификатор дисциплины (`meta_course.id`)",
                                "example": 72
                            },
                            "name": {
                                "type": "string",
                                "description": "Название дисциплины",
                                "example": "Основы программирования и алгоритмизации"
                            }
                        }
                    }
                }
            },
            401: SwaggerUtils.defoult_401,
            404: SwaggerUtils.defoult_404,
        }
    }

    course_api_get_courses_by_curriculum_metacourse_cathedra = {
        "tags": ["Электронные учебно-методические комплексы дисциплин"],
        "summary": "Получить список дисциплин РПД по фильтрам (учебный план, дисциплина, кафедра, учебный год, год набора)",
        "description": textwrap.dedent("""
            **Запрос возвращает список дисциплин РПД (`courses`)**, отфильтрованных по комбинации параметров:
            учебный план (`curriculum`), дисциплина (`meta_course`), кафедра (`cathedra`),
            учебный год (`xp_key`) и год набора (`syid`).

            🔹 **Логика работы:**
            - Если `idcurriculum` указан — выбираются курсы, входящие в этот учебный план.
            - Если `idmetacourse` указан — результат фильтруется по указанной дисциплине.
            - Если `idcathedra` указан — возвращаются только курсы, закреплённые за этой кафедрой (`course2cathedra`).
            - Если `xp_key` указан — выбираются курсы за соответствующий учебный год (`school_year.xp_key`).
            - Если `syid` указан — выбираются курсы, относящиеся к конкретному году набора (`StudyYears.syid`).
            - При отсутствии параметров (`=0`) возвращаются все доступные курсы.

            ⚙️ **Заявки:**
            6536
        """),
        "security": [SwaggerUtils.security],
        "parameters": [
            Param.lname(in_="path", required=True),
            Param.idcurriculum(in_="query", required=False, example=305),
            Param.idmetacourse(in_="query", required=False, example=72),
            Param.idcathedra(in_="query", required=False, example=14),
            Param.xp_key(in_="query", required=False, example=41),
            Param.syid(in_="query", required=False, example=120)
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — список курсов, соответствующих указанным фильтрам",
                "schema": {
                    "type": "array",
                    "items": {
                        "type": "object",
                        "properties": {
                            "cid": {
                                "type": "integer",
                                "description": "Идентификатор дисциплины РПД (`courses.cid`)",
                                "example": 415
                            },
                            "predmet": {
                                "type": "string",
                                "description": "Название дисциплины РПД с индексом (формат: 'Название (индекс)')",
                                "example": "Высшая математика (МТ-102)"
                            }
                        }
                    }
                }
            },
            401: SwaggerUtils.defoult_401,
            404: SwaggerUtils.defoult_404,
        }
    }

    education_directions_api_get_many_mps = {
        "tags": ["Военные специальности"],
        "summary": "Получить список военных профессий (военно-учетных специальностей)",
        "description": textwrap.dedent("""
               **Возвращает список военных профессий (ВУС)**, связанных с направлением подготовки обучающегося.

               🔹 **Логика работы:**
               - Возвращаются только активные военные профессии.
               - Если по указанному направлению подготовки не найдено ни одной военной профессии — возвращается ошибка 404.

               ⚙️ **Заявки:**
               6462
           """),
        "security": [SwaggerUtils.security],
        "parameters": [
            Param.lname(in_="path", required=True)
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — список военных профессий",
                "schema": {
                    "type": "array",
                    "items": {
                        "type": "object",
                        "properties": {
                            "mpid": {
                                "type": "integer",
                                "description": "Идентификатор военной профессии",
                                "example": 602
                            },
                            "name": {
                                "type": "string",
                                "description": "Полное наименование военной профессии",
                                "example": "Применение картографических и навигационных воинских частей и подразделений и эксплуатация топогеодезической техники"
                            },
                            "shortname": {
                                "type": "string",
                                "description": "Краткое наименование",
                                "example": "ВО спец 72"
                            },
                            "specialization": {
                                "type": ["string", "null"],
                                "description": "Специализация внутри ВУС",
                                "example": "Геоинформационная картография"
                            },
                            "ed_specialization": {
                                "type": ["string", "null"],
                                "description": "Образовательная специализация, связанная с ВУС",
                                "example": "Геоинформационная картография, Астрономогеодезия, Аэрокосмическая фототопография"
                            },
                            "edu_direct_id": {
                                "type": "integer",
                                "description": "Идентификатор направления подготовки",
                                "example": 104
                            },
                            "foreign_mp": {
                                "type": "integer",
                                "description": "Флаг иностранных ВУС (0 — нет, 1 — да)",
                                "example": 0
                            },
                            "qualification": {
                                "type": "integer",
                                "description": "Уровень квалификации (резервное поле)",
                                "example": 0
                            },
                            "purposegraduates": {
                                "type": "string",
                                "description": "Цель подготовки выпускников (может быть пустым)",
                                "example": ""
                            },
                            "status": {
                                "type": "integer",
                                "description": "Статус записи (например, 2 — активна)",
                                "example": 2
                            },
                            "code": {
                                "type": ["string", "null"],
                                "description": "Код ВУС (если применимо)",
                                "example": None
                            }
                        }
                    }
                }
            },
            401: SwaggerUtils.defoult_401,
            404: {
                "description": "❌ Не найдено военных профессий по указанному направлению подготовки (`lname`)",
                "schema": {
                    "type": "object",
                    "properties": {
                        "success": {
                            "type": "boolean",
                            "example": False
                        }
                    }
                }
            }
        }
    }

    group_api_get_get_groups_military_professional_portrait_of_a_student = {
        "tags": ["Военно-профессиональный портрет обучающегося"],
        "summary": "Получить список групп для режима «Военно-профессиональный портрет обучающегося»",
        "description": textwrap.dedent("""
            **Возвращает список учебных групп**, доступных в рамках режима «Военно-профессиональный портрет обучающегося».

            🔹 **Логика работы:**
            - Поддерживается фильтрация по учебному году (`year_id`), факультету (`faculty_id`)
              и военной профессии (`military_profession_id`).
            - Если параметр фильтра не указан или равен 0 — фильтрация по нему не применяется.
            - Возвращаются только группы, связанные с указанными критериями и имеющие обучающихся.

            ⚙️ **Заявки:**
            6462
        """),
        "security": [SwaggerUtils.security],
        "parameters": [
            Param.lname(in_="path", required=True),
            {
                "name": "year_id",
                "in": "query",
                "type": "integer",
                "required": False,
                "default": 0,
                "description": "Идентификатор учебного года (`educationyears.eyid`). Если 0 — фильтр не применяется.",
                "example": 3
            },
            {
                "name": "faculty_id",
                "in": "query",
                "type": "integer",
                "required": False,
                "default": 0,
                "description": "Идентификатор факультета (`faculties.fid`). Если 0 — фильтр не применяется.",
                "example": 12
            },
            {
                "name": "military_profession_id",
                "in": "query",
                "type": "integer",
                "required": False,
                "default": 0,
                "description": "Идентификатор военной профессии (`mpid`). Если 0 — фильтр не применяется.",
                "example": 602
            }
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — список групп",
                "schema": {
                    "type": "array",
                    "items": {
                        "type": "object",
                        "properties": {
                            "gid": {
                                "type": "integer",
                                "description": "Идентификатор группы (`groups.gid`)",
                                "example": 338
                            },
                            "name": {
                                "type": "string",
                                "description": "Название группы",
                                "example": "642"
                            }
                        }
                    }
                }
            },
            401: SwaggerUtils.defoult_401,
            404: SwaggerUtils.defoult_404
        }
    }

    questionset_api_get_question_and_answer_by_test = {
        "tags": ["Вопросники / Тесты"],
        "summary": "Получить вопросы и ответы по заданному вопроснику (тесту)",
        "description": textwrap.dedent("""
            **Возвращает полный список вопросов и связанных с ними вариантов ответов** для указанного вопросника (`id`).

            🔹 **Особенности:**
            - Доступ зависит от роли пользователя: преподаватели видят все данные, обучающиеся — только в рамках разрешённого доступа.
            - Возвращаются как текст вопроса, так и все возможные ответы с признаком правильности (`correct`).
            - Поддерживаются разные типы вопросов (одиночный выбор, множественный выбор, текстовый ввод и др.), определяемые полем `question_type`.

            ⚙️ **Заявки:**
            6472
        """),
        "security": [SwaggerUtils.security],
        "parameters": [
            Param.lname(in_="path", required=True),
            {
                "name": "id",
                "in": "path",
                "type": "integer",
                "required": True,
                "description": "Идентификатор вопросника (теста)",
                "example": 12345
            }
        ],
        "responses": {
            200: {
                "description": "✅ Успешный ответ — список вопросов с вариантами ответов",
                "schema": {
                    "type": "array",
                    "items": {
                        "type": "object",
                        "properties": {
                            "question_id": {
                                "type": "integer",
                                "description": "Идентификатор вопроса",
                                "example": 2013772
                            },
                            "question_text": {
                                "type": "string",
                                "description": "Текст вопроса",
                                "example": "На картинке изображена собака?\\r\\nесли правда, то напишите \\\"да\\\""
                            },
                            "question_type": {
                                "type": "integer",
                                "description": "Тип вопроса: 1 — одиночный выбор, 2 — множественный выбор, 3 — текстовый ответ и т.д.",
                                "example": 3
                            },
                            "q_num": {
                                "type": "integer",
                                "description": "Порядковый номер вопроса в тесте",
                                "example": 1
                            },
                            "score": {
                                "type": "number",
                                "format": "float",
                                "description": "Максимальный балл за вопрос",
                                "example": 1.0
                            },
                            "weight": {
                                "type": "number",
                                "format": "float",
                                "description": "Вес вопроса (используется при расчёте итогового результата)",
                                "example": 1.0
                            },
                            "worktime": {
                                "type": "integer",
                                "description": "Рекомендуемое время на ответ (в секундах)",
                                "example": 1
                            },
                            "cid": {
                                "type": "integer",
                                "description": "Идентификатор дисциплины",
                                "example": 6978
                            },
                            "eyid": {
                                "type": "integer",
                                "description": "Идентификатор учебного года (0 — если не привязан)",
                                "example": 0
                            },
                            "oid": {
                                "type": "integer",
                                "description": "Идентификатор организации (0 — если не привязан)",
                                "example": 0
                            },
                            "tlid": {
                                "type": "integer",
                                "description": "Идентификатор тематического блока (0 — если не привязан)",
                                "example": 0
                            },
                            "status": {
                                "type": "integer",
                                "description": "Статус вопроса (например, 0 — активен)",
                                "example": 0
                            },
                            "id_parent": {
                                "type": ["integer", "null"],
                                "description": "Идентификатор родительского вопроса (для вложенных структур)",
                                "example": None
                            },
                            "answer_id": {
                                "type": ["integer", "null"],
                                "description": "Идентификатор выбранного пользователем ответа (в данном эндпоинте всегда null, так как возвращаются шаблонные данные)",
                                "example": None
                            },
                            "answers": {
                                "type": "array",
                                "description": "Список возможных вариантов ответа",
                                "items": {
                                    "type": "object",
                                    "properties": {
                                        "id": {
                                            "type": "integer",
                                            "description": "Идентификатор варианта ответа",
                                            "example": 6041331
                                        },
                                        "answer_text": {
                                            "type": "string",
                                            "description": "Текст варианта ответа",
                                            "example": "да"
                                        },
                                        "correct": {
                                            "type": "boolean",
                                            "description": "Является ли ответ правильным",
                                            "example": True
                                        },
                                        "answer_group": {
                                            "type": "integer",
                                            "description": "Группа ответа (для логических группировок, например, в drag-and-drop)",
                                            "example": 1
                                        },
                                        "answer_pos": {
                                            "type": ["string", "null"],
                                            "description": "Уникальный идентификатор позиции (GUID для UI-компоновки)",
                                            "example": "{133D4F1E-4A63-43D5-AC02-2CEF26364871}"
                                        },
                                        "answer_html": {
                                            "type": ["string", "null"],
                                            "description": "HTML-представление ответа (если используется)",
                                            "example": None
                                        },
                                        "answer_length": {
                                            "type": ["integer", "null"],
                                            "description": "Ожидаемая длина текстового ответа (для type=3)",
                                            "example": None
                                        },
                                        "sel_start": {
                                            "type": ["integer", "null"],
                                            "description": "Начало выделения (для интерактивных заданий)",
                                            "example": None
                                        },
                                        "sel_len": {
                                            "type": ["integer", "null"],
                                            "description": "Длина выделения",
                                            "example": None
                                        },
                                        "value_num": {
                                            "type": ["number", "null"],
                                            "description": "Числовое значение (для числовых ответов)",
                                            "example": None
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            },
            401: SwaggerUtils.defoult_401,
            404: {
                "description": "❌ Вопросник не найден или недоступен для пользователя",
                "schema": {
                    "type": "object",
                    "properties": {
                        "success": {
                            "type": "boolean",
                            "example": False
                        }
                    }
                }
            }
        }
    }