from flask import current_app
from sqlalchemy.orm import sessionmaker

from LMSAPI import db
from LMSAPI.api.Models.TeacherReview import TeacherReviewScore, TeacherReviewScoreService


class CriteriaTeacherEvaluation(db.Model):
    __tablename__ = 'criteria_teacher_evaluation'

    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.Text, nullable=False, unique=True)
    is_archive = db.Column(db.Boolean, default=False)

    def __repr__(self):
        return "<Criteria(id={id}, name='{name}')>".format(id=self.id, name=self.name)

    def to_dict(self):
        return {
            'id': self.id,
            'name': self.name,
            'is_archive': self.is_archive
        }

class CriteriaTeacherEvaluationService:
    def __init__(self, lname, session=None):
        if session:
            self.session = session
            self.lname = lname
        else:
            engine = current_app.ms.db(lname)
            Session = sessionmaker(bind=engine)
            self.session = Session()
            self.lname = lname # теперь можно .query()

    def get_all(self, include_archived):
        query = self.session.query(CriteriaTeacherEvaluation)
        if not include_archived:
            query = query.filter_by(is_archive=False)
        return [c.to_dict() for c in query.order_by(CriteriaTeacherEvaluation.name).all()]

    def get_criteria_by_name(self, name):
        return self.session.query(CriteriaTeacherEvaluation).filter_by(name=name).first()

    def get_criteria_by_id(self, criteria_id):
        return self.session.query(CriteriaTeacherEvaluation).get(criteria_id)

    def check_existing_criteria(self, new_name, criteria_id):
        return self.session.query(CriteriaTeacherEvaluation).filter(
            CriteriaTeacherEvaluation.name == new_name,
            CriteriaTeacherEvaluation.id != criteria_id
        ).first()

    def create_criteria(self, name, is_archive=False):
        if CriteriaTeacherEvaluationService(self.lname).get_criteria_by_name(name):
            raise ValueError("Критерий с таким названием уже существует")

        new_criteria = CriteriaTeacherEvaluation(name=name.strip(), is_archive=is_archive)
        self.session.add(new_criteria)
        self.session.commit()
        return new_criteria

    def update_criteria(self, criteria_id, new_name=None, new_is_archive=None):
        criteria = self.get_criteria_by_id(criteria_id)
        is_used = TeacherReviewScoreService(self.lname).check_criterion_use(criteria_id=criteria_id)
        existing = self.check_existing_criteria(new_name=new_name, criteria_id=criteria_id)

        if not criteria:
            raise ValueError("Критерий не найден")

        if new_is_archive is not None and new_name is not None:
            criteria.is_archive = new_is_archive

            if not existing and not is_used:
                criteria.name = new_name
                self.session.commit()
                return criteria, None
            elif existing:
                self.session.commit()
                return criteria, "Критерий с таким названием уже существует"
            else:
                self.session.commit()
                return criteria, "Нельзя изменить название критерия — он уже использовался в отзывах"

        if new_is_archive is not None:
            criteria.is_archive = new_is_archive

        if new_name is not None:
            # Проверка использования
            if is_used:
                raise ValueError("Нельзя изменить название критерия — он уже использовался в отзывах")

            if existing:
                raise ValueError("Критерий с таким названием уже существует")

            criteria.name = new_name

        self.session.commit()
        return criteria, None

