from decimal import Decimal
from typing import Tuple, Union, List, Dict, Any

from flask import current_app
from sqlalchemy import text
from sqlalchemy.orm import sessionmaker

from LMSAPI import db

class ScholarshipType(db.Model):
    __tablename__ = 'scholarship_types'

    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.Text, nullable=False, unique=True)
    is_editable = db.Column(db.Boolean, nullable=False, default=True)

    settings = db.relationship("ScholarshipSetting", backref="scholarship_type", cascade="all, delete-orphan")

    def serialize(self):
        return {
            "id": self.id,
            "name": self.name,
            "is_editable": self.is_editable
        }



class ScholarshipSetting(db.Model):
    __tablename__ = 'scholarship_settings'

    id = db.Column(db.Integer, primary_key=True)
    scholarship_type_id = db.Column(db.Integer, db.ForeignKey('scholarship_types.id', ondelete="CASCADE"), nullable=False)
    xp_key = db.Column(db.Integer, nullable=False)
    education_level = db.Column(db.Integer, nullable=False)
    prep_struc_category_id = db.Column(db.Integer, nullable=False)
    course_from = db.Column(db.Integer)
    course_to = db.Column(db.Integer)
    average_grade = db.Column(db.Numeric(3, 2))
    max_average_grade = db.Column(db.Numeric(3, 2))
    min_achievements = db.Column(db.Integer)
    max_achievements = db.Column(db.Integer)
    date_from = db.Column(db.Date)
    date_to = db.Column(db.Date)
    trmids = db.relationship("ScholarshipSettingsTrmid", backref="setting", cascade="all, delete-orphan")

    def serialize(self, lname):
        return {
            "id": self.id,
            "scholarship_type_id": self.scholarship_type_id,
            "xp_key": self.xp_key,
            "education_level": self.education_level,
            "prep_struc_category_id": self.prep_struc_category_id,
            "course_from": self.course_from,
            "course_to": self.course_to,
            "average_grade": float(self.average_grade) if self.average_grade is not None else None,
            "max_average_grade": float(self.max_average_grade) if self.max_average_grade is not None else None,
            "min_achievements": self.min_achievements,
            "max_achievements": self.max_achievements,
            "date_from": self.date_from,
            "date_to": self.date_to,
            "trmids": ScholarshipSettingsTrmidService(lname).get_trmids(self.id)
        }


class ScholarshipSettingsTrmid(db.Model):
    __tablename__ = 'scholarship_settings_trmid'

    scholarship_settings_id = db.Column(db.Integer, db.ForeignKey('scholarship_settings.id', ondelete="CASCADE"), primary_key=True)
    trmid = db.Column(db.Integer, primary_key=True)

    def serialize(self):
        return {
            "scholarship_settings_id": self.scholarship_settings_id,
            "trmid": self.trmid
        }


class ScholarshipTypeService:
    def __init__(self, lname, session=None):
        if session:
            self.session = session
            self.lname = lname
        else:
            engine = current_app.ms.db(lname)
            Session = sessionmaker(bind=engine)
            self.session = Session()
            self.lname = lname

    def get_by_id(self, id_: int):
        return self.session.query(ScholarshipType).get(id_)

    def get_all(self):
        return self.session.query(ScholarshipType).all()

    def get_existing_name(self, name):
        return self.session.query(ScholarshipType).filter_by(name=name).first()

    def create(self, name, is_editable=True):
        existing = self.get_existing_name(name)
        if existing:
            raise ValueError("Тип с именем '{name}' уже существует".format(name=name))

        item = ScholarshipType(
            name=name,
            is_editable=is_editable
        )
        self.session.add(item)
        self.session.commit()
        return item.serialize()

    def update(self, id_, **kwargs):
        item = self.get_by_id(id_)
        if not item:
            raise ValueError("Элемент не найден")
        for key, value in kwargs.items():
            if hasattr(item, key):
                setattr(item, key, value)
        self.session.commit()
        return item.serialize()

    def delete(self, id_):
        item = self.get_by_id(id_)
        if not item:
            raise ValueError("Элемент не найден")
        self.session.delete(item)
        self.session.commit()
        return item.serialize()


class ScholarshipSettingService:
    def __init__(self, lname, session=None):
        if session:
            self.session = session
            self.lname = lname
        else:
            engine = current_app.ms.db(lname)
            Session = sessionmaker(bind=engine)
            self.session = Session()
            self.lname = lname

    def get_by_id(self, id_: int):
        return self.session.query(ScholarshipSetting).get(id_)

    def get_all(self, scholarship_type_id=None, xp_key=None):
        query = self.session.query(ScholarshipSetting)
        if scholarship_type_id is not None:
            query = query.filter(ScholarshipSetting.scholarship_type_id == scholarship_type_id)
        if xp_key is not None:
            query = query.filter(ScholarshipSetting.xp_key == xp_key)

        return query.all()

    def check_levels_of_education(self):
        sql = """
            SELECT DISTINCT
                psc.cid AS prep_struc_category_id,
                psc.name AS prep_struc_category,
                1 AS education_level
            FROM groupname g
            JOIN groupuser gu ON gu.gid = g.gid
            JOIN militaryprofession mp ON g.f_militaryprofession = mp.mpid
            JOIN edu_direction ed ON mp.edu_direct_id = ed.edu_direct_id
            JOIN preparation_structure ps ON ed.program_level = ps.ps_id
            JOIN xp_personal_file e ON e.mid = gu.mid 
            JOIN prep_struc_category psc ON psc.cid = e.category
            WHERE ps.education_level IN (1, 2) AND psc.cid IN (1, 2, 3)

            UNION ALL

            SELECT DISTINCT
                1 AS prep_struc_category_id,
                'СПО' AS prep_struc_category,
                2 AS education_level
            FROM groupname g
            JOIN groupuser gu ON gu.gid = g.gid
            JOIN militaryprofession mp ON g.f_militaryprofession = mp.mpid
            JOIN edu_direction ed ON mp.edu_direct_id = ed.edu_direct_id
            JOIN preparation_structure ps ON ed.program_level = ps.ps_id
            WHERE ps.education_level = 2
            ORDER BY prep_struc_category_id DESC
        """

        stmt = text(sql)
        result_proxy = self.session.execute(stmt)
        rows = result_proxy.fetchall()
        keys = result_proxy.keys()
        return [dict(zip(keys, row)) for row in rows]

    def create(self, data: dict):
        trmids = data.pop('trmids', None)
        setting = ScholarshipSetting(**data)
        self.session.add(setting)
        self.session.commit()
        if trmids is not None:
            ScholarshipSettingsTrmidService(self.lname).set_trmids(setting.id, trmids)
        return setting

    def update(self, data: dict):
        setting_id = data.pop("id")
        setting = self.get_by_id(setting_id)
        trmids = data.pop('trmids', None)
        if not setting:
            raise ValueError("Настройка не найдена")

        for key, value in data.items():
            if hasattr(setting, key):
                setattr(setting, key, value)
        self.session.commit()
        if trmids is not None:
            ScholarshipSettingsTrmidService(self.lname).set_trmids(setting_id, trmids)
        return setting

    def delete(self, id_: int):
        setting = self.get_by_id(id_)
        result = setting.serialize(self.lname)
        if not setting:
            raise ValueError("Настройка не найдена")

        self.session.delete(setting)
        self.session.commit()
        return result


class ScholarshipSettingsTrmidService:
    def __init__(self, lname, session=None):
        if session:
            self.session = session
            self.lname = lname
        else:
            engine = current_app.ms.db(lname)
            Session = sessionmaker(bind=engine)
            self.session = Session()
            self.lname = lname

    def set_trmids(self, setting_id, trmids):
        self.delete_by_setting_id(setting_id)
        for trm_id in trmids:
            link = ScholarshipSettingsTrmid(scholarship_settings_id=setting_id, trmid=trm_id)
            self.session.add(link)
        self.session.commit()

    def get_trmids(self, setting_id):
        sql = text("""
            SELECT sst.trmid, t.title
            FROM scholarship_settings_trmid sst
            JOIN terms t ON t.trmid = sst.trmid
            WHERE sst.scholarship_settings_id = :setting_id
        """)
        results = self.session.execute(sql, {"setting_id": setting_id}).fetchall()
        return [{"trmid": row.trmid, "title": row.title} for row in results]

    def delete_by_setting_id(self, setting_id: int):
        self.session.query(ScholarshipSettingsTrmid).filter_by(scholarship_settings_id=setting_id).delete()
        self.session.commit()
