import json
from datetime import datetime, timedelta

from flask import current_app, g
from sqlalchemy import CheckConstraint, text
from sqlalchemy.orm import sessionmaker

from LMSAPI import db
from LMSAPI.api.utils.request_utils import RequestUtils


class TeacherReview(db.Model):
    __tablename__ = 'teacher_reviews'

    id = db.Column(db.Integer, primary_key=True)
    teacher_id = db.Column(db.Integer, nullable=False)
    student_id = db.Column(db.Integer, nullable=False)
    review = db.Column(db.Text)
    created_at = db.Column(db.DateTime, default=datetime.utcnow)
    anonymous = db.Column(db.Boolean, default=False, server_default=text('false'))

    # Relationships
    scores = db.relationship("TeacherReviewScore", backref="review", cascade="all, delete-orphan")

    def __repr__(self):
        return "<Review(id={id}, teacher_id={teacher_id}, student_id={student_id})>".format(id=self.id, teacher_id=self.teacher_id, student_id=self.student_id)


class TeacherReviewScore(db.Model):
    __tablename__ = 'teacher_review_scores'

    id = db.Column(db.Integer, primary_key=True)
    review_id = db.Column(db.Integer, db.ForeignKey('teacher_reviews.id', ondelete="CASCADE"), nullable=False)
    criteria_id = db.Column(db.Integer, db.ForeignKey('criteria_teacher_evaluation.id'), nullable=False)
    score = db.Column(db.Integer, nullable=False)

    __table_args__ = (
        CheckConstraint('score >= 1 AND score <= 5', name='score_range'),
    )

    criteria = db.relationship("CriteriaTeacherEvaluation")

    def __repr__(self):
        return "<Score(review_id={review_id}, criteria_id={criteria_id}, score={score})>".format(review_id=self.review_id, criteria_id=self.criteria_id, score=self.score)


class TeacherReviewVote(db.Model):
    __tablename__ = 'teacher_review_votes'

    id = db.Column(db.Integer, primary_key=True)
    review_id = db.Column(db.Integer, db.ForeignKey('teacher_reviews.id', ondelete='CASCADE'), nullable=False)
    user_id = db.Column(db.Integer, nullable=False)
    vote_type = db.Column(db.String(10), nullable=False)  # 'like' или 'dislike'
    created_at = db.Column(db.DateTime, default=datetime.utcnow)

    __table_args__ = (
        db.UniqueConstraint('review_id', 'user_id', name='uq_review_user'),
        db.CheckConstraint("vote_type IN ('like', 'dislike')", name='check_vote_type'),
    )

    def to_dict(self):
        return {
            'id': self.id,
            'review_id': self.review_id,
            'user_id': self.user_id,
            'vote_type': self.vote_type,
            'created_at': self.created_at.isoformat()
        }



class TeacherReviewService:
    def __init__(self, lname):
        self.lname = lname
        engine = current_app.ms.db(self.lname)  # вернёт Engine
        Session = sessionmaker(bind=engine)
        self.session = Session()  # теперь можно .query()

    def get_review_by_id(self, review_id: int):
        return self.session.query(TeacherReview).get(review_id)

    def create_review(self, teacher_id, student_id, comment, anonymous, scores: list):
        # Проверка: не оставлял ли уже отзыв последние 30 дней
        existing = self.session.query(TeacherReview).filter(
            TeacherReview.teacher_id == teacher_id,
            TeacherReview.student_id == student_id,
            TeacherReview.created_at >= datetime.utcnow() - timedelta(days=30)
        ).first()

        if existing:
            raise ValueError("Вы уже оставляли отзыв в течение последних 30 дней")

        # Создание основного отзыва
        review = TeacherReview(
            teacher_id=teacher_id,
            student_id=student_id,
            review=comment,
            anonymous=anonymous
        )
        self.session.add(review)
        self.session.commit()

        # Добавляем оценки по критериям
        for item in scores:
            TeacherReviewScoreService(self.lname).create_score(review.id, item.get("criteria_id"), item.get("score"))
        return review

    def delete_review(self, review_id):
        review = self.get_review_by_id(review_id)
        if not review:
            raise ValueError("Отзыв не найден")

        self.session.delete(review)
        self.session.commit()

    def get_review_statistics(self, limit=None, offset=None, search=None):
        filters = ""
        params = {"current_user_id": g.user.mid}

        if search:
            filters += """
                HAVING LOWER(p.lastname || ' ' || p.firstname || ' ' || p.patronymic) LIKE :search
            """
            params["search"] = "%{s}%".format(s=search.lower())

        base_sql = """
                WITH stats AS (
                    SELECT
                        t.mid AS mid,
                        p.lastname || ' ' || p.firstname || ' ' || p.patronymic AS fio,
                        COUNT(DISTINCT tr.id) AS count_reviews,
                        ROUND(AVG(trs.score), 2) AS average_score
                    FROM teacher_reviews tr
                    JOIN teachers t ON t.mid = tr.teacher_id
                    JOIN people p ON p.mid = t.mid
                    JOIN xp_personal_file pf ON pf.mid=p.mid AND pf.workstatus='Работает'
                    JOIN teacher_review_scores trs ON trs.review_id = tr.id
                    LEFT JOIN teacher_review_votes trv ON trv.review_id = tr.id
                    GROUP BY t.mid, p.lastname, p.firstname, p.patronymic
                    {filters}
                )
                SELECT
                    (SELECT COUNT(*) FROM stats) AS total_count,
                    json_agg(s) AS statistics
                FROM (
                    SELECT * FROM stats
                    ORDER BY fio
                    {limit}
                    {offset}
                ) s
            """.format(
            filters=filters,
            limit='LIMIT {limit}'.format(limit=limit) if limit is not None else '',
            offset='OFFSET {offset}'.format(offset=offset) if offset is not None else ''
        )

        result = self.session.execute(base_sql, params).fetchone()
        statistics = result["statistics"]
        return {
            "count": result["total_count"],
            "statistics": RequestUtils.convert_to_float(statistics)
        }

    def get_reviews_by_teacher(self, teacher_id: int, limit=None, offset=None):
        sql = """
            SELECT 
                tr.teacher_id,
                teacher.lastname || ' ' || teacher.firstname || ' ' || teacher.patronymic AS teacher_fio,
                COUNT(DISTINCT tr.id) AS count_reviews,
                ROUND((
                    SELECT AVG(score)
                    FROM teacher_review_scores s
                    JOIN teacher_reviews r ON s.review_id = r.id
                    WHERE r.teacher_id = tr.teacher_id
                ), 2) AS average_score,
                (
                    SELECT json_agg(review_data ORDER BY created_at DESC)
                    FROM (
                        SELECT 
                            tr.id AS review_id,
                            CASE WHEN NOT tr.anonymous THEN tr.student_id ELSE NULL END AS student_id,
                            CASE WHEN NOT tr.anonymous THEN p.lastname || ' ' || p.firstname || ' ' || p.patronymic ELSE NULL END AS student_fio,
                            tr.review AS comment,
                            tr.created_at,
                            tr.anonymous,
                            (
                                SELECT json_agg(json_build_object(
                                    'criteria_id', cte.id,
                                    'criteria_name', cte.name,
                                    'score', trs.score
                                ) ORDER BY cte.name)
                                FROM teacher_review_scores trs
                                JOIN criteria_teacher_evaluation cte ON cte.id = trs.criteria_id
                                WHERE trs.review_id = tr.id
                            ) AS scores,
                            ROUND((
                                SELECT AVG(score) FROM teacher_review_scores WHERE review_id = tr.id
                            ), 2) AS average_score,
                            COALESCE(likes.count_like, 0) AS likes,
                            COALESCE(dislikes.count_dislike, 0) AS dislikes,
                            CASE WHEN uv.vote_type = 'like' THEN true ELSE false END AS liked_by_me,
                            CASE WHEN uv.vote_type = 'dislike' THEN true ELSE false END AS disliked_by_me
                        FROM teacher_reviews tr
                        LEFT JOIN people p ON p.mid = tr.student_id
                        LEFT JOIN (
                            SELECT review_id, COUNT(*) AS count_like
                            FROM teacher_review_votes
                            WHERE vote_type = 'like'
                            GROUP BY review_id
                        ) AS likes ON likes.review_id = tr.id
                        LEFT JOIN (
                            SELECT review_id, COUNT(*) AS count_dislike
                            FROM teacher_review_votes
                            WHERE vote_type = 'dislike'
                            GROUP BY review_id
                        ) AS dislikes ON dislikes.review_id = tr.id
                        LEFT JOIN (
                            SELECT review_id, vote_type
                            FROM teacher_review_votes
                            WHERE user_id = :user_id
                        ) AS uv ON uv.review_id = tr.id
                        WHERE tr.teacher_id = :teacher_id
                        ORDER BY tr.created_at DESC
                        LIMIT :limit
                        OFFSET :offset
                    ) AS review_data
                ) AS reviews
            FROM teacher_reviews tr
            LEFT JOIN people p ON p.mid = tr.student_id
            LEFT JOIN people teacher ON teacher.mid = tr.teacher_id
            LEFT JOIN (
                SELECT review_id, COUNT(*) AS count_like
                FROM teacher_review_votes
                WHERE vote_type = 'like'
                GROUP BY review_id
            ) AS likes ON likes.review_id = tr.id
            LEFT JOIN (
                SELECT review_id, COUNT(*) AS count_dislike
                FROM teacher_review_votes
                WHERE vote_type = 'dislike'
                GROUP BY review_id
            ) AS dislikes ON dislikes.review_id = tr.id
            LEFT JOIN (
                SELECT review_id, vote_type
                FROM teacher_review_votes
                WHERE user_id = :user_id
            ) AS uv ON uv.review_id = tr.id
            WHERE tr.teacher_id = :teacher_id
            GROUP BY tr.teacher_id, teacher.lastname, teacher.firstname, teacher.patronymic
        """

        query = self.session.execute(sql, {
            "teacher_id": teacher_id,
            "user_id": g.user.mid,
            "limit": limit,
            "offset": offset
        })
        row = query.fetchone()
        return RequestUtils.convert_to_float(dict(row)) if row else None


class TeacherReviewScoreService:
    def __init__(self, lname):
        self.lname = lname
        engine = current_app.ms.db(self.lname)  # вернёт Engine
        Session = sessionmaker(bind=engine)
        self.session = Session()  # теперь можно .query()

    def check_criterion_use(self, criteria_id: int):
        return self.session.query(TeacherReviewScore.id).filter_by(criteria_id=criteria_id).first()

    def create_score(self, review_id: int, criteria_id: int, score: int):
        if not criteria_id or not (1 <= score <= 5):
            raise ValueError("Некорректные данные в критерии: {item}".format(item={"criteria_id": criteria_id, "score": score}))
        new_score = TeacherReviewScore(
            review_id=review_id,
            criteria_id=criteria_id,
            score=score
        )
        self.session.add(new_score)
        self.session.commit()
        return new_score


class TeacherReviewVoteService:
    def __init__(self, lname):
        self.lname = lname
        engine = current_app.ms.db(self.lname)
        Session = sessionmaker(bind=engine)
        self.session = Session()

    def get_vote_by_user_and_review(self, user_id: int, review_id: int):
        return self.session.query(TeacherReviewVote).filter_by(
            review_id=review_id,
            user_id=user_id
        ).first()

    def create_vote(self, user_id: int, review_id: int, vote_type: str):
        new_vote = TeacherReviewVote(
            review_id=review_id,
            user_id=user_id,
            vote_type=vote_type
        )
        self.session.add(new_vote)
        self.session.commit()
        return new_vote

    def delete_vote(self, user_id: int, review_id: int):
        vote = self.get_vote_by_user_and_review(user_id, review_id)
        if vote is not None:
            self.session.delete(vote)
            self.session.commit()
        return vote

    def get_review_vote_summary(self, review_id: int):
        sql = text("""
            SELECT
                :review_id AS review_id,
                COUNT(*) FILTER (WHERE vote_type = 'like') AS likes,
                COUNT(*) FILTER (WHERE vote_type = 'dislike') AS dislikes,
                COALESCE(BOOL_OR(user_id = :user_id AND vote_type = 'like'), false) AS liked_by_me,
                COALESCE(BOOL_OR(user_id = :user_id AND vote_type = 'dislike'), false) AS disliked_by_me
            FROM (SELECT 1) AS dummy
            LEFT JOIN teacher_review_votes v ON v.review_id = :review_id
        """)

        result = self.session.execute(sql, {"review_id": review_id, "user_id": g.user.mid}).fetchone()

        return dict(result)

    def vote_review(self, review_id: int, user_id: int, vote_type: str):
        if vote_type not in ['like', 'dislike']:
            raise ValueError("vote_type должен быть 'like' или 'dislike'")

        vote = self.get_vote_by_user_and_review(user_id, review_id)

        if vote is None:
            # Создаём новую запись
            new_vote = self.create_vote(user_id, review_id, vote_type)
            return {"status": "added", "vote": self.get_review_vote_summary(review_id)}

        if vote.vote_type != vote_type:
            # Обновляем существующий голос
            vote.vote_type = vote_type
            vote.created_at = datetime.utcnow()
            self.session.commit()
            return {"status": "updated", "vote": self.get_review_vote_summary(review_id)}

        else:
            # Удаляем голос (повторное нажатие)
            self.delete_vote(user_id, review_id)
            return {"status": "removed", "vote": self.get_review_vote_summary(review_id)}
