from flask import jsonify, request, current_app, g, send_from_directory
from flask import Blueprint

from LMSAPI.api.Models.File import File
from LMSAPI.api.Models.UserActionLog import UserActionLog
from LMSAPI.api.Views.TokenAPI import auth
from LMSAPI.api.Models.Cathedra import Cathedra
from flask_cors import CORS
import json

from LMSAPI.api.utils.access_utils import user_permission_modes
from LMSAPI.api.utils.swagger_utils import swagger_doc
from LMSAPI.api.utils.swagger_comments import SwaggerComments

# https://stackoverflow.com/questions/39769666/flask-how-to-use-app-context-inside-blueprints
# Use app context from blueprint

cathedra_api = Blueprint("cathedra_api", __name__)

CORS(cathedra_api)


# Get all cathedras for a faculty
@cathedra_api.route("/lms/api/v1.0/<lname>/cathedra/f/<int:idfaculty>", methods=["GET"])
@swagger_doc(SwaggerComments.cathedra_api_getCathedras)
def getCathedras(lname, idfaculty):
    """
    Получить список кафедр по ID факультета
    """
    C = Cathedra(lname)
    result = C.getCathedraList(idfaculty)

    if result is None:
        return jsonify(success=False), 404

    serializable_list = [fa.serialize() for fa in result]

    return jsonify(serializable_list)


@cathedra_api.route("/lms/api/v1.0/<lname>/cathedra/by_user", methods=["GET"])
@auth.login_required
def get_cathedra_user(lname):
    """returns a json of the user's cathedras"""
    mid = g.user.mid
    cathedra_user = Cathedra(lname).get_cathedra_list_user(lname, mid)
    return (
        jsonify([cu.serialize() for cu in cathedra_user]),
        200,
        {"Content-Type": "text/json; charset=utf-8"},
    )


# Get all cathedras for a faculty (Teacher)
@cathedra_api.route(
    "/lms/api/v1.0/<lname>/cathedra/teacher/f/<int:id>", methods=["GET"]
)
@auth.login_required
def getCathedrasForTeacher(id, lname):
    """GET with faculty ID to get list of Cathedra"""
    if not g.user.isTeacher:
        return jsonify({"error": "Только для учителя"})
    C = Cathedra(lname)
    result = C.getCathedraListForTeacher(id, g.user.mid)

    if result is None:
        return jsonify(success=False), 404

    serializable_list = [fa.serialize() for fa in result]

    return jsonify(serializable_list)


@cathedra_api.route(
    "/lms/api/v1.0/<lname>/journal_of_accounting_of_training_sessions/cathedra/teacher/f/<int:id>", methods=["GET"]
)
@auth.login_required
def get_cathedras_for_teacher_journal_of_accounting_of_training_sessions(id, lname):
    """
    Режим: "Журнал учета учебных занятий"
    """
    if not g.user.isTeacher:
        return jsonify({"error": "Только для учителя"})
    C = Cathedra(lname)
    result = C.get_cathedra_list_for_teacher_journal_of_accounting_of_training_sessions(id)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


# Get all cathedras for a faculty
@cathedra_api.route("/lms/api/v1.0/<lname>/cathedra/<int:id>", methods=["GET"])
@auth.login_required
@user_permission_modes("Справочники", "Кафедры", ["Нет"])
def getCathedra(id, lname):
    """GET with ID to get all info for a Cathedra"""
    C = Cathedra(lname)
    result = C.getCathedra(id)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@cathedra_api.route("/lms/api/v1.0/<lname>/cathedra/f/<int:id>", methods=["POST"])
@auth.login_required
@user_permission_modes("Справочники", "Кафедры", ["Нет"])
def addCathedra(id, lname):
    data = request.data
    dataDict = json.loads(data)
    cathedraid = Cathedra(lname).addCathedra(id, dataDict)

    if cathedraid is None:
        return jsonify(success=False), 404

    return jsonify({"cathedraid": cathedraid})


@cathedra_api.route(
    "/lms/api/v1.0/<lname>/cathedra/<int:cathedra_id>/<filename>", methods=["GET"]
)
@auth.login_required
@user_permission_modes("Справочники", "Кафедры", ["Нет"])
def get_cathedra_file(lname, cathedra_id, filename):
    file_path = File(lname).get_cathedra_directory(cathedra_id=cathedra_id)
    ip_address = (
        request.headers.get("X-Forwarded-For")
        or request.headers.get("X-Real-IP")
        or request.environ.get("REMOTE_ADDR")
    )
    return File.send_from_directory(
        lname=lname,
        file_path=file_path,
        file_name=filename,
        action_source=request.environ.get("HTTP_USER_AGENT"),
        login=g.user.id,
        user_mid=g.user.mid,
        ip_address=ip_address,
    )


@cathedra_api.route("/lms/api/v1.0/<lname>/journal/cathedra", methods=["GET"])
@auth.login_required
@auth.login_required
def get_journal_cathedras(lname):
    """GET with faculty ID to get list of Cathedra"""
    C = Cathedra(lname)
    result = C.get_journal_cathedras()
    if result is None:
        return jsonify(success=False), 404
    return jsonify(result)


@cathedra_api.route("/lms/api/v1.0/<lname>/ktp/cathedra", methods=["GET"])
@auth.login_required
def get_ktp_cathedras(lname):
    result = Cathedra(lname).get_ktp_cathedras()
    if result is None:
        return jsonify(success=False), 404
    return jsonify(result)


@cathedra_api.route("/lms/api/v1.0/<lname>/umk/cathedra", methods=["GET"])
@auth.login_required
def get_umk_cathedras(lname):
    result = Cathedra(lname).get_umk_cathedras()
    if result is None:
        return jsonify(success=False), 404
    return jsonify(result)
