from flask import Flask, jsonify, request, current_app, g
from flask_restful import Resource, Api
from sqlalchemy import create_engine
from sqlalchemy.sql import text
from flask import Blueprint
from LMSAPI.api.Views.TokenAPI import auth
from LMSAPI.api.Models.User import User
from LMSAPI.api.Models.Message import Message
from flask_cors import CORS
import json, sys

# https://stackoverflow.com/questions/39769666/flask-how-to-use-app-context-inside-blueprints
# Use app context from blueprint

message_api = Blueprint('message_api', __name__)

CORS(message_api)


@message_api.route('/lms/api/v1.0/<lname>/message', methods=['GET'])
@auth.login_required
def getUnreadCount(lname):
    """GET unread count of messages"""
    um = Message(lname).getMessageCount(g.user.mid, 0)
    return jsonify({'unread': um})


@message_api.route('/lms/api/v1.0/<lname>/message/users', methods=['GET'])
@auth.login_required
def searchUsers(lname):
    """Search getUsers"""
    composition = request.args.get('composition', default=None, type=str)
    user = request.args.get('user', default=None, type=str)
    um = Message(lname).search_users(user, composition)
    return jsonify(um)


@message_api.route('/lms/api/v1.0/<lname>/message/chat', methods=['GET'])
@auth.login_required
def getChats(lname):
    """Get chat list"""

    chats = Message(lname).getChatList(g.user.mid)
    return jsonify(chats)


@message_api.route('/lms/api/v1.0/<lname>/message/chat', methods=['POST'])
@auth.login_required
def createChat(lname):
    """Create a new chat"""
    data = request.get_json(silent=True) or {}

    tomid = data.get('tomid', '')
    title = data.get('title', '')
    message = data.get('message', '')

    if not tomid:
        return jsonify(message="Выберите получателя"), 400
    if not title:
        return jsonify(message="Заполните тему сообщения"), 400
    if len(title) > 255:
        return jsonify(message="Превышено допустимое количество символов 255"), 400
    if not message:
        return jsonify(message="Заполните текст сообщения"), 400

    chatid = Message(lname).newChat(g.user.mid, tomid, title, message)

    return jsonify({"chatid": chatid})


@message_api.route('/lms/api/v1.0/<lname>/message/chat/<int:id>', methods=['GET'])
@auth.login_required
def getChatInfo(lname, id):
    """Get info for chat id"""

    chatinfo = Message(lname).getChatInfo(g.user.mid, id)
    return jsonify({"chatinfo": chatinfo})


@message_api.route('/lms/api/v1.0/<lname>/message/chat/<int:id>', methods=['DELETE'])
@auth.login_required
def deleteChat(lname, id):
    """Delete chat"""
    result = Message(lname).deleteChat(g.user.mid, id)
    return jsonify({"result": result})


@message_api.route('/lms/api/v1.0/<lname>/message/chat/<int:id>/message', methods=['GET'])
@auth.login_required
def getChatMessages(lname, id):
    """Get messages for chat id"""
    messages = Message(lname).getChat(g.user.mid, id)
    maxid = 0

    for m in messages:
        if m['id'] > maxid:
            maxid = m['id']

    Message(lname).updateMessagesChatFrom(g.user.mid, id, maxid)
    Message(lname).updateMessagesChatTo(g.user.mid, id, maxid)
    return jsonify({"messages": messages})


@message_api.route('/lms/api/v1.0/<lname>/message/chat/<int:id>/message/<int:messageid>', methods=['DELETE'])
@auth.login_required
def deleteChatMessage(lname, id, messageid):
    """Delete message for chat id"""
    result = Message(lname).deleteMessage(messageid, g.user.mid)
    return jsonify({"result": result})


@message_api.route('/lms/api/v1.0/<lname>/message/chat/<int:id>/message', methods=['POST'])
@auth.login_required
def createChatMessage(lname, id):
    """Create message for chat id"""

    data = request.data
    if sys.version_info[0] < 3:
        dataDict = json.loads(data)
    else:
        dataDict = json.loads(data.decode('utf-8'))

    current_app.logger.debug(dataDict)

    parent = dataDict['parent']
    message = dataDict['message']

    messageid = Message(lname).newMessage(id, parent, g.user.mid, message)
    Message(lname).updateMessagesChatFrom(g.user.mid, id, messageid)
    return jsonify({"messageid": messageid})
