# coding=UTF-8
import base64
import sys
from decimal import Decimal

from flask import jsonify, request, json, g
from flask import Blueprint

from LMSAPI.api.Models.EnrollAnswers import EnrollAnswers
from LMSAPI.api.Models.EnrollQuestions import EnrollQuestions
from LMSAPI.api.Models.File import File
from LMSAPI.api.Models.Journal import Journal
from LMSAPI.api.Models.QuestionAnswers import QuestionAnswers
from LMSAPI.api.Models.QuestionSet import QuestionSet, QuestionSetContent
from LMSAPI.api.Views.TokenAPI import auth
from flask_cors import CORS

from LMSAPI.api.utils.swagger_comments import SwaggerComments
from LMSAPI.api.utils.swagger_utils import swagger_doc

questionset_api = Blueprint("questionset_api", __name__)

CORS(questionset_api)


@questionset_api.route(
    "/lms/api/v1.0/<lname>/questionset",
    methods=["GET"],
)
@auth.login_required
def get_questionset_all(lname):
    cid = request.args.get("cid", default=None, type=int)
    private = request.args.get("private", default=None)
    oid = request.args.get("oid", default=None, type=int)

    # Корректная обработка значения параметра private
    if private is not None:
        private = private.lower() in ["true", "1", "t", "y", "yes"]

    result = QuestionSet().get_questionset_all(lname, cid, private, oid)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@questionset_api.route(
    "/lms/api/v1.0/<lname>/questionset/<int:id>/question/answers",
    methods=["GET"],
)
@auth.login_required
@swagger_doc(SwaggerComments.questionset_api_get_question_and_answer_by_test)
def get_question_and_answer_by_test(lname, id):
    result = QuestionSet().get_question_and_answer_by_test(lname, id, g.user.isTeacher)

    if result is None:
        return jsonify(success=False), 404

    # Convert Decimal to float
    for row in result:
        for key, value in row.items():
            if isinstance(value, Decimal):
                row[key] = float(value)

    return jsonify(result)


@questionset_api.route(
    "/lms/api/v1.0/<lname>/questionset/<int:id>",
    methods=["GET"],
)
@auth.login_required
def get_questionset_by_id(lname, id: int):
    result = QuestionSet().get_questionset_by_id(lname, id)

    if result is None:
        return jsonify(success=False), 404

    for key, value in result.items():
        if isinstance(value, Decimal):
            result[key] = float(value)

    availability_answers = QuestionAnswers().get_availability_answers_test_questions(
        lname, g.user.mid, id
    )
    if availability_answers["trying"] < result["trying"]:
        result["availability_answers"] = True
        result["user_trying"] = availability_answers["trying"]
    else:
        result["availability_answers"] = False
        result["user_trying"] = availability_answers["trying"]

    return jsonify(result)


@questionset_api.route(
    "/lms/api/v1.0/<lname>/questionset",
    methods=["POST"],
)
@auth.login_required
def create_questionset(lname):
    data = request.data
    if not data:
        return jsonify({"error": "No data provided"}), 400

    if sys.version_info[0] < 3:
        data = json.loads(data)
    else:
        data = json.loads(data.decode("utf-8"))

    # Проверка на наличие всех обязательных полей
    missing_fields = [
        field
        for field in ["name", "cid", "private", "description", "oid"]
        if field not in data
    ]
    if missing_fields:
        return (
            jsonify(
                {
                    "error": "Missing required fields: {message}".format(
                        message=", ".join(missing_fields)
                    )
                }
            ),
            400,
        )

    result = QuestionSet().create_questionset(lname, data)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(success=True), 201


@questionset_api.route(
    "/lms/api/v1.0/<lname>/questionset/<int:id>",
    methods=["PUT"],
)
@auth.login_required
def update_questionset(lname, id: int):
    data = request.data
    if not data:
        return jsonify({"error": "No data provided"}), 400

    if sys.version_info[0] < 3:
        data = json.loads(data)
    else:
        data = json.loads(data.decode("utf-8"))

    result = QuestionSet().update_questionset(lname, id, data)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(success=True), 201


@questionset_api.route(
    "/lms/api/v1.0/<lname>/questionset/<int:id>",
    methods=["DELETE"],
)
@auth.login_required
def delete_questionset(lname, id: int):
    result = QuestionSet().delete_questionset(lname, id)

    if result is None:
        return jsonify(success=False), 404
    return jsonify(success=True), 200


@questionset_api.route(
    "/lms/api/v1.0/<lname>/questionset/content",
    methods=["GET"],
)
@auth.login_required
def get_questionset_content_all(lname):
    id_set = request.args.get("id_set", default=None, type=int)
    id_question = request.args.get("id_question", default=None, type=int)

    result = QuestionSetContent().get_questionset_content_all(
        lname, id_question, id_set
    )
    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@questionset_api.route(
    "/lms/api/v1.0/<lname>/questionset/content",
    methods=["POST"],
)
@auth.login_required
def create_questionset_content(lname):
    data = request.data
    if not data:
        return jsonify({"error": "No data provided"}), 400

    if sys.version_info[0] < 3:
        data = json.loads(data)
    else:
        data = json.loads(data.decode("utf-8"))

    # Проверка на наличие всех обязательных полей
    missing_fields = [
        field for field in ["id_question", "id_set", "sortnum"] if field not in data
    ]
    if missing_fields:
        return (
            jsonify(
                {
                    "error": "Missing required fields: {message}".format(
                        message=", ".join(missing_fields)
                    )
                }
            ),
            400,
        )

    result = QuestionSetContent().create_questionset_content(lname, data)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(success=True), 201


@questionset_api.route(
    "/lms/api/v1.0/<lname>/questionset/content/<int:id_question>/<int:id_set>",
    methods=["DELETE"],
)
@auth.login_required
def delete_questionset_content(lname, id_question: int, id_set: int):
    result = QuestionSetContent().delete_questionset_content(lname, id_question, id_set)

    if result is None:
        return jsonify(success=False), 404
    return jsonify(success=True), 200


@questionset_api.route(
    "/lms/api/v1.0/<lname>/questionset/<int:id>/user/statistic",
    methods=["GET"],
)
@auth.login_required
def get_statistics_by_questionset(lname, id):
    count_question, text_question, correct_question_answers, incorrect_question_answers, user_score, add_grade_blocked, user_trying, user_correct_percentage = QuestionSet().get_stats_by_questionset(lname, id, g.user.mid)
    return jsonify(
        {
            "count_all_question": count_question,
            "text_question": text_question,
            "multiple_choice_questions": count_question - text_question,
            "correct_question_answers": correct_question_answers,
            "incorrect_question_answers": incorrect_question_answers,
            "user_correct_percentage": user_correct_percentage,
        }
    )

@questionset_api.route(
    "/lms/api/v1.0/<lname>/questionset/<int:id>/user/grade",
    methods=["POST"],
)
@auth.login_required
def add_grade_by_questionset(lname, id):
    questionset = QuestionSet().get_questionset_by_id(lname, id)
    try:
        count_question, text_question, correct_question_answers, incorrect_question_answers, user_score, add_grade_blocked, user_trying, user_correct_percentage = QuestionSet().get_stats_by_questionset(lname, id, g.user.mid)
    except:
        return jsonify(
            success=False,
            message="Нет такого типа вопроса",
        ), 404

    # Определяем итоговую оценку grade только если не заблокировано
    grade = None
    if add_grade_blocked == 0:
        ball3 = questionset.get("ball3")
        ball4 = questionset.get("ball4")
        ball5 = questionset.get("ball5")

        if user_score < ball3:
            grade = '2'
        elif ball3 <= user_score < ball4:
            grade = '3'
        elif ball4 <= user_score < ball5:
            grade = '4'
        else:
            grade = '5'

    # Если пользователь уже проходил тест, то не добавляем его в журнал
    lesson_info = QuestionSet().get_sheid_and_week_id_by_questionset(lname, id)
    if add_grade_blocked == 0 and grade is not None:
        for lesson in lesson_info:
            grade_info = Journal().get_grades_by_sheid_week_mid(lname, lesson["sheid"], lesson["week_id"], g.user.mid)
            new_grade = grade
            new_comment = "По результатам {user_trying} попытки, выставленна оценка {grade}".format(user_trying=user_trying, grade=grade)
            if grade_info:
                new_grade = grade_info["grade"] + grade
                new_comment = grade_info["comment"] + '\n- По результатам {user_trying} попытки, выставленна оценка {grade}'.format(user_trying=user_trying, grade=grade)

            Journal().setGrade(
                lname,
                lesson["sheid"],
                lesson["week_id"],
                new_grade,
                g.user.mid,
                questionset["author"],
                new_comment
            )

        return jsonify(
            {
                "count_all_question": count_question,
                "text_question": text_question,
                "multiple_choice_questions": count_question - text_question,
                "correct_question_answers": correct_question_answers,
                "incorrect_question_answers": incorrect_question_answers,
                "user_correct_percentage": user_correct_percentage,
                "user_trying": user_trying,
                "trying": questionset["trying"],
                "grade": grade,
            }
        )

    return jsonify(
        {
            "count_all_question": count_question,
            "text_question": text_question,
            "multiple_choice_questions": count_question - text_question,
            "correct_question_answers": correct_question_answers,
            "incorrect_question_answers": incorrect_question_answers,
            "user_correct_percentage": user_correct_percentage,
            "user_trying": user_trying,
            "trying": questionset["trying"]
        }
    )


@questionset_api.route(
    "/lms/api/v1.0/<lname>/questionset/<int:id>/file",
    methods=["GET"],
)
@auth.login_required
def get_question_file(lname, id):
    file_path = File(lname).get_question_directory(id)
    files_name = [file["file"] for file in File(lname).get_question_files(id)]

    files = []
    for file in files_name:
        with open(file_path + "/" + file, 'rb') as f:
            files.append({
                'name': file,
                'content': base64.b64encode(f.read()).decode('utf-8'),
                'mime': 'image/jpeg'
            })

    return jsonify({'files': files})
