from flask import Blueprint, g
from flask_cors import CORS
from flask import request, jsonify

from LMSAPI.api.Models.ScholarshipListOptions import ScholarshipListOptions
from LMSAPI.api.Models.ScholarshipTypes import ScholarshipTypeService, ScholarshipSettingService, \
    ScholarshipSettingsTrmidService
from LMSAPI.api.Views.TokenAPI import auth

scholarship_types = Blueprint("scholarship_types", __name__, url_prefix="/lms/api/v1.0/<lname>")
CORS(scholarship_types)


@scholarship_types.route("/scholarship_types", methods=["GET"])
@auth.login_required
def get_scholarship_types(lname):
    types = ScholarshipTypeService(lname).get_all()
    return jsonify([item.serialize() for item in types]), 200


@scholarship_types.route("/scholarship_types", methods=["POST"])
@auth.login_required
def create_scholarship_type(lname):
    data = request.get_json()
    try:
        new_type = ScholarshipTypeService(lname).create(data["name"], data["is_editable"])
        return jsonify(new_type), 201
    except ValueError as ve:
        return jsonify({"error": str(ve)}), 400
    except Exception as e:
        return jsonify({"error": "Внутренняя ошибка"}), 500


@scholarship_types.route("/scholarship_types/<int:type_id>", methods=["PUT"])
@auth.login_required
def update_scholarship_type(lname, type_id):
    data = request.get_json()
    try:
        new_type = ScholarshipTypeService(lname).update(type_id, name=data["name"])
        return jsonify(new_type), 200
    except Exception as e:
        return jsonify({"error": str(e)}), 400


@scholarship_types.route("/scholarship_types/<int:type_id>", methods=["DELETE"])
@auth.login_required
def delete_scholarship_type(lname, type_id):
    try:
        delete_type = ScholarshipTypeService(lname).delete(type_id)
        return jsonify(delete_type), 200
    except Exception as e:
        return jsonify({"error": str(e)}), 400

# CRUD for ScholarshipSetting
@scholarship_types.route("/scholarship_settings", methods=["GET"])
@auth.login_required
def get_scholarship_settings(lname):
    scholarship_type_id = request.args.get("scholarship_type_id", type=int)
    xp_key = request.args.get("xp_key", type=int)

    settings = ScholarshipSettingService(lname).get_all(
        scholarship_type_id=scholarship_type_id,
        xp_key=xp_key
    )
    return jsonify([item.serialize(lname) for item in settings]), 200


@scholarship_types.route("/scholarship_settings/education_levels", methods=["GET"])
@auth.login_required
def get_scholarship_types_education_levels(lname):
    education_levels = ScholarshipSettingService(lname).check_levels_of_education()
    return jsonify(education_levels), 200


@scholarship_types.route("/scholarship_settings", methods=["POST"])
@auth.login_required
def create_scholarship_setting(lname):
    data = request.get_json()
    try:
        settings = []
        for item in data:
            setting = ScholarshipSettingService(lname).create(item)
            settings.append(setting.serialize(lname))
        return jsonify(settings), 201
    except Exception as e:
        return jsonify({"error": str(e)}), 400


@scholarship_types.route("/scholarship_settings", methods=["PUT"])
@auth.login_required
def update_scholarship_setting(lname):
    data = request.get_json()
    try:
        updated_list = []
        for item in data:
            updated = ScholarshipSettingService(lname).update(item)
            updated_list.append(updated.serialize(lname))
        return jsonify(updated_list), 200
    except Exception as e:
        return jsonify({"error": str(e)}), 400


@scholarship_types.route("/scholarship_settings/<int:setting_id>", methods=["DELETE"])
@auth.login_required
def delete_scholarship_setting(lname, setting_id):
    try:
        result = ScholarshipSettingService(lname).delete(setting_id)
        return jsonify(result), 200
    except Exception as e:
        return jsonify({"error": str(e)}), 400