# coding=UTF-8
# LMSAPI/API/Journal.py
from flask import current_app, g
import pickle

from sqlalchemy import text


class EnrollAnswers:
    # Список возможных полей для таблицы EnrollQuestions кроме id
    possible_fields = [
        "question",
        "answer_text",
        "answer_group",
        "answer_length",
        "answer_pos",
        "answer_html",
        "sel_start",
        "sel_len",
        "correct",
        "value_num",
    ]

    def get_enroll_answers_all(self, lname, question, answer_group, correct):
        conn = current_app.ms.db(lname).connect()
        where = ""
        if question:
            where += " and question = :question "
        if answer_group:
            where += " and answer_group = :answer_group "
        if correct is not None:
            where += " and correct = :correct "

        sql = """
            select * from enroll_answers
            where 1=1
            {where}
            """.format(
            where=where
        )

        stmt = text(sql)
        if question:
            stmt = stmt.bindparams(question=question)
        if answer_group:
            stmt = stmt.bindparams(answer_group=answer_group)
        if correct is not None:
            stmt = stmt.bindparams(correct=correct)

        query = conn.execute(stmt)
        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    def get_enroll_answers_by_id(self, lname, id):
        conn = current_app.ms.db(lname).connect()

        sql = """
            select * from enroll_answers
            where id = :id
            """
        stmt = text(sql)
        stmt = stmt.bindparams(id=id)
        query = conn.execute(stmt)
        result = query.fetchone()

        if result is None:
            return None

        result_dict = dict(zip(query.keys(), result))
        return result_dict

    def create_enroll_answers(self, lname, data):
        conn = current_app.ms.db(lname).connect()

        params = {}
        for field in self.possible_fields:
            params[field] = data[field] if field in data else None

        sql = """
            insert into enroll_answers(question, answer_text, answer_group, answer_length, answer_pos, answer_html, 
                                       sel_start, sel_len, correct, value_num)
            values (:question, :answer_text, :answer_group, :answer_length, :answer_pos, :answer_html, :sel_start,
                    :sel_len, :correct, :value_num)
            returning id
            """
        stmt = text(sql)
        stmt = stmt.bindparams(**params)
        query = conn.execute(stmt)

        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    def update_enroll_answers(self, lname, id, data):
        conn = current_app.ms.db(lname).connect()

        # Формируем SET выражение и параметры
        set_clauses = []
        params = {"id": id}

        for field in self.possible_fields:
            if data.get(field) is not None:
                set_clauses.append("{field} = :{field}".format(field=field))
                params[field] = data.get(field)

        # Проверяем, есть ли поля для обновления
        if not set_clauses:
            raise ValueError("No fields to update")

        # Формируем SQL запрос
        set_clause = ", ".join(set_clauses)
        sql = """
            UPDATE enroll_answers
            SET {set_clause}
            WHERE id = :id
            RETURNING id
        """.format(
            set_clause=set_clause
        )

        stmt = text(sql).bindparams(**params)
        query = conn.execute(stmt)

        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    def delete_enroll_answers(self, lname, id):
        conn = current_app.ms.db(lname).connect()
        sql = """
            DELETE FROM enroll_answers
            WHERE id = :id
            RETURNING id
        """

        stmt = text(sql).bindparams(id=id)
        query = conn.execute(stmt)

        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    def delete_enroll_answers_by_question(self, lname, question_id):
        conn = current_app.ms.db(lname).connect()
        sql = """
            DELETE FROM enroll_answers
            WHERE question = :question_id
            RETURNING id
        """

        stmt = text(sql).bindparams(question_id=question_id)
        query = conn.execute(stmt)

        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    def get_answers_by_test(self, lname, id):
        conn = current_app.ms.db(lname).connect()

        sql = """
            WITH RECURSIVE question_hierarchy AS (
                SELECT 
                    q.*
                FROM 
                    enroll_questions q
                WHERE 
                    q.id IN (
                        SELECT id_question 
                        FROM questionset_content 
                        WHERE id_set = :id
                    )
                UNION ALL
                SELECT 
                    q.*
                FROM 
                    enroll_questions q
                JOIN 
                    question_hierarchy qh ON q.id_parent = qh.id
            )
            SELECT 
                a.*
            FROM 
                question_hierarchy qh
            LEFT JOIN 
                enroll_answers a ON qh.id = a.question
            """

        stmt = text(sql)
        stmt = stmt.bindparams(id=id)

        query = conn.execute(stmt)
        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    def get_correct_and_incorrect_counts(self, lname, id):
        conn = current_app.ms.db(lname).connect()

        sql = """
            WITH RECURSIVE question_hierarchy AS (
                SELECT 
                    q.*
                FROM 
                    enroll_questions q
                WHERE 
                    q.id IN (
                        SELECT id_question 
                        FROM questionset_content 
                        WHERE id_set = :id
                    )
                UNION ALL
                SELECT 
                    q.*
                FROM 
                    enroll_questions q
                JOIN 
                    question_hierarchy qh ON q.id_parent = qh.id
            )
            SELECT 
                SUM(CASE WHEN a.correct = true THEN 1 ELSE 0 END) AS correct_count,
                SUM(CASE WHEN a.correct = false OR a.correct IS NULL THEN 1 ELSE 0 END) AS incorrect_count
            FROM 
                question_hierarchy qh
            LEFT JOIN 
                enroll_answers a ON qh.id = a.question
            """

        stmt = text(sql)
        stmt = stmt.bindparams(id=id)

        query = conn.execute(stmt)
        result = query.fetchone()

        return result["correct_count"], result["incorrect_count"]
