# LMSAPI/API/Faculty.py

from sqlalchemy import create_engine
from sqlalchemy.sql import text
from LMSAPI.api.Models.User import User
from flask import current_app, g
import pickle


class Faculty:
    def __init__(self, cname):
        self.idfaculty = 1
        self.faculty = ""
        self.lname = cname

    def serialize(self):
        return {"idfaculty": self.idfaculty, "faculty": self.faculty}

    def get_faculty_list_from_redis(self):
        cachedFlist = current_app.ms.redis(self.lname).get("SYSFLIST")
        if cachedFlist is not None:
            flist = pickle.loads(cachedFlist)
            return flist
        return

    def set_faculty_list_to_redis(self, flist):
        current_app.ms.redis(self.lname).setex("SYSFLIST", 5, pickle.dumps(flist))
        return

    def get_faculty_from_redis(self, id):
        cachedFlist = current_app.ms.redis(self.lname).get("FA" + str(id))
        if cachedFlist is not None:
            flist = pickle.loads(cachedFlist)
            return flist
        return

    def set_faculty_to_redis(self, result):
        current_app.ms.redis(self.lname).setex(
            "FA" + str(result["faculty"][0]["idfaculty"]), 5, pickle.dumps(result)
        )
        return

    def addFaculty(self, faculty):
        conn = current_app.ms.db(self.lname).connect()

        sql = """INSERT INTO faculty(
	 faculty, state, "ShortName", prefix_code)
	VALUES (:faculty, :state, :shortname, :prefix_code);"""
        stmt = text(sql)

        stmt = stmt.bindparams(faculty=faculty["faculty"])
        stmt = stmt.bindparams(state=faculty["state"])
        stmt = stmt.bindparams(shortname=faculty["shortname"])
        stmt = stmt.bindparams(prefix_code=faculty["prefix_code"])

        conn.execute(stmt)

        sql = "SELECT currval(pg_get_serial_sequence('faculty','idfaculty'))"
        stmt = text(sql)
        query = conn.execute(stmt)
        newId = 0
        for row in query:
            newId = row.currval

        return newId

    def getFaculty(self, lname, id):
        fc = Faculty(lname)
        result = fc.get_faculty_from_redis(id)

        if result is not None:
            return result

        conn = current_app.ms.db(lname).connect()
        stmt = text("""
            with cte AS 
            (SELECT 
                sdid | 268435456 AS id,
                COALESCE(l.levelname,'Подразделение') AS "Тип",
                0 AS "Kind",
                divisionname::character varying(254) AS name
              FROM 
              struct_divisions s
              LEFT JOIN struct_divisions_levels l ON l.sdlid=s.sdlid
              UNION 
                SELECT  
                0, 
                'Академия',
                -1,
                (SELECT  convert_from(value, 'utf8') FROM options WHERE name = 'GOU_Name')::character varying(254) 
              ORDER BY 2,4
            )
            select f.*, tfo.full_title as object_umb, cte.name as higher_unit
            from faculty f
            left join training_facilites_objects tfo ON tfo.tfoid = f.tfoid
            left join cte ON f.f_owner = cte.id
            where f.idfaculty = :id
        """)
        stmt = stmt.bindparams(id=id)
        query = conn.execute(stmt)

        result = {"faculty": [dict(zip(tuple(query.keys()), i)) for i in query.cursor]}

        if len(result["faculty"]) == 0:
            return None

        dean = User(lname)

        dean = dean.get_user_from_sql_id(result["faculty"][0]["dean"])
        if dean is not None:
            result["faculty"][0]["dean"] = dean.serialize()

        adean = User(lname)

        adean = adean.get_user_from_sql_id(result["faculty"][0]["AssistantDean"])
        if adean is not None:
            result["faculty"][0]["AssistantDean"] = adean.serialize()

        EducationYearChief = User(lname)
        EducationYearChief = EducationYearChief.get_user_from_sql_id(
            result["faculty"][0]["EducationYearChief"]
        )
        if EducationYearChief is not None:
            result["faculty"][0]["EducationYearChief"] = EducationYearChief.serialize()

        EducationYearOfficer = User(lname)
        EducationYearOfficer = EducationYearOfficer.get_user_from_sql_id(
            result["faculty"][0]["EducationYearOfficer"]
        )
        if EducationYearOfficer is not None:
            result["faculty"][0][
                "EducationYearOfficer"
            ] = EducationYearOfficer.serialize()

        result["faculty"][0]["cathedras"] = self.getDepartments(lname, id)
        result["faculty"][0]["staff"] = self.getStaff(lname, id)

        fc.set_faculty_to_redis(result)

        return result

    def getFacultyList(self, lname):
        f = Faculty(lname)

        facultyList = f.get_faculty_list_from_redis()

        if facultyList is not None:
            return facultyList

        conn = current_app.ms.db(lname).connect()

        query = conn.execute(
            "SELECT * FROM faculty WHERE STATE='Действующий' ORDER BY faculty"
        )

        facultyList = []

        for row in query:
            faculty = Faculty("")
            faculty.idfaculty = row.idfaculty
            faculty.faculty = row.faculty
            facultyList.append(faculty)

        f.set_faculty_list_to_redis(facultyList)

        return facultyList

    @staticmethod
    def get_faculty_list_user(lname, mid):
        """returns a list of the user's faculties"""
        sql = """
            SELECT faculty, pe.mid FROM faculty fa
                JOIN groupname gn ON gn.idcathedra = fa.idfaculty
                JOIN groupuser gu ON gu.gid = gn.gid
                JOIN people pe ON pe.mid = gu.mid
            WHERE pe.mid = :mid
            """
        conn = current_app.ms.db(lname).connect()
        stmt = text(sql)
        stmt = stmt.bindparams(mid=mid)
        query = conn.execute(stmt)

        faculty_user = []

        for row in query:
            faculty = Faculty("")
            faculty.faculty = row.faculty
            faculty_user.append(faculty)

        return faculty_user

    def getFacultyListForTeacher(self, lname, mid):
        f = Faculty(lname)

        facultyList = f.get_faculty_list_from_redis()

        if facultyList is not None:
            return facultyList

        sql = """select distinct faculty.idfaculty,faculty.faculty 
			from faculty
            inner join cathedras c on c.faculty = faculty.idfaculty
            inner join cathedra_personnel cp on cp.cid = c.idcathedra
            where  cp.mid = :mid
            order by faculty.faculty"""
        stmt = text(sql)
        stmt = stmt.bindparams(mid=mid)
        if g.user.user_is_rasdel_admin(
            g.user.mid, "Журнал учета учебных занятий. Администратор"
        ):
            sql = """
            select distinct faculty.idfaculty,faculty.faculty 
			from faculty
            inner join cathedras c on c.faculty = faculty.idfaculty
            inner join cathedra_personnel cp on cp.cid = c.idcathedra
            order by faculty.faculty
            """
            stmt = text(sql)

        conn = current_app.ms.db(lname).connect()
        query = conn.execute(stmt)

        facultyList = []

        for row in query:
            faculty = Faculty("")
            faculty.idfaculty = row.idfaculty
            faculty.faculty = row.faculty
            facultyList.append(faculty)

        f.set_faculty_list_to_redis(facultyList)

        return facultyList

    def get_faculty_list_for_teacher_journal_of_accounting_of_training_sessions(self, lname):
        """
        Режим: "Журнал учета учебных занятий"
        """
        sql = """
        SELECT idfaculty, faculty 
        FROM faculty
        WHERE state='Действующий' 
            or COALESCE(
              (SELECT CONVERT_FROM(value, 'UTF8') FROM options WHERE name = 'showhidendivisions') = 'true', false
            ) = true
        ORDER BY (For_Sorted_With_Numder4 (faculty))
        """
        stmt = text(sql)

        conn = current_app.ms.db(lname).connect()
        query = conn.execute(stmt)

        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    def getDepartments(self, lname, id):
        conn = current_app.ms.db(lname).connect()
        sql = "select * from xp_get_departments(:id, 0) d order by d.UnitName"
        stmt = text(sql)
        stmt = stmt.bindparams(id=id)
        query = conn.execute(stmt)
        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    def getStaff(seldf, lname, id):
        sql = """
        DROP TABLE IF EXISTS tmpCP; DROP TABLE IF EXISTS tmpPF;
CREATE TEMPORARY TABLE tmpCP (pid INTEGER, mid INTEGER, id INTEGER, mode INTEGER, adate DATE);
CREATE TEMPORARY TABLE tmpPF (pid INTEGER, mid INTEGER);

SELECT cp.id, cp.date, cp.date,
 coalesce(pf.workstatus,'Совмещает') workstatus, 
 pos.name as position, cp.rate, cp.pos_overlay, 
 if(pf.mid is not null,pf."MilitaryRank", cp.military_rank) as rank, 
 if(pf.mid is not null,mrpf.short_title, mr.short_title) as rank_name, if(pf.mid is not null, pf.vus, cp.vus) as vus, 
 if(pf.mid is not null, vpf.code, v.code) as vus_name, 
 cp.tariff, if(pf.mid is not null,pf."OfficerTrainingLevelID",cp.c_officer_level) as off_lvl, 
 if(pf.mid is not null,colpf.shortname,col.shortname) as off_lvl_name, 
 xp_format_fio(p.lastname,p.firstname,p.patronymic,0)  as personnel_name, 
case 
 when cp.deputy_mid>0 and now() between  cp.deputy_date_start and coalesce(cp.deputy_date_end,now()) then 
 concat_ws(' ',xp_f_get_mid_fio(cp.deputy_mid,1), cp.deputy_type, 'с',to_char(cp.deputy_date_start,'dd.mm.yyyy')) 
else null 
end as deputy 
FROM (
select 0 as mode, dp.dpid as id, dp.sdid as structid,dp.positionid,
   coalesce(t.adate,dp.date) as date, case when t.id is not null then t.mid else dp.mid end as mid, 
   dp.pos_overlay,dp.rate, dp.c_officer_level, 
  dp.military_rank, dp.tariff,dp.vus,dp.code,dp.code2,dp.code3, dp.deputy_mid, dp.deputy_date_start,dp.deputy_date_end, dp.deputy_type  
  from division_personnel dp LEFT JOIN tmpCP t ON t.id=dp.dpid AND t.mode=0 
  union 
  select 1 as mode, dp.fpid as id, dp.facultyid as structid,dp.positionid,
   coalesce(t.adate,dp.date) as date,case when t.id is not null then t.mid else dp.mid end as mid, 
   dp.pos_overlay,dp.rate, dp.c_officer_level, 
  dp.military_rank, dp.tariff,dp.vus,dp.code,dp.code2,dp.code3, dp.deputy_mid, dp.deputy_date_start,dp.deputy_date_end, dp.deputy_type 
  from faculty_personnel dp LEFT JOIN tmpCP t ON t.id=dp.fpid AND t.mode=1 
  union 
  select 2 as mode, dp.cpid as id, dp.cid as structid,dp.positionid, 
   coalesce(t.adate,dp.date) as date, case when t.id is not null then t.mid else dp.mid end as mid, 
  dp.pos_overlay,dp.rate, dp.c_officer_level,
  dp.military_rank,dp.tariff,dp.vus,dp.code,dp.code2,dp.code3, dp.deputy_mid, dp.deputy_date_start,dp.deputy_date_end, dp.deputy_type 
  from cathedra_personnel dp LEFT JOIN tmpCP t ON t.id=dp.cpid AND t.mode=2
) cp 
 LEFT JOIN military_rank mr on cp.military_rank = mr.id_mil_rank 
 LEFT JOIN c_officer_level col on col.id = cp.c_officer_level 
 LEFT JOIN (select v.id,v.code from vus v left join vus_custom vc on vc.code=v.code) as v on v.id = cp.vus 
 JOIN positions pos ON pos.pid=cp.positionid 
 LEFT JOIN people p ON p.mid=cp.mid 
  LEFT JOIN xp_personal_file pf ON pf.mid = p.mid 
  LEFT JOIN military_rank mrpf on pf."MilitaryRank" = mrpf.id_mil_rank 
  LEFT JOIN c_officer_level colpf on colpf.id = pf."OfficerTrainingLevelID" 
  LEFT JOIN (select v.id,v.code from vus v left join vus_custom vc on vc.code=v.code) as vpf on vpf.id = pf.vus 
WHERE cp.mode=1 AND cp.structid=:id
        """
        conn = current_app.ms.db(lname).connect()
        stmt = text(sql)
        stmt = stmt.bindparams(id=id)
        query = conn.execute(stmt)
        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    def getFacultyForSRW(self, lname):
        conn = current_app.ms.db(lname).connect()

        sql = """
        select * 
            from (select idfaculty, "ShortName", 1 as flag from faculty union select 0, 'Общеакадемический', 0) t 
        ORDER BY flag, for_sorted_with_numder4("ShortName")
                """

        stmt = text(sql)
        query = conn.execute(stmt)

        result = {"faculty": [dict(zip(tuple(query.keys()), i)) for i in query.cursor]}

        if len(result["faculty"]) == 0:
            return None

        return result

    def get_journal_faculty(self):
        conn = current_app.ms.db(self.lname).connect()

        stmt = text(
            """
            select  idfaculty, coalesce(nullif("ShortName",''),faculty) as faculty 
            from faculty 
            where state='Действующий' 
            order by (substring("ShortName", '^[0-9]+'))::bigint , substring("ShortName", '[^0-9_].*$') ;
            """
        )
        query = conn.execute(stmt)
        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    def get_faculty_and_cathedras(self, all: int):
        conn = current_app.ms.db(self.lname).connect()

        if all == 1:
            stmt = text(
                """
                SELECT 
                    idfaculty,
                    idcathedra,
                    name
                FROM (
                    SELECT 
                        idfaculty,
                        NULL AS idcathedra,
                        faculty AS name
                    FROM faculty
                    WHERE state = 'Действующий'
                
                    UNION ALL
                
                    SELECT 
                        NULL AS idfaculty,
                        idcathedra,
                        cathedra AS name
                    FROM cathedras
                    WHERE state = 'Действующая'
                ) AS combined
                ORDER BY name;
                """
            )
        else:
            stmt = text(
                """
                SELECT 
                    f.idfaculty,
                    f.faculty,
                    c.idcathedra,
                    c.cathedra
                FROM faculty f
                FULL JOIN cathedras c 
                       ON f.idfaculty = c.faculty  -- если связь по названию факультета
                WHERE 
                    (f.state = 'Действующий' OR f.state IS NULL)
                    AND (c.state = 'Действующая' OR c.state IS NULL)
                ORDER BY 
                    f.faculty NULLS LAST,
                    For_Sorted_With_Numder4(c.cathedra) NULLS LAST;
                """
            )
        query = conn.execute(stmt)
        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    def get_faculty_and_cathedras_all(self):
        conn = current_app.ms.db(self.lname).connect()

        stmt = text(
            """
            SELECT 
                f.idfaculty,
                f.faculty,
                c.idcathedra,
                c.cathedra
            FROM faculty f
            FULL JOIN cathedras c 
                   ON f.idfaculty = c.faculty  -- если связь по названию факультета
            WHERE 
                (f.state = 'Действующий' OR f.state IS NULL)
                AND (c.state = 'Действующая' OR c.state IS NULL)
            ORDER BY 
                f.faculty NULLS LAST,
                For_Sorted_With_Numder4(c.cathedra) NULLS LAST;
            """
        )
        query = conn.execute(stmt)
        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]
