from flask import jsonify, current_app, send_from_directory, request, g
from flask import Blueprint
from flask_cors import CORS
from sqlalchemy import text
from sqlalchemy.exc import SQLAlchemyError

from LMSAPI import db
from LMSAPI.api.Views.TokenAPI import auth

methodical_plan_api = Blueprint('methodical_plan_api', __name__)

CORS(methodical_plan_api)


class MethodicalPlan:
    """ План методической деятельности вуза на учебный год """

    def __init__(self, cname):
        self.lname = cname

    def get_years_plan(self):
        sql = """
        SELECT id as mpid, s.name AS "period"
        FROM method_plan p
            LEFT JOIN school_year s ON s.xp_key=p.school_year
        WHERE p.status = 3
        ORDER BY s.begdate
        """

        conn = current_app.ms.db(self.lname).connect()
        stmt = text(sql)
        query = conn.execute(stmt)
        years = [dict(a) for a in query]

        return years

    def get_plan_methodical_activity_university_school_year(self):
        """
            Список записей для
            "План методической деятельности вуза на учебный год"
        """
        sql = """
                SELECT id, s.name AS "Уч. год",
                    CASE COALESCE(p.status, 0)
                        WHEN 1 THEN 'проект'
                        WHEN 2 THEN 'готов'
                        WHEN 3 THEN 'утвержден'
                        WHEN 4 THEN 'архив'
                    END AS "Статус",
                    concat_ws(' ', pos.name, xp_f_get_mid_fio(p.resp_mid, 0)) as "Ответственный"
                FROM method_plan p
                    LEFT JOIN school_year s ON s.xp_key=p.school_year
                    LEFT JOIN positions pos ON pos.pid=p.resp_pid
                ORDER BY s.begdate
                """
        conn = current_app.ms.db(self.lname).connect()
        stmt = text(sql)
        query = conn.execute(stmt)

        activity_list = [dict(a) for a in query]

        return activity_list

    @staticmethod
    def get_status():
        status = {
            "1": "Проект",
            "2": "Готов",
            "3": "Утвержден",
            "4": "Архив"
        }
        return status

    def get_position(self):
        """ Список ФИО для раздела "Новая запись" при добавлении новой деятельности """
        sql = """
                SELECT distinct pf."Position", ps.name
                FROM people p
                JOIN xp_personal_file pf ON pf.mid=p.mid AND pf.workstatus='Работает'
                JOIN positions ps ON ps.pid = pf."Position"
                WHERE p.mid IN (SELECT mid FROM teachers) AND pf."Position" <> 0
                ORDER BY 2
                """
        conn = current_app.ms.db(self.lname).connect()
        stmt = text(sql)
        query = conn.execute(stmt)

        position_list = [dict(a) for a in query]

        return position_list

    def get_fio(self, pid):
        """ Список ФИО для раздела "Новая запись" при добавлении новой деятельности """
        sql = """
                SELECT p.mid, xp_f_get_mid_fio(p.mid, 1)::varchar(254) AS fio 
                FROM people p
                JOIN xp_personal_file pf ON pf.mid=p.mid AND pf.workstatus='Работает'
                WHERE p.mid IN (SELECT mid FROM teachers)
                """

        if pid:
            sql += """ AND pf."Position" = :pid """

        sql += " ORDER BY 2"
        conn = current_app.ms.db(self.lname).connect()
        stmt = text(sql)
        if pid:
            stmt = stmt.bindparams(pid=pid)
        query = conn.execute(stmt)

        fio_list = [dict(a) for a in query]

        return fio_list

    def post(self, faculty):
        """
            Создание записей для
            "План методической деятельности вуза на учебный год"
        """
        conn = current_app.ms.db(self.lname).connect()

        sql = """
            INSERT INTO method_plan(school_year, resp_pid, resp_mid, status, description, shorts)
            VALUES (:school_year, :resp_pid, :resp_mid, :status, :description, :shorts)
            RETURNING id;
        """

        stmt = text(sql).bindparams(
            school_year=faculty["school_year"],
            resp_pid=faculty["resp_pid"],
            resp_mid=faculty["resp_mid"],
            status=faculty["status"],
            description=faculty["description"],
            shorts=faculty["shorts"]
        )

        result = conn.execute(stmt)
        new_id = result.scalar()  # Получение нового ID из запроса

        return new_id

    def get_main_goals_and_objectives(self, mpid):
        """ Основные цели и задачи методической деятельности(вторая вкладка при редактировании) """
        sql = """
                SELECT 
                    goal1_name, goal1_desc, 
                    goal2_name, goal2_desc, 
                    goal3_name, goal3_desc, 
                    goal4_name, goal4_desc, 
                    goal5_name, goal5_desc, 
                    goal6_name, goal6_desc
                FROM method_plan
                WHERE id = :mpid
                """
        conn = current_app.ms.db(self.lname).connect()
        stmt = text(sql)
        stmt = stmt.bindparams(mpid=mpid)
        query = conn.execute(stmt)

        goal_list = [dict(a) for a in query]

        return goal_list

    def get_decree(self, mpid):
        """ Организационно-методические указания(третья вкладка при редактировании) """
        sql = """
                SELECT ukaz
                FROM method_plan
                WHERE id = :mpid
                """
        conn = current_app.ms.db(self.lname).connect()
        stmt = text(sql)
        stmt = stmt.bindparams(mpid=mpid)
        query = conn.execute(stmt)

        decree = [dict(a) for a in query][0]

        return decree

    def get_approved_plan(self, mpid):
        conn = current_app.ms.db(self.lname).connect()

        sql = """
        SELECT EXISTS(
            SELECT 1 FROM method_plan
            WHERE id = :mpid
            )
        """

        stmt = text(sql)
        stmt = stmt.bindparams(mpid=mpid)
        query = conn.execute(stmt).scalar()

        return query

    def get_part(self):
        conn = current_app.ms.db(self.lname).connect()
        sql = """
            SELECT * FROM method_plan_event_sections
            ORDER BY title
        """
        stmt = text(sql)
        query = conn.execute(stmt)

        return [dict(q) for q in query]

    def get_content_part(self, mpid, event_type):
        conn = current_app.ms.db(self.lname).connect()

        sql_group_and_section = """
                SELECT DISTINCT 
                    title, caption_code, code_section, code_group, struct_id
                FROM ( 
                SELECT 
                    CONCAT(code_section, '.', 0) AS caption_code, 
                    1 AS is_section, 
                    event_section, 
                    code_section, 
                    0 AS event_group, 
                    0 AS code_group, 
                    mpes.title AS title, 
                    mpet.struct_id 
                FROM method_plan_events mpe 
                JOIN method_plan_event_sections mpes ON mpe.event_section = mpes.id 
                LEFT JOIN method_plan_events_tocalendar mpet ON mpe.mpeid = mpet.mpeid 
                WHERE mpe.mpid = :mpid
                    AND mpe.event_type = :event_type 
                    -- Все подразделения 
                UNION ALL 
                SELECT 
                    CONCAT(code_section, '.', code_group) AS caption_code, 
                    0 AS is_section, 
                    event_section, 
                    code_section, 
                    event_group, 
                    code_group, 
                    mpeg.title AS title, 
                    mpet.struct_id 
                FROM method_plan_events mpe 
                JOIN method_plan_event_groups mpeg ON mpe.event_group = mpeg.id 
                LEFT JOIN method_plan_events_tocalendar mpet ON mpe.mpeid = mpet.mpeid 
                WHERE mpe.mpid = :mpid
                    AND mpe.event_type = :event_type 
                    -- Все подразделения 
                ) q 
                ORDER BY caption_code;
                """

        stmt_group_and_section = text(sql_group_and_section)
        stmt_group_and_section = stmt_group_and_section.bindparams(mpid=mpid, event_type=event_type)
        query = conn.execute(stmt_group_and_section)

        query = [dict(row) for row in query]
        result = {}

        for row in query:
            key = str(row['code_section'])
            if key in result:
                result[key].append(row['title'])
            else:
                result[key] = [row['title']]

        sql_events = """
            SELECT 
              mpe.mpeid, mpe.mpid, 
              mpe.event_level, mpe.event_section, mpe.event_group, 
              mpe.code, 
              mpe.code_section, mpe.code_group, mpe.code_vuz, 
              mpe.code_faculty, mpe.code_cathedra, mpe.code_division, 
              mpe.event_section, mpe.event_group, mpe.event_number, 
              mpe.title, 
              mpe.from_day, mpe.from_month, mpe.to_year, 
              mpe.to_day, mpe.to_month, mpe.to_year, 
              mpe.from_date, mpe.to_date, 
              mpe.is_done, 
            -- 
              ( 
                SELECT -- cnt - кол-во привязанных к месяцу, если их нет, то "Весь период" не добавляем 
                  IF(SUM(cnt) > 0, CONCAT('--- Весь период ', E'\n', STRING_AGG(title, E'\n')), STRING_AGG(title, E'\n')) 
                FROM ( 
                  SELECT 0 AS cnt, STRING_AGG(title, E'\n') AS title -- Весь период 
                  FROM method_plan_events_responsibles mper1 
                  WHERE mper1.mpe_id = mpe.mpeid 
                    AND resp_year = 0 
                    AND resp_month = 0 
                    AND event_type = :event_type
                  UNION ALL 
                  SELECT COUNT(title) AS cnt, STRING_AGG(title, E'\n') -- объединяем строки по месяцам 
                  FROM ( 
                    SELECT title 
                    FROM ( 
                      SELECT -- подбираем заголовки 
                        CONCAT(resp_year, '.', resp_month) AS sort, 
                        CONCAT('--- ', xp_fget_rus_month_flex(to_date(CONCAT(resp_year, resp_month, '01'), 'YYYYMMDD'), 1), ', ', resp_year) AS title 
                      FROM method_plan_events_responsibles mper2 
                      WHERE mper2.mpe_id = mpe.mpeid 
                        AND resp_year > 0 
                        AND resp_month > 0 
                        AND event_type = :event_type
                      GROUP BY sort, resp_year, resp_month 
                      UNION ALL 
                      SELECT -- подбираем строки 
                        CONCAT(resp_year, '.', resp_month, title) AS sort, 
                        title 
                      FROM method_plan_events_responsibles mper3 
                      WHERE mper3.mpe_id = mpe.mpeid 
                        AND resp_year > 0 
                        AND resp_month > 0 
                        AND event_type = :event_type
                    ) q 
                  ORDER BY sort 
                  ) q2 
                ) q3 
              ) AS responsibles, 
            -- 
              ( 
                SELECT 
                  IF(SUM(cnt) > 0, CONCAT('--- Весь период ', E'\n', STRING_AGG(title, E'\n')), STRING_AGG(title, E'\n')) 
                FROM ( 
                  SELECT 0 AS cnt, STRING_AGG(title, E'\n') AS title 
                  FROM method_plan_events_members mpem1 
                  WHERE mpem1.mpe_id = mpe.mpeid 
                    AND mem_year = 0 
                    AND mem_month = 0 
                    AND event_type = :event_type
                  UNION ALL 
                  SELECT COUNT(title) AS cnt, STRING_AGG(title, E'\n') 
                  FROM ( 
                    SELECT title 
                    FROM ( 
                      SELECT 
                        CONCAT(mem_year, '.', mem_month) AS sort, 
                        CONCAT('--- ', xp_fget_rus_month_flex(to_date(CONCAT(mem_year, mem_month, '01'), 'YYYYMMDD'), 1), ', ', mem_year) AS title 
                      FROM method_plan_events_members mpem2 
                      WHERE mpem2.mpe_id = mpe.mpeid 
                        AND mem_year > 0 
                        AND mem_month > 0 
                        AND event_type = :event_type
                      GROUP BY sort, mem_year, mem_month 
                      UNION ALL 
                      SELECT 
                        CONCAT(mem_year, '.', mem_month, title) AS sort, 
                        title 
                      FROM method_plan_events_members mpem3 
                      WHERE mpem3.mpe_id = mpe.mpeid
                        AND mem_year > 0 
                        AND mem_month > 0 
                        AND event_type = :event_type
                    ) q 
                  ORDER BY sort 
                  ) q2 
                ) q3 
              ) AS members 
            FROM method_plan_events mpe 
            WHERE mpid = :mpid
              AND event_type = :event_type
            ORDER BY code_group, code_section, event_number
        """

        stmt_events = text(sql_events)
        stmt_events = stmt_events.bindparams(mpid=mpid, event_type=event_type)
        query = conn.execute(stmt_events)

        query = [dict(row) for row in query]

        for row in query:
            key = str(row['code_section'])
            if key in result:
                result[key].append(row)
            else:
                result[key] = [row]

        return result

    def get_changes(self, mpid):
        conn = current_app.ms.db(self.lname).connect()

        sql = "SELECT * FROM method_plan_event_changes mpec WHERE mp_id = :mpid ORDER BY change_number;"

        stmt = text(sql)
        stmt = stmt.bindparams(mpid=mpid)
        query = conn.execute(stmt)

        return [dict(row) for row in query]