# coding=UTF-8
# LMSAPI/API/Journal.py
from flask import current_app, g
import pickle

from sqlalchemy import text
from werkzeug import exceptions

from LMSAPI.api.Models.QuestionAnswers import QuestionAnswers


class QuestionSet:
    def get_questionset_all(self, lname, cid, private, oid):
        conn = current_app.ms.db(lname).connect()
        where = ""
        if cid:
            where += " and cid = :cid "
        if private is not None:
            where += " and private = :private "
        if oid:
            where += " and oid = :oid "

        sql = """
            select * from questionset
            where 1=1
            {where}
            """.format(
            where=where
        )

        stmt = text(sql)
        if cid:
            stmt = stmt.bindparams(cid=cid)
        if private is not None:
            stmt = stmt.bindparams(private=private)
        if oid:
            stmt = stmt.bindparams(oid=oid)

        query = conn.execute(stmt)
        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    def get_sheid_and_week_id_by_questionset(self, lname, questionset):
        conn = current_app.ms.db(lname).connect()

        sql = """
            SELECT t.sheid, t.week_id
            FROM (
                SELECT DISTINCT
                    hw.sheid AS sheid,
                    hw.week_id AS week_id
                FROM nnz_homework hw
                LEFT JOIN nnz_homework_student hws ON hws.hwid = hw.hwid AND hws.student_mid = :mid
                WHERE hw.questionset = :questionset 
                    OR hws.questionset = :questionset 
                UNION
                SELECT DISTINCT
                    hwf.sheid AS sheid,
                    hwf.week_id AS week_id
                FROM nnz_homework_fact hwf
                LEFT JOIN nnz_homework_student hws1 ON hws1.hwid = hwf.hwid AND hws1.student_mid = :mid
                WHERE hwf.questionset = :questionset 
                    OR hws1.questionset = :questionset 
            ) as t
            JOIN vw_schedule s ON s.sheid = t.sheid AND s.week_id = t.week_id
            JOIN groupuser g ON g.gid = s.gid AND g.mid = :mid
            """
        stmt = text(sql)
        stmt = stmt.bindparams(mid=g.user.mid, questionset=questionset)

        query = conn.execute(stmt)
        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    def get_question_and_answer_by_test(self, lname, id, teacher):
        conn = current_app.ms.db(lname).connect()

        correct = ", 'correct', ea.correct"
        if teacher == False:
            correct = ""

        sql = """
            WITH questions_with_children AS (
                SELECT 
                    eq.id as question_id,
                    eq.question_type as question_type,
                    eq.cid as cid,
                    eq.oid as oid,
                    eq.tlid as tlid,
                    eq.question_text as question_text,
                    eq.q_num as q_num,
                    eq.weight as weight,
                    eq.id_parent as id_parent,
                    eq.status as status,
                    eq.eyid as eyid,
                    eq.answer_id as answer_id,
                    eq.worktime as worktime,
                    eq.score as score,
                    qc.sortnum
                FROM 
                    questionset q
                JOIN questionset_content qc ON qc.id_set = q.id
                JOIN enroll_questions eq ON eq.id = qc.id_question
                WHERE 
                    q.id = :id
                    AND eq.question_type <> 6  -- Берём все вопросы, кроме тех, где question_type = 6
                    AND eq.question_type <> 5  -- Не берём вопросы с question_type = 5
            )
            SELECT 
                qwc.question_id,
                qwc.question_type,
                qwc.cid,
                qwc.oid,
                qwc.tlid,
                qwc.question_text,
                qwc.q_num,
                qwc.weight,
                qwc.id_parent,
                qwc.status,
                qwc.eyid,
                qwc.answer_id,
                qwc.worktime,
                qwc.score as score,
                qwc.sortnum,
                json_agg(
                    json_build_object(
                        'id', ea.id,
                        'answer_text', ea.answer_text,
                        'answer_group', ea.answer_group,  
                        'answer_length', ea.answer_length,
                        'answer_pos', ea.answer_pos,
                        'answer_html', ea.answer_html,
                        'sel_start', ea.sel_start,
                        'sel_len', ea.sel_len,
                        'value_num', ea.value_num
                        {correct}
                    )
                ) AS answers
            FROM 
                questions_with_children qwc
            LEFT JOIN 
                enroll_answers ea ON ea.question = qwc.question_id
            GROUP BY 
                qwc.question_id, qwc.question_type, qwc.cid, qwc.oid, qwc.tlid, qwc.question_text, 
                qwc.q_num, qwc.weight, qwc.id_parent, qwc.status, qwc.eyid, qwc.answer_id, qwc.worktime, qwc.score, qwc.sortnum
            ORDER BY 
                qwc.sortnum;
            """.format(
            correct=correct
        )

        stmt = text(sql)
        stmt = stmt.bindparams(id=id)

        query = conn.execute(stmt)
        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    def get_questionset_by_id(self, lname, questionset):
        conn = current_app.ms.db(lname).connect()

        # Основной запрос по questionset
        sql = """
            SELECT *
            FROM questionset
            WHERE id = :questionset
        """
        stmt = text(sql).bindparams(questionset=questionset)
        query = conn.execute(stmt)
        result = query.fetchone()

        if result is None:
            return None

        result_dict = dict(zip(query.keys(), result))

        # Только если все вопросы имеют worktime > 0, считаем сумму
        sum_sql = """
            WITH counts AS (
                SELECT
                    COUNT(*) FILTER (WHERE eq.worktime > 0) AS with_time,
                    COUNT(*) AS total
                FROM questionset_content qc
                JOIN enroll_questions eq ON eq.id = qc.id_question
                WHERE qc.id_set = :questionset
                  AND eq.question_type NOT IN (5, 6)
            )
            SELECT CASE
                WHEN with_time = total THEN
                    (SELECT SUM(eq.worktime)
                     FROM questionset_content qc
                     JOIN enroll_questions eq ON eq.id = qc.id_question
                     WHERE qc.id_set = :questionset
                       AND eq.question_type NOT IN (5, 6))
                ELSE NULL
            END AS total_worktime
            FROM counts
        """
        sum_stmt = text(sum_sql).bindparams(questionset=questionset)
        sum_query = conn.execute(sum_stmt)
        total_worktime = sum_query.scalar()

        result_dict["total_worktime"] = int(total_worktime) if total_worktime else None

        return result_dict

    def create_questionset(self, lname, data):
        conn = current_app.ms.db(lname).connect()

        sql = """ 
            insert into questionset(name, cid, private, description, oid) 
            values (:name, :cid, :private, :description, :oid) 
            returning id 
            """
        stmt = text(sql)
        stmt = stmt.bindparams(
            name=data.get("name"),
            cid=data.get("cid"),
            private=data.get("private"),
            description=data.get("description"),
            oid=data.get("oid"),
        )
        query = conn.execute(stmt)

        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    def update_questionset(self, lname, id, data):
        conn = current_app.ms.db(lname).connect()

        # Список для хранения частей SET выражения и параметров
        set_clauses = []
        params = {"id": id}

        # Добавляем поля в SET выражение, если они есть в данных
        if data.get("cid") is not None:
            set_clauses.append("cid = :cid")
            params["cid"] = data.get("cid")
        if data.get("private") is not None:
            set_clauses.append("private = :private")
            params["private"] = data.get("private")
        if data.get("oid") is not None:
            set_clauses.append("oid = :oid")
            params["oid"] = data.get("oid")
        if data.get("name") is not None:
            set_clauses.append("name = :name")
            params["name"] = data.get("name")
        if data.get("description") is not None:
            set_clauses.append("description = :description")
            params["description"] = data.get("description")

        # Проверяем, есть ли поля для обновления
        if not set_clauses:
            raise ValueError("No fields to update")

        # Формируем SQL запрос
        set_clause = ", ".join(set_clauses)
        sql = """
            UPDATE questionset
            SET {set_clause}
            WHERE id = :id
            RETURNING id
        """.format(
            set_clause=set_clause
        )

        stmt = text(sql).bindparams(**params)
        query = conn.execute(stmt)

        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    def delete_questionset(self, lname, id):
        conn = current_app.ms.db(lname).connect()
        sql = """
            DELETE FROM questionset
            WHERE id = :id
            RETURNING id
        """

        stmt = text(sql).bindparams(id=id)
        query = conn.execute(stmt)

        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    def get_stats_by_questionset(self, lname, id, mid):
        questionset = QuestionSet().get_questionset_by_id(lname, id)
        all_question_answers = QuestionSet().get_question_and_answer_by_test(
            lname, id, True
        )
        count_question = len(all_question_answers)
        text_question = 0
        correct_question_answers = 0
        incorrect_question_answers = 0
        user_score = 0
        add_grade_blocked = 0
        user_trying = QuestionAnswers().get_max_traying(lname, mid, id)

        # Проверяем, заполнены ли у всех вопросов баллы за выполнение, если хотябы один вопрос не заполнен, то блокируем выставление оценки по результатам выполнения теста
        for q in all_question_answers:
            if q.get("score") is None or q.get("score") == 0:
                add_grade_blocked = 1
                break
        # Проверяем, заполнены ли у теста баллы для каждой оценки, если не заполнена хотябы для одной оценки, то блокируем выставление оценки по результатам выполнения теста
        for field in ["ball3", "ball4", "ball5"]:
            if questionset.get(field) is None or questionset.get(field) == 0:
                add_grade_blocked = 1
                break

        for question in all_question_answers:
            user_question_answers = QuestionAnswers().get_question_answers_all(
                lname, hwid=None, mid=mid, questionset=id, question=question["question_id"]
            )
            # Фильтруем только те ответы, у которых trying совпадает с user_trying
            user_question_answers = [
                ans for ans in user_question_answers if ans.get("trying") == user_trying
            ]

            if question["question_type"] == 1:
                # Проверяем, что есть ровно один пользовательский ответ
                if len(user_question_answers) == 1:
                    user_answer_id = user_question_answers[0]["answer_id"]

                    # Находим ответ с таким же answer_id в вопросе и проверяем его корректность
                    correct_answer_found = False
                    for answer in question["answers"]:
                        if answer["id"] == user_answer_id and answer.get("correct") is True:
                            correct_answer_found = True
                            break

                    # Увеличиваем счетчик правильных ответов, если найден правильный ответ
                    if correct_answer_found:
                        correct_question_answers += 1
                        user_score += question.get("score") or 0
                    else:
                        incorrect_question_answers += 1
                elif len(user_question_answers) < 1:
                    incorrect_question_answers += 1

            elif question["question_type"] == 2:
                # Для типа вопроса 1, пользователю разрешено выбрать несколько правильных ответов

                # Находим все правильные ответы для этого вопроса
                correct_answers = {
                    answer["id"]
                    for answer in question["answers"]
                    if answer.get("correct") is True
                }
                user_answers = {
                    user_answer["answer_id"] for user_answer in user_question_answers
                }

                # Проверка: все ли пользовательские ответы являются правильными и их количество соответствует
                if user_answers == correct_answers:
                    correct_question_answers += 1
                    user_score += question.get("score") or 0
                else:
                    incorrect_question_answers += 1

            elif question["question_type"] == 3:
                # Текстовый ответ, не возможно узнать правильно ли ответил пользователь
                text_question += 1
                add_grade_blocked = 1

            elif question["question_type"] == 4:
                # Вопрос типа 3: проверка порядка ответов

                is_correct = True  # Флаг для отслеживания правильности ответа на вопрос

                for user_answer in user_question_answers:
                    # Получаем id ответа и место, которое пользователь выбрал для этого ответа
                    user_answer_id = user_answer["answer_id"]
                    user_answer_group = user_answer["answer_num"]

                    # Находим правильный ответ с тем же id
                    correct_answer = next(
                        (
                            answer
                            for answer in question["answers"]
                            if answer["id"] == user_answer_id
                        ),
                        None,
                    )

                    # Проверяем соответствие
                    if (
                            not correct_answer
                            or correct_answer["answer_group"] != user_answer_group
                    ):
                        is_correct = False
                        break

                if is_correct:
                    correct_question_answers += 1
                    user_score += question.get("score") or 0
                else:
                    incorrect_question_answers += 1

            elif question["question_type"] == 5:
                if len(question["answers"]) == 1:
                    text_question += 1
                else:
                    user_answer_id = user_question_answers[0]["answer_id"]

                    # Находим ответ с таким же answer_id в вопросе и проверяем его корректность
                    correct_answer_found = False
                    for answer in question["answers"]:
                        if answer["id"] == user_answer_id and answer.get("correct") is True:
                            correct_answer_found = True
                            break

                    # Увеличиваем счетчик правильных ответов, если найден правильный ответ
                    if correct_answer_found:
                        correct_question_answers += 1
                        user_score += question.get("score") or 0
                    else:
                        incorrect_question_answers += 1
            else:
                raise exceptions.BadRequest("Нет такого типа вопроса")

        if count_question - text_question > 0:
            user_correct_percentage = (
                                              correct_question_answers / (count_question - text_question)
                                      ) * 100
        else:
            user_correct_percentage = 0

        return count_question, text_question, correct_question_answers, incorrect_question_answers, user_score, add_grade_blocked, user_trying, user_correct_percentage


class QuestionSetContent:
    def get_questionset_content_all(self, lname, id_question, id_set):
        conn = current_app.ms.db(lname).connect()
        where = ""
        if id_question:
            where += " and id_question = :id_question "
        if id_set:
            where += " and id_set = :id_set "

        sql = """
            select * from questionset_content
            where 1=1
            {where}
            """.format(
            where=where
        )

        stmt = text(sql)
        if id_question:
            stmt = stmt.bindparams(id_question=id_question)
        if id_set:
            stmt = stmt.bindparams(id_set=id_set)

        query = conn.execute(stmt)
        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    def create_questionset_content(self, lname, data):
        conn = current_app.ms.db(lname).connect()

        sql = """ 
            insert into questionset_content(id_set, id_question, sortnum) 
            values (:id_set, :id_question, :sortnum) 
            returning id_set 
            """
        stmt = text(sql)
        stmt = stmt.bindparams(
            id_set=data.get("id_set"),
            id_question=data.get("id_question"),
            sortnum=data.get("sortnum"),
        )
        query = conn.execute(stmt)

        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

    def delete_questionset_content(self, lname, id_question, id_set):
        conn = current_app.ms.db(lname).connect()
        sql = """
            DELETE FROM questionset_content
            WHERE id_set = :id_set and id_question = :id_question
            RETURNING id_set
        """

        stmt = text(sql).bindparams(id_question=id_question, id_set=id_set)
        query = conn.execute(stmt)

        return [dict(zip(tuple(query.keys()), i)) for i in query.cursor]
