# LMSAPI/API/Room.py
import os
from decimal import Decimal

from sqlalchemy import create_engine
from sqlalchemy.sql import text

from LMSAPI.api.Models.File import File
from LMSAPI.api.Models.User import User
from flask import current_app, g

from LMSAPI.api.utils.request_utils import RequestUtils


class Rating:
    def __init__(self, cname):
        self.lname = cname

    def has_true_stipend(self):
        conn = current_app.ms.db(self.lname).connect()
        sqlt = "SELECT EXISTS (SELECT 1 FROM rating_schema WHERE stipend = True)"
        stmt = text(sqlt)
        result = conn.execute(stmt).scalar()  # Returns True if exists, False otherwise
        conn.close()
        return result

    def getRatingSchema(self, school_year):
        conn = current_app.ms.db(self.lname).connect()
        sqlt = "SELECT * from rating_schema  where :school_year = ANY(school_year) order by title "
        stmt = text(sqlt)
        stmt = stmt.bindparams(school_year=school_year)
        query = conn.execute(stmt)

        result = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

        return result

    def getRating(self, schema, grpupid, date, type):
        conn = current_app.ms.db(self.lname).connect()
        sqlt = """
            SELECT 
                p.mid, 
                p.lastname, 
                p.firstname, 
                p.patronymic, 
                pi.*, 
                v.value, 
                v.points, 
                v.id as personnel_rating_id,
                CASE 
                    WHEN v.title IS NULL 
                      AND v.document_number IS NULL 
                      AND v.document_date IS NULL 
                      AND v.document_date_to IS NULL 
                      AND v.achievement_result IS NULL 
                      AND v.info IS NULL 
                      AND v.classification_mark IS NULL 
                    THEN 0 
                    ELSE 1 
                END AS check_info
            FROM people p
            INNER JOIN groupuser gu ON p.mid = gu.mid
            LEFT OUTER JOIN personnel_indicators pi ON pi.schema = :schema
            LEFT OUTER JOIN personnel_rating_values v ON v.mid = p.mid AND v.indicator = pi.id AND v.period_type = :type AND periodend = :date
            where gu.gid = :grpupid
            order by  p.lastname, p.firstname, p.patronymic, pi.id_group, pi.sorting, pi.title
            """

        stmt = text(sqlt)
        stmt = stmt.bindparams(schema=schema)
        stmt = stmt.bindparams(grpupid=grpupid)
        stmt = stmt.bindparams(date=date)
        stmt = stmt.bindparams(type=type)
        query = conn.execute(stmt)

        result = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

        return result

    def update_rating(self, id, data_dict):
        conn = current_app.ms.db(self.lname).connect()

        sqlt = """
            UPDATE personnel_rating_values 
            SET value = :value
            WHERE id = :id
            """

        stmt = text(sqlt)
        stmt = stmt.bindparams(value=data_dict["value"], id=id)

        try:
            query = conn.execute(stmt)
        except Exception as e:
            return e.args
        return id

    def create_rating(self, data_dict):
        conn = current_app.ms.db(self.lname).connect()

        sqlt = """
                INSERT INTO personnel_rating_values 
                    (mid, indicator, period_type, periodend, value, who_change, when_change)
                VALUES
                    (:mid, :indicator, :period_type, :periodend, :value, :who_change, now())
                RETURNING id
            """

        stmt = text(sqlt)
        stmt = stmt.bindparams(
            mid=data_dict["mid"],
            indicator=data_dict["indicator"],
            period_type=data_dict["period_type"],
            periodend=data_dict["periodend"],
            value=data_dict["value"],
            who_change=g.user.mid,
        )

        try:
            query = conn.execute(stmt)
            inserted_id = query.scalar()
        except Exception as e:
            return e.args
        return inserted_id

    def setRating(self, data_dict):
        conn = current_app.ms.db(self.lname).connect()

        sqlt = """
        SELECT * 
        FROM personnel_rating_values 
        WHERE mid = :mid 
            AND indicator = :indicator 
            AND period_type = :period_type 
            AND periodend = :periodend
        """
        stmt = text(sqlt)
        stmt = stmt.bindparams(
            mid=data_dict["mid"],
            indicator=data_dict["indicator"],
            period_type=data_dict["period_type"],
            periodend=data_dict["periodend"],
        )
        query = conn.execute(stmt)
        personnel_rating = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

        if len(personnel_rating) > 0:
            if data_dict["value"] is None or data_dict["value"] == 0:
                self.update_rating_info(
                    personnel_rating[0]["id"],
                    {
                        "title": None,
                        "document_number": None,
                        "document_date": None,
                        "document_date_to": None,
                        "achievement_result": None,
                        "info": None,
                        "classification_mark": None,
                    },
                )
                if personnel_rating[0]["link_type"] == 1:
                    os.remove(
                        os.path.join(
                            File(self.lname).get_personnel_rating_values_directory(
                                personnel_rating[0]["id"]
                            ),
                            personnel_rating[0]["link"],
                        )
                    )
                self.update_rating_link(personnel_rating[0]["id"], None, None)
            return self.update_rating(personnel_rating[0]["id"], data_dict)
        else:
            return self.create_rating(data_dict)

    def getRatingCalc(self, schema, grpupid, date, type):
        print(schema, grpupid, date, type)
        conn = current_app.ms.db(self.lname).connect()
        sqlt = """SELECT p.mid, p.lastname, p.firstname, p.patronymic, v.points 
                FROM people p
                INNER JOIN groupuser gu ON p.mid = gu.mid
                LEFT OUTER JOIN personnel_rating_values v ON v.mid = p.mid AND v.indicator = -:schema AND v.period_type = :type AND v.periodend = :date
                where gu.gid = :grpupid
                order by coalesce(v.points, 0) DESC, p.lastname, p.firstname, p.patronymic"""

        stmt = text(sqlt)
        stmt = stmt.bindparams(schema=schema)
        stmt = stmt.bindparams(grpupid=grpupid)
        stmt = stmt.bindparams(date=date)
        stmt = stmt.bindparams(type=type)
        query = conn.execute(stmt)

        result = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

        return result

    def getRatingCalcUpdated(self, schema, groupid, date, period_type):
        conn = current_app.ms.db(self.lname).connect()
        new_date = None
        period_query = " "

        if period_type:
            period_query = " AND v.period_type = :period_type"
            date_query = " AND v.periodend = :date"
        else:
            new_date = str(int(date) + 1) + "-07-01"
            date = date + "-08-31"
            date_query = " AND v.periodend >= :date AND v.periodend <= :new_date"

        schema_query = " "
        if schema:
            schema_query = " AND v.indicator = :schema "

        sqlt = """SELECT p.mid, p.lastname, p.firstname, p.patronymic, v.points 
                FROM people p
                INNER JOIN groupuser gu ON p.mid = gu.mid
                LEFT OUTER JOIN personnel_rating_values v ON v.mid = p.mid
                    {schema_query}
                    {period_query}
                    {date_query}
                where gu.gid = :groupid
                order by coalesce(v.points, 0) DESC, p.lastname, p.firstname, p.patronymic""".format(
            schema_query=schema_query, period_query=period_query, date_query=date_query
        )

        stmt = text(sqlt)
        stmt = stmt.bindparams(groupid=groupid)
        stmt = stmt.bindparams(date=date)

        if schema:
            stmt = stmt.bindparams(schema=schema)
        if period_type:
            stmt = stmt.bindparams(period_type=period_type)
        else:
            stmt = stmt.bindparams(new_date=new_date)
        query = conn.execute(stmt)

        result = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

        return result

    def get_rating_calc_details(self, schema, gid, date, type):
        conn = current_app.ms.db(self.lname).connect()
        sqlt = """  
            WITH RECURSIVE indicator_tree AS (
              SELECT 
                pi.id AS current_id,
                pi.title AS current_title,
                pi.id_group AS previous_id_group,
                pi.id AS first_level_id,
                pi.title AS first_level_title,
                1 AS level,
                COALESCE(pi.weight, 1) AS cumulative_weight
              FROM personnel_indicators pi
              JOIN rating_schema rs ON rs.id = pi.schema
              WHERE pi.id_group = 0 AND rs.id = :schema
            
              UNION ALL
            
              SELECT 
                pi.id AS current_id,
                pi.title AS current_title,
                pi.id_group AS previous_id_group,
                it.first_level_id,
                it.first_level_title,
                it.level + 1 AS level,
                it.cumulative_weight * COALESCE(pi.weight, 1) AS cumulative_weight
              FROM personnel_indicators pi
              JOIN rating_schema rs ON rs.id = pi.schema
              JOIN indicator_tree it ON pi.id_group = it.current_id
              WHERE rs.id = :schema
            ),
            
            raw_data AS (
              SELECT 
                p.mid,
                p.lastname, p.firstname, p.patronymic,
                it.first_level_id,
                it.first_level_title,
                SUM(prv.value) AS last_level_count,
                ROUND(SUM(prv.value * it.cumulative_weight), 2) AS last_level_points
              FROM people p
              INNER JOIN groupuser gu ON p.mid = gu.mid
              JOIN indicator_tree it ON true
              LEFT JOIN personnel_rating_values prv 
                ON prv.mid = p.mid 
                AND prv.indicator = it.current_id
                AND prv.period_type = :type 
                AND prv.periodend = :date
              WHERE gu.gid = :gid
                AND it.level >= 1
              GROUP BY p.mid, p.lastname, p.firstname, p.patronymic, it.first_level_id, it.first_level_title
            )
            
            SELECT 
              mid,
              lastname, firstname, patronymic,
              SUM(last_level_count) AS final_count,
              ROUND(SUM(last_level_points), 2) AS final_point,
              json_agg(
                json_build_object(
                  'first_level_title', first_level_title,
                  'last_level_count', last_level_count,
                  'last_level_points', last_level_points
                )
                ORDER BY first_level_id
              ) AS details
            FROM raw_data
            GROUP BY mid, lastname, firstname, patronymic
            ORDER BY final_point DESC NULLS LAST;
        """

        stmt = text(sqlt)
        stmt = stmt.bindparams(schema=schema)
        stmt = stmt.bindparams(gid=gid)
        stmt = stmt.bindparams(date=date)
        stmt = stmt.bindparams(type=type)
        query = conn.execute(stmt)

        result = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

        return result

    def get_rating_calc_details_diagrems(self, schema, gid, date, type):
        conn = current_app.ms.db(self.lname).connect()
        sqlt = """  
            WITH RECURSIVE indicator_tree AS (
                SELECT 
                    pi.id AS current_id,
                    pi.title AS current_title,
                    pi.id_group AS previous_id_group,
                    pi.id AS first_level_id,
                    pi.title AS first_level_title,
                    1 AS level,
                    COALESCE(pi.weight, 1) AS cumulative_weight
                FROM personnel_indicators pi
                WHERE pi.id_group = 0 AND pi.schema = :schema
                
                UNION ALL
                
                SELECT 
                    pi.id AS current_id,
                    pi.title AS current_title,
                    pi.id_group AS previous_id_group,
                    it.first_level_id,
                    it.first_level_title,
                    it.level + 1 AS level,
                    it.cumulative_weight * COALESCE(pi.weight, 1) AS cumulative_weight
                FROM personnel_indicators pi
                JOIN indicator_tree it ON pi.id_group = it.current_id
                WHERE pi.schema = :schema
            ),

            raw_data AS (
                SELECT 
                    p.mid,
                    p.lastname, p.firstname, p.patronymic,
                    it.first_level_id,
                    it.first_level_title,
                    SUM(prv.value) AS last_level_count,
                    ROUND(SUM(prv.value * it.cumulative_weight), 2) AS last_level_points
                FROM people p
                INNER JOIN groupuser gu ON p.mid = gu.mid
                JOIN indicator_tree it ON true
                LEFT JOIN personnel_rating_values prv 
                    ON prv.mid = p.mid 
                    AND prv.indicator = it.current_id
                    AND prv.period_type = :type 
                    AND prv.periodend = :date
                WHERE gu.gid = :gid
                    AND it.level >= 1
                GROUP BY p.mid, p.lastname, p.firstname, p.patronymic, it.first_level_id, it.first_level_title
            )

            SELECT 
                first_level_id,
                first_level_title,
                json_agg(
                    json_build_object(
                        'mid', mid,
                        'lastname', lastname,
                        'firstname', firstname,
                        'patronymic', patronymic,
                        'last_level_count', last_level_count,
                        'last_level_points', last_level_points
                    ) ORDER BY lastname, firstname, patronymic
                ) AS user_results
            FROM raw_data
            GROUP BY first_level_id, first_level_title
        """

        stmt = text(sqlt)
        stmt = stmt.bindparams(schema=schema)
        stmt = stmt.bindparams(gid=gid)
        stmt = stmt.bindparams(date=date)
        stmt = stmt.bindparams(type=type)
        query = conn.execute(stmt)

        result = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

        return result

    def getDigitalPortrait(self, schema, mid, yid):
        conn = current_app.ms.db(self.lname).connect()
        sqlt = """SELECT p.mid, p.lastname, p.firstname, p.patronymic, pi.*, vd.*
                    FROM people p
                    LEFT OUTER JOIN school_year sy ON sy.xp_key = :yid
                    LEFT OUTER JOIN personnel_indicators pi ON pi.schema = :schema  and (coalesce(pi.id_group, 0) = 0 or pi.top_level = true)
                    LEFT OUTER JOIN personnel_rating_values vd ON vd.mid = p.mid AND vd.indicator = pi.id AND vd.period_type = 1  AND vd.periodend BETWEEN sy.begdate and sy.enddate
                    where p.mid = :mid
                    order by pi.sorting, vd.periodend"""

        stmt = text(sqlt)
        stmt = stmt.bindparams(schema=schema)
        stmt = stmt.bindparams(mid=mid)
        stmt = stmt.bindparams(yid=yid)
        query = conn.execute(stmt)

        result = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

        return result

    def get_rating_edge(self, schema, type, date, kind, id):
        """
        kind
            1 - группа
            2 - курс
            3 - факультет
            4 - ВУЗ
        """
        conn = current_app.ms.db(self.lname).connect()

        where = ""
        if kind == 3:
            where = " and f.idfaculty = " + str(id)

        if kind == 2:
            where = " and g.year = " + str(id)

        if kind == 1:
            where = " and g.gid = " + str(id)

        sqlt = """  
            WITH RECURSIVE indicator_tree AS (
                SELECT 
                    pi.id AS current_id,
                    pi.title AS current_title,
                    pi.id_group AS previous_id_group,
                    pi.id AS first_level_id,
                    pi.title AS first_level_title,
                    1 AS level,
                    COALESCE(pi.weight, 1) AS cumulative_weight
                FROM personnel_indicators pi
                WHERE pi.id_group = 0 AND pi.schema = :schema
                
                UNION ALL
                
                SELECT 
                    pi.id AS current_id,
                    pi.title AS current_title,
                    pi.id_group AS previous_id_group,
                    it.first_level_id,
                    it.first_level_title,
                    it.level + 1 AS level,
                    it.cumulative_weight * COALESCE(pi.weight, 1) AS cumulative_weight
                FROM personnel_indicators pi
                JOIN indicator_tree it ON pi.id_group = it.current_id
                WHERE pi.schema = :schema
            ),

            raw_data AS (
                SELECT 
                    p.mid,
                    p.lastname, p.firstname, p.patronymic,
                    it.first_level_id,
                    it.first_level_title,
                    SUM(prv.value) AS last_level_count,
                    ROUND(SUM(prv.value * it.cumulative_weight), 2) AS last_level_points
                FROM people p
                INNER JOIN groupuser gu ON p.mid = gu.mid and gu.cid = -1
                JOIN indicator_tree it ON true
                LEFT JOIN personnel_rating_values prv 
                    ON prv.mid = p.mid 
                    AND prv.indicator = it.current_id
                    AND prv.period_type = :type 
                    AND prv.periodend = :date
                LEFT JOIN groupname g ON gu.gid = g.gid AND g.cid = -1
                LEFT JOIN cathedras c ON c.idcathedra = g.idcathedra
                LEFT JOIN faculty f ON f.idfaculty = coalesce(g.idfaculty, c.faculty)
                WHERE 1 = 1 {where} 
                    AND it.level >= 1
                GROUP BY p.mid, p.lastname, p.firstname, p.patronymic, it.first_level_id, it.first_level_title
            )

            SELECT 
                first_level_id,
                first_level_title,
                json_agg(
                    json_build_object(
                        'mid', mid,
                        'lastname', lastname,
                        'firstname', firstname,
                        'patronymic', patronymic,
                        'last_level_count', last_level_count,
                        'last_level_points', last_level_points
                    ) ORDER BY lastname, firstname, patronymic
                ) AS user_results
            FROM raw_data
            GROUP BY first_level_id, first_level_title
        """.format(where=where)

        stmt = text(sqlt)
        stmt = stmt.bindparams(schema=schema)
        stmt = stmt.bindparams(date=date)
        stmt = stmt.bindparams(type=type)
        query = conn.execute(stmt)

        result = [dict(zip(tuple(query.keys()), i)) for i in query.cursor]

        return result

    def getRatingReacal(self, schema, type, date):
        conn = (
            current_app.ms.db(self.lname).connect().execution_options(autocommit=True)
        )

        sqlt = "select refresh_plan_values_calculated(:schema, :type, :date)"
        stmt = text(sqlt)
        stmt = stmt.bindparams(schema=schema)
        stmt = stmt.bindparams(type=type)
        stmt = stmt.bindparams(date=date)
        conn.execute(stmt)

        return True

    def get_rating_info(self, id):
        conn = current_app.ms.db(self.lname).connect()
        sqlt = """
            SELECT 
                prv.*, 
                xp_format_fio(p.lastname,p.firstname,p.patronymic,0) as fio,
                ey.number yeducationyearid,
                ey.name yeducationyear,
                g.idcathedra cathedraid,
                cathedras.cathedra,
                g.name groupname,
                g.gid
            FROM personnel_rating_values prv 
            JOIN people p on p.mid = prv.mid
            left outer join groupuser gu on p.mid = gu.mid and gu.cid = -1           
            left outer join groupname g on gu.gid = g.gid and g.cid = -1           
            left outer join cathedras on cathedras.idcathedra = g.idcathedra
            left outer join educationyears ey on ey.number = g.year 
            WHERE id = :id
            """
        stmt = text(sqlt)
        stmt = stmt.bindparams(id=id)
        query = conn.execute(stmt)
        result = query.fetchone()
        return RequestUtils.convert_decimal_to_float(dict(result))

    def update_rating_info(self, id, data_dict):
        conn = current_app.ms.db(self.lname).connect()

        sqlt = """
            UPDATE personnel_rating_values 
            SET title = :title, document_number = :document_number, document_date = :document_date, 
                document_date_to = :document_date_to, achievement_result = :result, info = :info, 
                classification_mark = :classification_mark
            WHERE id = :id
            """

        stmt = text(sqlt)
        stmt = stmt.bindparams(
            id=id,
            title=data_dict["title"],
            document_number=data_dict["document_number"],
            document_date=data_dict["document_date"],
            document_date_to=data_dict["document_date_to"],
            result=data_dict["achievement_result"],
            info=data_dict["info"],
            classification_mark=data_dict["classification_mark"],
        )

        try:
            query = conn.execute(stmt)
        except Exception as e:
            return e.args
        return True

    def update_rating_link(self, id, link_type, link):
        conn = current_app.ms.db(self.lname).connect()

        sqlt = """
            UPDATE personnel_rating_values 
            SET link_type = :link_type, link = :link
            WHERE id = :id
            """

        stmt = text(sqlt)
        stmt = stmt.bindparams(
            id=id,
            link_type=link_type,
            link=link,
        )

        try:
            query = conn.execute(stmt)
        except Exception as e:
            return e.args
        return True
