from decimal import Decimal

from flask import current_app
from sqlalchemy import text


class ScholarshipCandidatesFilesStorage:
    def __init__(self, cname):
        self.lname = cname

    @staticmethod
    def get_files(
        lname: str,
        task_id,
        xp_key: int,
        education_level: int,
        scholarship_type: str,
        prep_struc_category_id: int,
    ):
        # Базовый SQL запрос
        sql = """
            WITH parsed_data AS (
                SELECT 
                    jsonb_array_elements(
                        CASE
                            WHEN :prep_struc_category_id = 1 AND :education_level = 1 THEN data->'scholarship_higher_education_cadet'
                            WHEN :prep_struc_category_id = 2 AND :education_level = 1 THEN data->'scholarship_higher_education_listener'
                            WHEN :prep_struc_category_id = 3 AND :education_level = 1 THEN data->'scholarship_higher_education_adjunct'
                            WHEN :prep_struc_category_id = 1 AND :education_level = 2 THEN data->'scholarship_secondary_education'
                            ELSE NULL
                        END
                    ) AS candidate_data
                FROM degree_candidates_task
                WHERE task_id = :task_id
            ),
            candidate_data AS (
                SELECT
                    CAST(candidate_data->>'mid' AS INT) AS mid,
                    CAST(candidate_data->>'prep_struc_category_id' AS INT) AS prep_struc_category_id,
                    CAST(candidate_data->>'education_level' AS INT) AS education_level
                FROM parsed_data
            )
				SELECT 
					scfs.file_path,
					scfs.file_name,
					scfs.сhapter,
					scfs.mid,
					p.lastname || ' ' || p.firstname || ' ' || p.patronymic AS fio
				FROM candidate_data dc
				JOIN scholarship_candidates sc 
					ON dc.mid = sc.mid AND sc.xp_key = :xp_key AND sc.education_level = :education_level
				LEFT JOIN scholarship_candidates_files_storage scfs ON scfs.mid = dc.mid
				JOIN people p ON p.mid = scfs.mid
				WHERE dc.education_level = :education_level
				  AND dc.prep_struc_category_id = :prep_struc_category_id
				  AND (sc.scholarship_type IS NULL OR sc.scholarship_type <> 'Нет')
				  AND sc.scholarship_type = :scholarship_type
        """

        # Выполнение запроса
        conn = current_app.ms.db(lname).connect()
        stmt = text(sql)
        stmt = stmt.bindparams(
            xp_key=xp_key, education_level=education_level, prep_struc_category_id=prep_struc_category_id, task_id=task_id, scholarship_type=scholarship_type
        )
        query = conn.execute(stmt)
        result = [dict(zip(tuple(query.keys()), row)) for row in query.cursor]
        return result

    def get_files_by_mid(self, mid):
        sql = """
            SELECT * FROM scholarship_candidates_files_storage
            WHERE mid = :mid
            ORDER BY сhapter
        """
        conn = current_app.ms.db(self.lname).connect()
        stmt = text(sql).bindparams(mid=mid)
        query = conn.execute(stmt)
        return [dict(zip(tuple(query.keys()), row)) for row in query.cursor]

    @staticmethod
    def create_files(
            lname: str,
            file_path: str,
            file_name: str,
            mid: int,
            сhapter: int,
    ):
        # Базовый SQL запрос
        sql = """
            INSERT INTO scholarship_candidates_files_storage (file_path, file_name, mid, сhapter)
            VALUES (:file_path, :file_name, :mid, :сhapter)
            RETURNING *
            """

        # Выполнение запроса
        conn = current_app.ms.db(lname).connect()
        stmt = text(sql)
        stmt = stmt.bindparams(
            file_path=file_path, file_name=file_name, mid=mid, сhapter=сhapter
        )
        query = conn.execute(stmt)
        return [dict(zip(tuple(query.keys()), row)) for row in query.cursor]

    @staticmethod
    def delete_file_record(lname, file_name, mid, сhapter):
        sql = """
            DELETE FROM scholarship_candidates_files_storage
            WHERE file_name = :file_name AND mid = :mid AND сhapter = :сhapter
            RETURNING *
        """
        conn = current_app.ms.db(lname).connect()
        stmt = text(sql).bindparams(file_name=file_name, mid=mid, сhapter=сhapter)
        result = conn.execute(stmt)
        return result.rowcount > 0
