# coding=UTF-8
from flask import Blueprint, jsonify, g, request
from flask_cors import CORS

from LMSAPI.api.Models.Announcement import Announcement
from LMSAPI.api.Models.File import File
from LMSAPI.api.Views.TokenAPI import auth

announcement_api = Blueprint('announcement_api', __name__)

CORS(announcement_api)


@announcement_api.route(
    "/lms/api/v1.0/<lname>/announcement",
    methods=["GET"],
)
@auth.login_required
def get_all_announcements(lname):
    beg_date = request.args.get("beg_date", default='10000101', type=str)
    end_date = request.args.get("end_date", default='30000101', type=str)
    result = Announcement(lname).get_all_announcements(g.user.mid, beg_date, end_date)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@announcement_api.route(
    "/lms/api/v1.0/<lname>/announcement/<int:post_id>",
    methods=["GET"],
)
@auth.login_required
def get_announcement(lname, post_id):
    result = Announcement(lname).get_announcement(post_id, g.user.mid)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@announcement_api.route(
    "/lms/api/v1.0/<lname>/announcement/<int:post_id>/file",
    methods=["GET"],
)
@auth.login_required
def get_announcement_file(lname, post_id):
    file_path = Announcement.get_announcement_image_path(lname=lname, post_id=post_id)
    ip_address = (
            request.headers.get('X-Forwarded-For') or
            request.headers.get('X-Real-IP') or
            request.environ.get("REMOTE_ADDR")
    )
    return File.send_from_directory(
        lname=lname,
        file_path=file_path,
        file_name='picture.png',
        action_source=request.environ.get("HTTP_USER_AGENT"),
        login=None,
        user_mid=None,
        ip_address=ip_address
    )


@announcement_api.route(
    "/lms/api/v1.0/<lname>/announcement/<int:post_id>/file/<int:num_chapter>",
    methods=["GET"],
)
@auth.login_required
def get_announcement_file_by_chapter(lname, post_id, num_chapter):
    file_path = Announcement.get_announcement_image_path(lname=lname, post_id=post_id)
    ip_address = (
            request.headers.get('X-Forwarded-For') or
            request.headers.get('X-Real-IP') or
            request.environ.get("REMOTE_ADDR")
    )
    return File.send_from_directory(
        lname=lname,
        file_path="part_picture_" + file_path + ".png",
        file_name=num_chapter,
        action_source=request.environ.get("HTTP_USER_AGENT"),
        login=None,
        user_mid=None,
        ip_address=ip_address
    )


@announcement_api.route(
    "/lms/api/v1.0/<lname>/announcement/how_unread_announcement",
    methods=["GET"],
)
@auth.login_required
def how_many_unread_announcement(lname):
    result = Announcement(lname).how_many_unread_announcement(g.user.mid)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)
