# coding=UTF-8
import sys
from decimal import Decimal

from flask import jsonify, request, json
from flask import Blueprint

from LMSAPI.api.Models.EnrollAnswers import EnrollAnswers
from LMSAPI.api.Models.EnrollQuestions import EnrollQuestions
from LMSAPI.api.Views.TokenAPI import auth
from flask_cors import CORS

enroll_questions_api = Blueprint("enroll_questions_api", __name__)

CORS(enroll_questions_api)


@enroll_questions_api.route(
    "/lms/api/v1.0/<lname>/enroll_questions",
    methods=["GET"],
)
@auth.login_required
def get_enroll_questions_all(lname):
    question_type = request.args.get("question_type", default=None, type=int)
    cid = request.args.get("cid", default=None, type=int)
    oid = request.args.get("oid", default=None, type=int)
    tlid = request.args.get("tlid", default=None, type=int)
    eyid = request.args.get("eyid", default=None, type=int)
    status = request.args.get("status", default=None, type=int)

    result = EnrollQuestions().get_enroll_questions_all(
        lname, question_type, cid, oid, tlid, eyid, status
    )

    if result is None:
        return jsonify(success=False), 404

    # Convert Decimal to float
    for row in result:
        for key, value in row.items():
            if isinstance(value, Decimal):
                row[key] = float(value)

    return jsonify(result)


@enroll_questions_api.route(
    "/lms/api/v1.0/<lname>/enroll_questions/<int:id>",
    methods=["GET"],
)
@auth.login_required
def get_enroll_questions_by_id(lname, id: int):
    result = EnrollQuestions().get_enroll_questions_by_id(lname, id)

    if result is None:
        return jsonify(success=False), 404

    for key, value in result.items():
        if isinstance(value, Decimal):
            result[key] = float(value)

    return jsonify(result)


@enroll_questions_api.route(
    "/lms/api/v1.0/<lname>/enroll_questions",
    methods=["POST"],
)
@auth.login_required
def create_enroll_questions(lname):
    data = request.data
    if not data:
        return jsonify({"error": "No data provided"}), 400

    if sys.version_info[0] < 3:
        data = json.loads(data)
    else:
        data = json.loads(data.decode("utf-8"))

    result = EnrollQuestions().create_enroll_questions(lname, data)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(success=True), 201


@enroll_questions_api.route(
    "/lms/api/v1.0/<lname>/enroll_questions/<int:id>",
    methods=["PUT"],
)
@auth.login_required
def update_enroll_questions(lname, id: int):
    data = request.data
    if not data:
        return jsonify({"error": "No data provided"}), 400

    if sys.version_info[0] < 3:
        data = json.loads(data)
    else:
        data = json.loads(data.decode("utf-8"))

    result = EnrollQuestions().update_enroll_questions(lname, id, data)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(success=True), 201


@enroll_questions_api.route(
    "/lms/api/v1.0/<lname>/enroll_questions/<int:id>",
    methods=["DELETE"],
)
@auth.login_required
def delete_enroll_questions(lname, id: int):
    result = EnrollQuestions().delete_enroll_questions(lname, id)

    if result is None:
        return jsonify(success=False), 404
    return jsonify(success=True), 200


def validate_data_in_questions_and_answer(data):
    if isinstance(data, list):
        # Одиночные вопросы
        if len(data) == 1:
            question_type = data[0].get("question_type")
            data[0].pop("id_parent", None)
            if question_type not in [1, 2, 3, 4]:
                return (
                    jsonify(
                        {
                            "error": "Если вопрос один, его question_type должен быть от 1 до 4"
                        }
                    ),
                    400,
                )

            answers = data[0].get("answers", [])
            questions = {
                key: value for key, value in data[0].items() if key != "answers"
            }

            if questions.get("question_type") == 1:
                correct_answers = sum(
                    answer.get("correct", False) for answer in answers
                )
                if correct_answers != 1:
                    return (
                        jsonify(
                            {
                                "error": "Не может быть более одного или ни одного правильного ответа для question_type = 1"
                            }
                        ),
                        400,
                    )
                for answer in data[0].get("answers", []):
                    answer.pop("answer_group", None)

            if questions.get("question_type") == 2:
                correct_answers = sum(
                    answer.get("correct", False) for answer in answers
                )
                if correct_answers <= 1:
                    return (
                        jsonify(
                            {
                                "error": "Не может быть один и меньше правильного ответа для question_type = 2"
                            }
                        ),
                        400,
                    )
                for answer in data[0].get("answers", []):
                    answer.pop("answer_group", None)

            if questions.get("question_type") == 3:
                if len(answers) != 1 or not answers[0].get("correct", False):
                    return (
                        jsonify(
                            {
                                "error": "Для question_type = 3 должен быть ровно один ответ, и он должен быть правильным"
                            }
                        ),
                        400,
                    )
                data[0].get("answers", [])[0].pop("answer_group", None)

            if questions.get("question_type") == 4:
                required_groups = {1, 2}
                answer_groups = {answer.get("answer_group") for answer in answers}
                if not required_groups.issubset(answer_groups):
                    return (
                        jsonify(
                            {
                                "error": "Для question_type = 4 ответы должны содержать порядок как минимум в двух ответах с значениями 1 и 2 в answer_group"
                            }
                        ),
                        400,
                    )
                # Проверка, что ответы с answer_group 1 и 2 имеют correct=True
                for answer in answers:
                    if answer.get("answer_group") in required_groups and not answer.get(
                        "correct", False
                    ):
                        return (
                            jsonify(
                                {
                                    "error": "Ответы с answer_group 1 и 2 должны иметь correct=True для question_type = 4"
                                }
                            ),
                            400,
                        )

            return data  # Успешный результат - возвращаем данные

        # Связанные вопросы
        elif len(data) > 1:
            for question in data:
                if question.get("question_type") != 5:
                    return (
                        jsonify(
                            {
                                "error": "Если вопросов больше одного, их question_type должен быть равен 5"
                            }
                        ),
                        400,
                    )

                correct_answers = sum(
                    answer.get("correct", False)
                    for answer in question.get("answers", [])
                )
                if correct_answers != 1:
                    return (
                        jsonify(
                            {
                                "error": "У каждого вопроса с question_type = 5 должен быть ровно один правильный ответ"
                            }
                        ),
                        400,
                    )
                for answer in question.get("answers", []):
                    answer.pop("answer_group", None)

            return data  # Успешный результат - возвращаем данные
    else:
        return (
            jsonify({"error": "Неверный формат данных. Ожидается список вопросов."}),
            400,
        )


@enroll_questions_api.route("/lms/api/v1.0/<lname>/questions/answers", methods=["POST"])
@auth.login_required
def add_questions_and_answer(lname):
    data = request.data
    if not data:
        return jsonify({"error": "No data provided"}), 400

    if sys.version_info[0] < 3:
        data = json.loads(data)
    else:
        data = json.loads(data.decode("utf-8"))

    # Валидация данных
    data = validate_data_in_questions_and_answer(data)

    if isinstance(data, tuple):
        # Если вернулся кортеж, это ошибка, возвращаем её
        return data

    if isinstance(data, list) and len(data) == 1:
        # Удаление answers из данных вопроса и создание записи в EnrollQuestions
        question_data = data[0]
        answers = question_data.pop("answers", [])

        question_result = EnrollQuestions().create_enroll_questions(
            lname, question_data
        )
        question_id = question_result[0]["id"] if question_result else None
        if not question_id:
            return jsonify({"error": "Не удалось создать запись вопроса"}), 500
        question_data["question_id"] = question_id

        # Добавление question_id в каждый ответ и создание записей в EnrollAnswers
        for answer in answers:
            answer["question"] = question_id
            answer_result = EnrollAnswers().create_enroll_answers(lname, answer)
            answer_id = answer_result[0]["id"] if answer_result else None
            if not answer_id:
                return jsonify({"error": "Не удалось создать запись ответа"}), 500
            answer["answer_id"] = answer_id
        question_data["answers"] = answers
        return jsonify(success=True, data=question_data), 201

    # Связанные вопросы
    elif isinstance(data, list) and len(data) > 1:
        previous_question_id = None

        result = []
        for question_data in data:
            answers = question_data.pop("answers", [])

            # Установка поля id_parent для всех вопросов
            if previous_question_id is None:
                question_data["id_parent"] = 0  # Первый вопрос
            else:
                question_data["id_parent"] = previous_question_id

            # Создание записи вопроса
            question_result = EnrollQuestions().create_enroll_questions(
                lname, question_data
            )
            question_id = question_result[0]["id"] if question_result else None
            if not question_id:
                return jsonify({"error": "Не удалось создать запись вопроса"}), 500
            question_data["question_id"] = question_id

            # Обновление предыдущего вопроса ID
            previous_question_id = question_id

            # Добавление question_id в каждый ответ и создание записей в EnrollAnswers
            for answer in answers:
                answer["question"] = question_id
                answer_result = EnrollAnswers().create_enroll_answers(lname, answer)
                answer_id = answer_result[0]["id"] if answer_result else None
                if not answer_id:
                    return jsonify({"error": "Не удалось создать запись ответа"}), 500
                answer["answer_id"] = answer_id
            question_data["answers"] = answers
            result.append(question_data)
        return jsonify(success=True, data=result), 201

    return (
        jsonify({"error": "Неверный формат данных. Ожидается список вопросов."}),
        400,
    )


@enroll_questions_api.route(
    "/lms/api/v1.0/<lname>/questions/<int:question_id>/answers",
    methods=["PUT"],
)
@auth.login_required
def put_questions_and_answer(lname, question_id: int):
    data = request.data
    if not data:
        return jsonify({"error": "No data provided"}), 400

    if sys.version_info[0] < 3:
        data = json.loads(data)
    else:
        data = json.loads(data.decode("utf-8"))

    # Валидация данных
    data = validate_data_in_questions_and_answer(data)

    if isinstance(data, tuple):
        # Если вернулся кортеж, это ошибка, возвращаем её
        return data

    if isinstance(data, list) and len(data) == 1:
        # Удаление answers из данных вопроса и создание записи в EnrollQuestions
        question_data = data[0]
        answers = question_data.pop("answers", [])

        question_result = EnrollQuestions().update_enroll_questions(
            lname, question_id, question_data
        )
        question_id = question_result[0]["id"] if question_result else None
        if not question_id:
            return jsonify({"error": "Не удалось изменить запись вопроса"}), 500

        # Добавление question_id в каждый ответ и создание записей в EnrollAnswers
        for answer in answers:
            answer["question"] = question_id
            # Проверка есть ли answer_id в каждом из ответов, если есть обновляем записи, если нет то создаем новую
            answer_id = answer.pop("id", None)
            if answer_id:
                answer_result = EnrollAnswers().update_enroll_answers(
                    lname, answer_id, answer
                )
            else:
                answer_result = EnrollAnswers().create_enroll_answers(lname, answer)
            if not answer_result:
                return (
                    jsonify({"error": "Не удалось создать/обновить запись ответа"}),
                    500,
                )
        question_data["answers"] = answers
        return jsonify(success=True, data=question_data), 201

    # Связанные вопросы
    elif isinstance(data, list) and len(data) > 1:
        previous_question_id = None
        result = []

        for question_data in data:
            answers = question_data.pop("answers", [])
            question_id = question_data.pop("id", None)

            # Установка поля id_parent для всех вопросов
            if previous_question_id is None:
                question_data["id_parent"] = 0  # Первый вопрос
            else:
                question_data["id_parent"] = previous_question_id

            if question_id:
                # Изменения записи вопроса
                question_result = EnrollQuestions().update_enroll_questions(
                    lname, question_id, question_data
                )
            else:
                # Создание записи вопроса
                question_result = EnrollQuestions().create_enroll_questions(
                    lname, question_data
                )
            question_id = question_result[0]["id"] if question_result else None
            if not question_id:
                return (
                    jsonify({"error": "Не удалось создать/изменить запись вопроса"}),
                    500,
                )

            # Обновление предыдущего вопроса ID
            previous_question_id = question_id

            # Добавление question_id в каждый ответ и создание записей в EnrollAnswers
            for answer in answers:
                answer["question"] = question_id
                answer_id = answer.pop("id", None)
                if answer_id:
                    answer_result = EnrollAnswers().update_enroll_answers(
                        lname, answer_id, answer
                    )
                else:
                    answer_result = EnrollAnswers().create_enroll_answers(lname, answer)
                if not answer_result:
                    return (
                        jsonify({"error": "Не удалось создать/обновить запись ответа"}),
                        500,
                    )
            question_data["answers"] = answers
            result.append(question_data)
        return jsonify(success=True, data=result), 200

    return (
        jsonify({"error": "Неверный формат данных. Ожидается список вопросов."}),
        400,
    )


@enroll_questions_api.route(
    "/lms/api/v1.0/<lname>/questions/<int:question_id>/answers",
    methods=["DELETE"],
)
@auth.login_required
def delete_questions_and_answer(lname, question_id: int):
    # Удаления ответов на вопрос
    answer_result = EnrollAnswers().delete_enroll_answers_by_question(
        lname, question_id
    )
    answer_id = answer_result[0]["id"] if answer_result else None
    if not answer_id:
        return jsonify({"error": "Не удалось удалить ответы"}), 500

    # Удаления вопросов
    question_result = EnrollQuestions().delete_enroll_questions(lname, question_id)
    question_id = question_result[0]["id"] if question_result else None
    if not question_id:
        return jsonify({"error": "Не удалось удалить вопроса"}), 500

    return jsonify(success=True, question_id=question_id), 200
