import json
import os
import threading
import uuid
import zipfile

from flask import Blueprint, jsonify, request, g, current_app
from flask_cors import CORS
from werkzeug.utils import secure_filename

from LMSAPI.api.Models.File import File
from LMSAPI.api.Models.ImportApplicant import ImportApplicant
from LMSAPI.api.Views.TokenAPI import auth
from LMSAPI.api.utils.gpghlms import importPGPKey, statusPGPKey
from LMSAPI.api.utils.gpghelper import getGPGVersion
from LMSAPI.api.utils.access_utils import user_permission_modes

gpg_api = Blueprint("gpg_api", __name__)

CORS(gpg_api)


@gpg_api.route("/lms/api/v1.0/<lname>/gpg_key", methods=["POST"])
@auth.login_required
@user_permission_modes("Настройка", "Ключи шифрования", ["Нет"])
def import_private_key(lname):

    if "file" not in request.files:
        return jsonify({"error": "No file part in the request"}), 400

    file = request.files["file"]
    filename = file.filename

    if filename == "":
        return jsonify({"error": "No file selected"}), 400
    if not filename.lower().endswith(".gpg"):
        return jsonify({"error": "Загруженный файл не является файлом GPG"}), 400

    force = request.args.get('force')

    
    forcearg = False
    if force == "true":
        forcearg = True

    
    fstream = file.stream
    keydata = fstream.read().decode("utf-8")

    lservers = current_app.config.get("LMSSERVERS")
    lserver = next((lserver for lserver in lservers if lserver.get("NAME") == lname), None)
    jakarta_serial = lserver.get("KEYSERIAL")
    jakarta_keyname = lserver.get("KEYNAME")

    if jakarta_serial is not None and jakarta_keyname is not None:
        source = 'config'
    else:
        jakarta_serial = current_app.jakarta_serial
        jakarta_keyname = current_app.jakarta_keyname
        source = 'key'

    if jakarta_serial is None or jakarta_keyname is None :
        return jsonify({ "result": "error", "error": "Секретного ключа не существует (USB/Config)", "keyerror": current_app.jakarta_error}),400

    status = importPGPKey(keydata,jakarta_serial, jakarta_keyname, forcearg)
    status['keysource'] = source

    if status['result']=='error':
        return jsonify(status),400    

    return jsonify(status)


@gpg_api.route("/lms/api/v1.0/<lname>/gpg_key", methods=["GET"])
@auth.login_required
@user_permission_modes("Настройка", "Ключи шифрования", ["Нет"])
def key_status(lname):    

    status = statusPGPKey()

    lservers = current_app.config.get("LMSSERVERS")
    lserver = next((lserver for lserver in lservers if lserver.get("NAME") == lname), None)
    jakarta_serial = lserver.get("KEYSERIAL")
    jakarta_keyname = lserver.get("KEYNAME")

    if jakarta_serial is not None and jakarta_keyname is not None:
        status['keysource'] = 'config'
    else:
        status['keysource'] = 'key'
        if current_app.jakarta_error is not None:
            status['keyerror'] = current_app.jakarta_error
            status['result'] = 'error'

    return jsonify(status)

@gpg_api.route("/lms/api/v1.0/<lname>/gpg_version", methods=["GET"])
@auth.login_required
@user_permission_modes("Настройка", "Ключи шифрования", ["Нет"])
def gpg_version(lname):    

    status = getGPGVersion()
    try:
        pkcs_lib = current_app.config['PKCS11_MODULE']
    except:
        pkcs_lib = None

    if pkcs_lib is None:
        pkcs_lib = '/usr/lib/libjcPKCS11-2.so'

    status['PKCS11_MODULE'] = pkcs_lib

    return jsonify(status)
