# coding=UTF-8
import os
from io import BytesIO
from urllib.parse import urlparse

from flask import Flask, jsonify, request, current_app, g, send_file
from flask_restful import Resource, Api
from sqlalchemy import create_engine
from sqlalchemy.sql import text
from flask import Blueprint

from LMSAPI.api.Models.File import File
from LMSAPI.api.Views.TokenAPI import auth
from LMSAPI.api.Models.User import User
from LMSAPI.api.Models.Journal import Journal
from flask_cors import CORS
import json, sys

from LMSAPI.api.utils.access_utils import user_permission_modes
from LMSAPI.api.utils.request_utils import RequestUtils

journal_api = Blueprint("journal_api", __name__)

CORS(journal_api)


@journal_api.route(
    "/lms/api/v1.0/<lname>/journal/<int:termid>/<onedate>/<int:courseid>/<int:groupid>",
    methods=["GET"],
)
@auth.login_required
@user_permission_modes("Режимы", "Журнал учета учебных занятий", ["Нет"])
def getJournal(lname, termid, onedate, courseid, groupid):
    """GET for Journal"""
    security = request.args.get("security", default=0, type=int)
    open_lesson = request.args.get("open_lesson", default=0, type=int)
    their_lesson = request.args.get("their_lesson", default=0, type=int)
    result = Journal().getJournal(
        lname,
        termid,
        onedate,
        courseid,
        groupid,
        0,
        security,
        open_lesson,
        their_lesson,
    )

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@journal_api.route("/lms/api/v1.0/<lname>/journal/grade", methods=["POST"])
@auth.login_required
def putGrade(lname):
    if not g.user.isTeacher:
        return jsonify("Access denied!"), 403

    data = request.data
    if sys.version_info[0] < 3:
        dataDict = json.loads(data)
    else:
        dataDict = json.loads(data.decode("utf-8"))

    grade = None
    if "grade" in dataDict:
        grade = dataDict["grade"]
    if grade == None:
        return jsonify("missing grade parameter"), 502

    if grade != "":
        result = Journal().checkGrade(lname, grade)
        if result is False:
            return jsonify("invalid grade " + grade), 502

    sheid = None
    if "sheid" in dataDict:
        sheid = dataDict["sheid"]
    if sheid == None:
        return jsonify("missing sheid parameter"), 502

    week_id = None
    if "week_id" in dataDict:
        week_id = dataDict["week_id"]
    if week_id == None:
        return jsonify("missing week_id parameter"), 502

    teacher_mid = None
    if "teacher_mid" in dataDict:
        teacher_mid = dataDict["teacher_mid"]
    if teacher_mid == None:
        return jsonify("missing teacher_mid parameter"), 502

    mid = None
    if "mid" in dataDict:
        mid = dataDict["mid"]
    if mid == None:
        return jsonify("missing mid parameter"), 502

    Journal().setGrade(lname, sheid, week_id, grade, mid, teacher_mid)

    return jsonify(True)


@journal_api.route("/lms/api/v1.0/<lname>/journal/termgrade", methods=["POST"])
@auth.login_required
def putTermGrade(lname):
    if g.user.isTeacher == False:
        return jsonify("Access denied!"), 403

    data = request.data
    if sys.version_info[0] < 3:
        dataDict = json.loads(data)
    else:
        dataDict = json.loads(data.decode("utf-8"))

    grade = None
    if "grade" in dataDict:
        grade = dataDict["grade"]
    if grade == None:
        return jsonify("missing grade parameter"), 502

    trmid = None
    if "trmid" in dataDict:
        trmid = dataDict["trmid"]
    if trmid == None:
        return jsonify("missing trmid parameter"), 502

    cid = None
    if "cid" in dataDict:
        cid = dataDict["cid"]
    if cid == None:
        return jsonify("missing cid parameter"), 502

    teacher_mid = None
    if "teacher_mid" in dataDict:
        teacher_mid = dataDict["teacher_mid"]
    if teacher_mid == None:
        return jsonify("missing teacher_mid parameter"), 502

    mid = None
    if "mid" in dataDict:
        mid = dataDict["mid"]
    if mid == None:
        return jsonify("missing mid parameter"), 502

    Journal().setTermGrade(lname, trmid, cid, grade, mid, teacher_mid)

    return jsonify(True)


@journal_api.route(
    "/lms/api/v1.0/<lname>/section_and_topic/<int:sheid>/<int:module_id>",
    methods=["GET"],
)
@auth.login_required
def get_number_and_name_of_the_section_and_topic(lname, sheid, module_id):
    result = Journal().get_number_and_name_of_the_section_and_topic_one(
        lname, sheid, module_id
    )

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@journal_api.route(
    "/lms/api/v1.0/<lname>/section_and_topic/all/<int:sheid>/<int:module_id>",
    methods=["GET"],
)
@auth.login_required
def get_number_and_name_of_the_section_and_topic_all(lname, sheid, module_id):
    result = Journal().get_number_and_name_of_the_section_and_topic_all(
        lname, sheid, module_id
    )

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@journal_api.route(
    "/lms/api/v1.0/<lname>/section_and_topic",
    methods=["POST"],
)
@auth.login_required
def create_number_and_name_of_the_section_and_topic(lname):
    data_dict = RequestUtils.get_body(request)
    result = Journal().create_number_and_name_of_the_section_and_topic(lname, data_dict)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@journal_api.route(
    "/lms/api/v1.0/<lname>/journal_rashod/<onedate>/<int:groupid>", methods=["GET"]
)
@auth.login_required
def getJournalRashod(lname, onedate, groupid):
    result = Journal().getJournalRashod(lname, onedate, groupid)
    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@journal_api.route("/lms/api/v1.0/<lname>/journal/grade_values", methods=["GET"])
@auth.login_required
def getJournalGradeValues(lname):
    result = Journal().getJournalGradeValues(lname)
    if result is None:
        return jsonify(success=False), 404
    return jsonify(result)


@journal_api.route("/lms/api/v1.0/<lname>/journal/grade_values/absent", methods=["GET"])
@auth.login_required
def getJournalAbsentGrades(lname):
    result = Journal().getJournalAbsentGrades(lname)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@journal_api.route(
    "/lms/api/v1.0/<lname>/journal/planner_file/<int:sheid>/<int:week_id>",
    methods=["GET"],
)
@auth.login_required
def getUserFiles(lname, sheid, week_id):
    if g.user.isTeacher == False:
        return jsonify("Error: Not teacher owning project id"), 401

    f = File(lname)
    return jsonify(f.getJournalPlanner(sheid, week_id))


@journal_api.route(
    "/lms/api/v1.0/<lname>/journal/planner_file/<int:sheid>/<int:week_id>/<filename>",
    methods=["GET"],
)
@auth.login_required
def getUserFile(lname, sheid, week_id, filename):
    if g.user.isTeacher == False:
        return jsonify("Error: Not teacher owning project id"), 401

    path = File(lname).getJournalPlannerDirectory(sheid, week_id)
    ip_address = (
        request.headers.get("X-Forwarded-For")
        or request.headers.get("X-Real-IP")
        or request.environ.get("REMOTE_ADDR")
    )
    return File.send_from_directory(
        lname=lname,
        file_path=path,
        file_name=filename,
        action_source=request.environ.get("HTTP_USER_AGENT"),
        login=g.user.id,
        user_mid=g.user.mid,
        ip_address=ip_address,
    )


@journal_api.route(
    "/lms/api/v1.0/<lname>/journal/planner_file/<int:sheid>/<int:week_id>",
    methods=["POST"],
)
@auth.login_required
def save_planner_file(lname, sheid, week_id):
    if g.user.isTeacher == False:
        return jsonify("Error: Not teacher owning project id"), 401

    submitted_file = request.files["file"]

    ip_address = (
        request.headers.get("X-Forwarded-For")
        or request.headers.get("X-Real-IP")
        or request.environ.get("REMOTE_ADDR")
    )

    if submitted_file:
        File(lname).createJournalPlannerDirectory(sheid, week_id)
        file_path = File(lname).getJournalPlannerDirectory(sheid, week_id)
        return File.save_file_to_path(
            lname=lname,
            submitted_file=submitted_file,
            file_path=file_path,
            action_source=request.environ.get("HTTP_USER_AGENT"),
            login=g.user.id,
            user_mid=g.user.mid,
            ip_address=ip_address,
        )

    return jsonify({"success": False, "error": "upload file with key file"})


@journal_api.route(
    "/lms/api/v1.0/<lname>/journal/planner_file/<int:sheid>/<int:week_id>/<filename>",
    methods=["DELETE"],
)
@auth.login_required
def delete_planner_file(lname, sheid, week_id, filename):
    if g.user.isTeacher == False:
        return jsonify("Error: Not teacher owning project id"), 401

    try:
        os.remove(
            os.path.join(
                File(lname).getJournalPlannerDirectory(sheid, week_id), filename
            )
        )
        os.rmdir(File(lname).getJournalPlannerDirectory(sheid, week_id))
        return jsonify(success=True)

    except:
        current_app.logger.error(
            "delete_planner_file(" + sheid + "," + week_id + "," + filename + ")"
        )
        return jsonify(success=False), 404


@journal_api.route(
    "/lms/api/v1.0/<lname>/journal/lesson_type_icons/<int:lessontypeid>",
    methods=["GET"],
)
@auth.login_required
def get_lesson_type_iconss(lname, lessontypeid):
    value = Journal().get_lesson_type_icons(lname, lessontypeid=lessontypeid)
    if value[0]["bmp"] and value[0]["bmp"] != b"NULL":
        return send_file(BytesIO(value[0]["bmp"]), mimetype="image/png")
    else:
        value = Journal().get_lesson_type_icons(lname, lessontypeid=0)
        return send_file(BytesIO(value[0]["bmp"]), mimetype="image/png")


@journal_api.route(
    "/lms/api/v1.0/<lname>/journal/lesson_type_icons/<int:lessontypeid>",
    methods=["POST"],
)
@auth.login_required
def create_lesson_type_iconss(lname, lessontypeid):
    file = request.files["file"]
    value = Journal().get_lesson_type_icons(lname, lessontypeid=lessontypeid)
    if value[0]["bmp"] and value[0]["bmp"] != b"NULL":
        Journal().delete_lesson_type_icons(lname, lessontypeid=lessontypeid)
    Journal().create_lesson_type_icons(
        lname, lessontypeid=lessontypeid, image_bytes=file.read()
    )
    return jsonify(success=True), 201


@journal_api.route(
    "/lms/api/v1.0/<lname>/journal/lesson_type_icons/<int:lessontypeid>",
    methods=["DELETE"],
)
@auth.login_required
def delete_lesson_type_iconss(lname, lessontypeid):
    Journal().delete_lesson_type_icons(lname, lessontypeid=lessontypeid)
    return jsonify(success=True), 201


@journal_api.route(
    "/lms/api/v1.0/<lname>/tmp_sheid_weeks/<int:termid>/<int:courseid>/<int:groupid>",
    methods=["GET"],
)
@auth.login_required
def get_tmp_sheid_weeks(lname, termid, courseid, groupid):
    """GET for Journal"""
    security = request.args.get("security", default=0, type=int)
    open_lesson = request.args.get("open_lesson", default=0, type=int)
    their_lesson = request.args.get("their_lesson", default=0, type=int)
    onedate = request.args.get("onedate", type=str)
    result = Journal().get_tmp_sheid_weeks(
        lname,
        termid,
        onedate,
        courseid,
        groupid,
        0,
        security,
        open_lesson,
        their_lesson,
    )

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@journal_api.route(
    "/lms/api/v1.0/<lname>/tmp_sheid_weeks/held",
    methods=["POST"],
)
@auth.login_required
def post_held_tmp_sheid_weeks(lname):
    data_dict = RequestUtils.get_body(request)
    result = Journal().post_held_tmp_sheid_weeks(lname, data_dict)

    if result is False:
        return jsonify(success=False), 404

    return jsonify(success=True), 200


@journal_api.route(
    "/lms/api/v1.0/<lname>/сlass_control/<int:courseid>/<int:groupid>",
    methods=["GET"],
)
@auth.login_required
def get_control_of_class(lname, courseid, groupid):
    """GET for Journal"""
    idcathedra = request.args.get("idcathedra", default=0, type=int)
    idfaculty = request.args.get("idfaculty", default=0, type=int)
    trmid = request.args.get("trmid", default=0, type=int)
    onedate = request.args.get("onedate", type=str)
    result = Journal().get_control_of_class(
        lname, idcathedra, courseid, groupid, idfaculty, trmid, onedate
    )

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@journal_api.route(
    "/lms/api/v1.0/<lname>/eventtools",
    methods=["GET"],
)
@auth.login_required
def get_eventtools(lname):
    result = Journal().get_eventtools(lname)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@journal_api.route(
    "/lms/api/v1.0/<lname>/class_topic/<int:sheid>/<int:week_id>",
    methods=["GET"],
)
@auth.login_required
def get_class_topic(lname, sheid, week_id):
    result = Journal().get_class_topic(lname, sheid, week_id)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@journal_api.route(
    "/lms/api/v1.0/<lname>/class_topic/<int:sheid>/<int:week_id>",
    methods=["PUT"],
)
@auth.login_required
def update_class_topic(lname, sheid, week_id):
    data = request.json

    if not data:
        return jsonify({"error": "No data provided"}), 400

    result = Journal().update_class_topic(lname, sheid, week_id, data)

    if result is False:
        return jsonify(success=False), 404

    return jsonify(success=True), 200


@journal_api.route(
    "/lms/api/v1.0/<lname>/class_topic/<int:sheid>/<int:week_id>",
    methods=["DELETE"],
)
@auth.login_required
def delete_class_topic(lname, sheid, week_id):
    result = Journal().delete_class_topic(lname, sheid, week_id)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(success=True), 200


@journal_api.route(
    "/lms/api/v1.0/<lname>/plan_table",
    methods=["GET"],
)
@auth.login_required
def get_plan_table(lname):
    course_tema_detail_id = request.args.get(
        "course_tema_detail_id", default=None, type=int
    )
    result = Journal().get_plan_table(lname, course_tema_detail_id)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@journal_api.route(
    "/lms/api/v1.0/<lname>/electronic_test/<int:sheid>/<int:week_id>",
    methods=["GET"],
)
@auth.login_required
def get_electronic_test(lname, sheid, week_id):
    mids = request.args.get("mids")

    if not mids:
        return jsonify({"error": "Missing mids parameter"}), 400

    try:
        mid_list = [int(mid) for mid in mids.split(",")]
    except ValueError:
        return jsonify({"error": "Invalid mids parameter"}), 400

    result = Journal().get_electronic_test(lname, sheid, week_id, mid_list)

    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@journal_api.route(
    "/lms/api/v1.0/<lname>/journal_lesson_detail/<int:sheid>/<int:instance>/<filename>",
    methods=["GET"],
)
@auth.login_required
def get_journal_lesson_detail_file(lname, sheid, instance, filename):
    file_path = File(lname).get_journal_lesson_detail_directory(sheid, instance)
    ip_address = (
        request.headers.get("X-Forwarded-For")
        or request.headers.get("X-Real-IP")
        or request.environ.get("REMOTE_ADDR")
    )
    return File.send_from_directory(
        lname=lname,
        file_path=file_path,
        file_name=filename,
        action_source=request.environ.get("HTTP_USER_AGENT"),
        login=g.user.id,
        user_mid=g.user.mid,
        ip_address=ip_address,
    )


@journal_api.route(
    "/lms/api/v1.0/<lname>/journal_lesson_detail/link/<int:sheid>/<int:instance>",
    methods=["POST"],
)
@auth.login_required
def post_journal_lesson_detail_reference(lname, sheid, instance):
    if g.user.isTeacher == False:
        return jsonify(success=False), 403

    submitted_link = request.form.get("link")  # Получаем ссылку из запроса
    if submitted_link:
        File(lname).create_journal_lesson_detail_directory(sheid, instance)
        file_path = File(lname).get_journal_lesson_detail_directory(sheid, instance)

        # Извлекаем имя файла из ссылки
        parsed_url = urlparse(submitted_link)
        file_name = os.path.basename(parsed_url.path) + ".url"

        # Создаем файл с расширением .url и записываем в него ссылку
        file_path = os.path.join(file_path, file_name)
        with open(file_path, "w") as f:
            f.write("[InternetShortcut]\n")
            f.write("URL=" + submitted_link + "\n")

        return jsonify(
            success=True,
            message="Link saved successfully as '{file_name}'".format(
                file_name=file_name
            ),
        )

    return jsonify({"success": False, "error": "link not provided in the request"})


@journal_api.route(
    "/lms/api/v1.0/<lname>/journal_lesson_detail/<int:sheid>/<int:instance>",
    methods=["POST"],
)
@auth.login_required
def post_journal_lesson_detail(lname, sheid, instance):
    if g.user.isTeacher == False:
        return jsonify(success=False), 403

    ip_address = (
        request.headers.get("X-Forwarded-For")
        or request.headers.get("X-Real-IP")
        or request.environ.get("REMOTE_ADDR")
    )

    submitted_file = request.files["file"]
    if submitted_file:
        File(lname).create_journal_lesson_detail_directory(sheid, instance)
        file_path = File(lname).get_journal_lesson_detail_directory(sheid, instance)
        return File.save_file_to_path(
            lname=lname,
            submitted_file=submitted_file,
            file_path=file_path,
            action_source=request.environ.get("HTTP_USER_AGENT"),
            login=g.user.id,
            user_mid=g.user.mid,
            ip_address=ip_address,
        )
    return jsonify({"success": False, "error": "upload file with key file"})


@journal_api.route(
    "/lms/api/v1.0/<lname>/journal_lesson_detail/<int:sheid>/<int:instance>/<filename>",
    methods=["DELETE"],
)
@auth.login_required
def delete_journal_lesson_detail_file(lname, sheid, instance, filename):
    try:
        os.remove(
            os.path.join(
                File(lname).get_journal_lesson_detail_directory(sheid, instance),
                filename,
            )
        )
        return jsonify(success=True)
    except:
        current_app.logger.error("getUserFile(" + filename + ")")
        return jsonify(success=False), 404
