from flask import Flask, jsonify, request, current_app, g
from flask_restful import Resource, Api
from sqlalchemy import create_engine
from sqlalchemy.sql import text
from flask import Blueprint

from LMSAPI.api.Models.Orders import Orders
from LMSAPI.api.Views.TokenAPI import auth
from LMSAPI.api.Models.User import User
from LMSAPI.api.Models.Course import Course
from flask_cors import CORS

from LMSAPI.api.utils.access_utils import user_permission_modes
from LMSAPI.api.utils.swagger_comments import SwaggerComments
from LMSAPI.api.utils.swagger_utils import swagger_doc

# https://stackoverflow.com/questions/39769666/flask-how-to-use-app-context-inside-blueprints
# Use app context from blueprint

orders_api = Blueprint("orders_api", __name__)

CORS(orders_api)


# Get all course for a faculty
@orders_api.route("/lms/api/v1.0/<lname>/ordertypes", methods=["GET"])
@auth.login_required
@swagger_doc(SwaggerComments.orders_api_getOrderTypes)
def getOrderTypes(lname):
    """
    Получить список видов приказов (работ)
    """
    result = Orders().getOrderTypes(lname)
    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@orders_api.route("/lms/api/v1.0/<lname>/orderdate", methods=["GET"])
@auth.login_required
def getOrderDate(lname):
    result = Orders().getOrderDate(lname)
    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)


@orders_api.route(
    "/lms/api/v1.0/<lname>/srw/<int:idorderdate>/<int:idfaculty>/<int:order_type_id>/<int:order_state>/<ordernum>",
    methods=["GET"],
)
@auth.login_required
@user_permission_modes("Режимы", "Научные работы", ["Нет"])
@swagger_doc(SwaggerComments.orders_api_getSRW_comment)
def getSRW(lname, idorderdate, idfaculty, order_type_id, order_state, ordernum):
    """
    Получить список Научно-исследовательских работ (НИР, SRW)
    """
    by_user = True if request.args.get("by_user") == "True" else False
    to_date = request.args.get("to_date")
    from_date = request.args.get("from_date")
    paginateFrom = request.args.get("paginateFrom") or request.args.get("offset")
    maxResults = request.args.get("maxResults") or request.args.get("limit")
    mid = g.user.mid

    # admin_flag = g.user.access_modes["Справочники"]["Постоянный состав. Администратор"] in ["Полный", "Чтение"]

    result = Orders().getSRW(
        lname=lname,
        # admin_flag=admin_flag,
        idorderdate=idorderdate,
        idfaculty=idfaculty,
        order_type_id=order_type_id,
        order_state=order_state,
        ordernum=ordernum,
        by_user=by_user,
        mid=mid,
        to_date=to_date,
        from_date=from_date,
        paginateFrom=paginateFrom,
        maxResults=maxResults,
    )
    if result is None:
        return jsonify(success=False), 404

    return jsonify(result)
